from __future__ import annotations

from typing import Any, Optional

import click

from rsconnect.exception import RSConnectException


def get_parameter_source_name_from_ctx(
    var_or_param_name: str,
    ctx: Optional[click.Context],
) -> str:
    if ctx:
        varName = var_or_param_name.replace("-", "_")
        source = ctx.get_parameter_source(varName)
        if source and source.name:
            return source.name
    return "<source unknown>"


def _get_present_options(
    options: dict[str, Optional[Any]],
    ctx: Optional[click.Context],
) -> list[str]:
    result: list[str] = []
    for k, v in options.items():
        if v:
            parts = k.split("--")
            if ctx and len(parts) == 2:
                sourceName = get_parameter_source_name_from_ctx(parts[1], ctx)
                result.append(f"{k} (from {sourceName})")
            else:
                result.append(f"{k}")
    return result


def validate_connection_options(
    ctx: Optional[click.Context],
    url: Optional[str],
    api_key: Optional[str],
    insecure: bool,
    cacert: Optional[str],
    account_name: Optional[str],
    token: Optional[str],
    secret: Optional[str],
    name: Optional[str] = None,
    snowflake_connection_name: Optional[str] = None,
):
    """
    Validates provided Connect or shinyapps.io connection options and returns which target to use given the provided
    options.

    rsconnect deploy api --name localhost ./python-bottle-py3
    should fail w/
    -s/--server or CONNECT_SERVER
    -T/--token or SHINYAPPS_TOKEN or RSCLOUD_TOKEN
    -S/--secret or SHINYAPPS_SECRET or RSCLOUD_SECRET
    -A/--account or SHINYAPPS_ACCOUNT

    FAILURE if not any of:
    -n/--name
    -s/--server or CONNECT_SERVER
    -T/--token or SHINYAPPS_TOKEN or RSCLOUD_TOKEN
    -S/--secret or SHINYAPPS_SECRET or RSCLOUD_SECRET
    -A/--account or SHINYAPPS_ACCOUNT
    --snowflake-connection-name

    FAILURE if any of:
    -k/--api-key or CONNECT_API_KEY
    -i/--insecure or CONNECT_INSECURE
    -c/--cacert or CONNECT_CA_CERTIFICATE
    AND any of:
    -T/--token or SHINYAPPS_TOKEN
    -S/--secret or SHINYAPPS_SECRET
    -A/--account or SHINYAPPS_ACCOUNT

    FAILURE if any of following are specified, without the rest:
    -T/--token or SHINYAPPS_TOKEN
    -S/--secret or SHINYAPPS_SECRET
    -A/--account or SHINYAPPS_ACCOUNT


    FAILURE if -s/--server or CONNECT_SERVER include "snowflakecomputing.app"
    and not
    --snowflake-connection-name
    """
    connect_options = {"-k/--api-key": api_key, "-i/--insecure": insecure, "-c/--cacert": cacert}
    shinyapps_options = {"-T/--token": token, "-S/--secret": secret, "-A/--account": account_name}
    spcs_options = {"--snowflake-connection-name": snowflake_connection_name}
    options_mutually_exclusive_with_name = {"-s/--server": url, **shinyapps_options}
    present_options_mutually_exclusive_with_name = _get_present_options(options_mutually_exclusive_with_name, ctx)

    if name and present_options_mutually_exclusive_with_name:
        name_source = get_parameter_source_name_from_ctx("name", ctx)
        raise RSConnectException(
            f"-n/--name (from {name_source}) cannot be specified in conjunction with options \
{', '.join(present_options_mutually_exclusive_with_name)}. See command help for further details."
        )

    if not name and not url and not shinyapps_options:
        raise RSConnectException(
            "You must specify one of -n/--name OR -s/--server OR  T/--token, -S/--secret, \
either via command options or environment variables. See command help for further details."
        )

    present_connect_options = _get_present_options(connect_options, ctx)
    present_shinyapps_options = _get_present_options(shinyapps_options, ctx)
    present_spcs_options = _get_present_options(spcs_options, ctx)

    if present_connect_options and present_shinyapps_options:
        raise RSConnectException(
            f"Connect options ({', '.join(present_connect_options)}) may not be passed \
alongside shinyapps.io options ({', '.join(present_shinyapps_options)}). \
See command help for further details."
        )

    if snowflake_connection_name and not url:
        raise RSConnectException(
            "--snowflake-connection-name requires -s/--server to be specified. \
See command help for further details."
        )

    if present_shinyapps_options and present_spcs_options:
        raise RSConnectException(
            f"Shinyapps.io options ({', '.join(present_shinyapps_options)}) may not be passed \
alongside SPCS options ({', '.join(present_spcs_options)}). \
    See command help for further details."
        )

    if present_shinyapps_options:
        if len(present_shinyapps_options) != len(shinyapps_options):
            raise RSConnectException(
                "-A/--account, -T/--token, and -S/--secret must all be provided \
for shinyapps.io. See command help for further details."
            )


class PythonVersionParamType(click.ParamType):
    name = "python-version"

    def convert(self, value: str, param: Optional[click.Parameter], ctx: Optional[click.Context]):
        try:
            parts = list(map(int, value.split(".")))
            if len(parts) == 3:
                return value
            elif len(parts) == 2:
                return value + ".0"
            else:
                raise ValueError
        except (AttributeError, ValueError):
            self.fail(f"{value!r} is not a valid python version; expected 3.x or 3.x.y", param, ctx)


PYTHON_VERSION = PythonVersionParamType()
