use crate::helpers::file::get_file;
use crate::types::master::Master;
use std::{env, fs};
use std::sync::OnceLock;
use std::sync::{Arc, Mutex};
use tracing::{info};

// OnceLock for lazy initialization
static INSTANCE: OnceLock<Store> = OnceLock::new();
pub struct Store {
    pub volume: Arc<Mutex<String>>,
    pub master: Arc<Mutex<Master>>,
}

impl Store {
    pub fn get() -> &'static Store {
        INSTANCE.get_or_init(|| {
            match Store::new() {
                Ok(store) => store,
                Err(e) => panic!("Error while getting store, {}", e),
            }
        })
    }

    pub fn delete_volume(&self) -> Result<(), String> {
        let volume = self.volume.lock().unwrap().as_str().to_string();
        match fs::remove_file(volume) {
            Ok(_) => Ok(()),
            Err(e) => Err(format!("Error removing volume, {}", e)),
        }
    }

    fn new() -> Result<Store, String> {
        let volume = Self::get_vol_path()?;
        let master: Master = match get_file(&volume) {
            Ok(v) => v,
            Err(e) => {
                return Err(format!("Error getting master, {}", e));
            }
        };
        Ok(Store {
            volume: Arc::new(Mutex::new(volume)),
            master: Arc::new(Mutex::new(master)),
        })
    }


    fn get_vol_path() -> Result<String, String> {
        match env::current_dir() {
            Ok(path) => {
                let dir:String = String::from(path.as_os_str().to_str().unwrap());
                let path = format!("{}/.rg-volume.bin", dir);
                info!("Created store at: {}", path);
                Ok(path)
            },
            Err(e) => {
                return Err(format!("Cannot get current directory, {}", e));
            }
        }
    }
}