use crate::helpers::file;
use crate::helpers::file::is_bad_edge;
use crate::types::constants::{BAD_JSON_MESSAGE, DATA, FROM, LABEL, TO, TYPE};
use crate::types::master::{Collection, NodePayload};
use crate::types::store::Store;
use serde_json::Value;
use std::collections::HashMap;
use std::fs::File;
use std::io;
use std::io::BufRead;
use std::path::Path;
use std::process::Child;
use std::string::String;
use std::sync::{Arc, Mutex};
use tracing::{debug, error, info};
use crate::helpers::lock_helper::{perform_operation, write_master};

pub fn lookup( // _collection_name: Option<String>, _detailed: bool
) {
    let master_lock = Store::get().master.lock().unwrap();
    info!("{:#?}", master_lock);
    drop(master_lock);
}

pub fn delete_volume() {
    let store = Store::get();
    match store.delete_volume() {
        Ok(()) => {
            info!("Successfully deleted volume");
        }
        Err(error) => {
            error!(error);
        }
    }
}

pub fn remove_node(collection_name: String, label: String) {
    let result: Result<String, String> = perform_operation(&collection_name, |collection| {
        collection.attempt_remove_node(label.clone())?;
        Ok(1)
    })
    .and_then(|count| {
        write_master()?;
        Ok(format!("Removed {} node(s) from collection", count))
    });

    match result {
        Ok(info) => info!(info),
        Err(error) => error!(error),
    }
}

pub fn add_edge(collection_name: String, data: String) {
    let value: Value = serde_json::from_str(&*data).expect(BAD_JSON_MESSAGE);
    if is_bad_edge(&value) {
        error!(
            "Error: Invalid edge format. Make sure .json file contains only \
                an array of objects and each of those objects contains \"{}\", \"{}\" and \"{}.{}\" keys \
                with String values",
            FROM, TO, DATA, LABEL
        );
        return;
    }

    let result: Result<String, String> = perform_operation(&collection_name, |collection| {
        collection.add_edges(&vec![value.clone()])?;
        Ok(1)
    })
    .and_then(|count| {
        write_master()?;
        Ok(format!(
            "Added {} edges(s) to collection: {}",
            count, collection_name
        ))
    });

    match result {
        Ok(info) => info!(info),
        Err(error) => error!(error),
    }
}

pub fn add_edges(collection_name: String, json_path: String) {
    let result: Result<String, String> = file::get_json_from_file(json_path).map_err(|e| {
        format!("Error getting json from file: {}", e)
    }).and_then(|json: Value| {
        if !file::is_valid_edge_file(&json) {
            Err(format!("Error: Invalid edge format. Make sure .json file contains only \
                an array of objects and each of those objects contains \"{}\", \"{}\" and \"{}.{}\" keys \
                with String values", FROM, TO, DATA, LABEL))
        } else {
            perform_operation(&collection_name, |collection| {
                Ok(collection.add_edges(json.as_array().unwrap())?)
            }).and_then(|count| {
                write_master()?;
                Ok(format!("Added {} edges(s) to collection: {}", count, collection_name))
            })
        }
    });

    match result {
        Ok(info) => info!(info),
        Err(error) => error!(error),
    }
}

pub fn remove_edge(collection_name: String, from_label: String, to_label: String) {
    let result: Result<String, String> = perform_operation(&collection_name, |collection| {
        collection.remove_edge(from_label.clone(), to_label.clone())?;
        Ok(1)
    })
    .and_then(|count| {
        write_master()?;
        Ok(format!(
            "Removed {} edges(s) from collection: {}",
            count, collection_name
        ))
    });

    match result {
        Ok(info) => info!(info),
        Err(error) => error!(error),
    }
}

pub fn draw(// collection_name: String, active_processes: &mut Arc<Mutex<HashMap<String, Child>>>
) {
    todo!()
    // let store = Store::get();
    // let master_lock = store.master.lock().unwrap();
    // match master_lock.get_collection(collection_name) {
    //     Ok(collection) => crate::helpers::gviz_init::runner(collection, active_processes),
    //     Err(error) => error!(error),
    // }
    // drop(master_lock);
}

pub fn file_to_lines(file_path: String) -> io::Result<Vec<String>> {
    let path = Path::new(&file_path);
    let file = File::open(path)?;
    let reader = io::BufReader::new(file);
    Ok(reader
        .lines()
        .filter_map(Result::ok)
        .collect::<Vec<String>>())
}
