use crate::helpers::file;
use crate::types::master::{Collection, Master};
use crate::types::store::Store;
use std::sync::MutexGuard;
use tracing::error;

pub fn get_master_lock<F>(write_master: bool, mut store_closure: F) -> Result<(), String>
where
    F: FnMut(&mut MutexGuard<Master>, &mut MutexGuard<String>) -> Result<(), String>,
{
    let store = Store::get();
    let mut master_lock: MutexGuard<Master> = store.master.lock().unwrap();
    let mut volume_lock: MutexGuard<String> = store.volume.lock().unwrap();
    store_closure(&mut master_lock, &mut volume_lock)?;
    if write_master {
        if let Err(e) = file::write_to_file(&volume_lock, &master_lock) {
            error!("Error while persisting changes: {}", e.to_string());
        }
    }
    Ok(())
}

pub fn get_collection_lock<F>(
    master: &mut MutexGuard<Master>,
    collection_name: &String,
    error_when_missing: bool,
    mut collection_closure: F,
) -> Result<(), String>
where
    F: FnMut(&mut Collection),
{
    let collection_exist = master
        .collections
        .iter()
        .any(|c| c.name.eq(collection_name));
    if !collection_exist {
        let err_msg = format!(
            "Collection does not exist, collection name: {}",
            collection_name
        );
        error!(err_msg);
        if error_when_missing {
            return Err(err_msg);
        }
    } else {
        collection_closure(master.get_collection_mut(&collection_name).unwrap());
    }
    Ok(())
}

#[deprecated]
pub fn perform_operation<F>(collection_name: &String, mut closure: F) -> Result<usize, String>
where
    F: FnMut(&mut Collection) -> Result<usize, String>,
{
    let store = Store::get();
    let mut master_lock = store.master.lock().unwrap();
    let collection = master_lock.get_collection_mut(&collection_name)?;
    let count_affected_entities = closure(collection)?;
    Ok(count_affected_entities)
}

#[deprecated]
pub fn write_master() -> Result<(), String> {
    let store = Store::get();
    let master_lock = store.master.lock().unwrap();
    let volume_lock = store.volume.lock().unwrap();

    let res = if let Err(e) = file::write_to_file(&volume_lock, &master_lock) {
        Err(e.to_string())
    } else {
        Ok(())
    };

    drop(master_lock);
    drop(volume_lock);
    res
}
