use crate::helpers::file;
use crate::helpers::lock_helper::{get_collection_lock, get_master_lock, perform_operation};
use crate::types::master::Collection;
use crate::types::store::Store;
use tracing::{debug, error};

pub fn create_collection(collection_name: String) -> bool {
    debug!("Creating collection: {}", collection_name);
    let mut res: bool = false;
    get_master_lock(true, |master, _| {
        if master
            .collections
            .iter()
            .any(|c| c.name.eq(collection_name.as_str()))
        {
            error!("Collection already exists: {}", collection_name);
        } else {
            master
                .collections
                .push(Collection::new(collection_name.clone()));
            res = true;
        }
        Ok(())
    })
    .unwrap();
    res
}

pub fn clear_collection(collection_name: String) -> bool {
    debug!("Clearing collection: {}", collection_name);
    let mut res: bool = false;
    get_master_lock(true, |master, _| {
        get_collection_lock(master, &collection_name, false, |collection| {
            collection.clear();
            debug!(
                "Collection cleared successfully, collection name: {}",
                collection_name
            );
            res = true;
        })
    })
    .unwrap();
    res
}

pub fn delete_collection(collection_name: String) -> bool {
    debug!("Deleting collection: {}", collection_name);
    let mut res: bool = false;
    get_master_lock(true, |master, _| {
        if let Some(collection_index) = master
            .collections
            .iter()
            .position(|c| c.name.eq(collection_name.as_str()))
        {
            master.collections.swap_remove(collection_index);
            debug!(
                "Collection deleted successfully, collection name: {}",
                collection_name
            );
            res = true;
        } else {
            error!("Collection does not exists: {}", collection_name);
        }
        Ok(())
    })
    .unwrap();
    res
}
