use crate::types::master::{Master, NodePayload};
use serde_json::Value;
use simplestcrypt::{deserialize_and_decrypt, encrypt_and_serialize};
use std::collections::HashSet;
use std::fs::{File, OpenOptions};
use std::io;
use std::io::{BufReader, ErrorKind, Read, Write};
use std::path::Path;
use crate::types::constants::{DATA, FROM, LABEL, TO};

const ENC_PASS: &[u8; 19] = b"see_you_at_valhalla";

fn encrypt_master(master: &Master) -> Result<Vec<u8>, String> {
    serde_json::to_string(master)
        .map_err(|error| format!("Could not convert Master to json, Error: {}", error))
        .and_then(|json_string| {
            encrypt_and_serialize(&ENC_PASS[..], json_string.as_bytes())
                .map_err(|error| format!("Encryption error: {:?}", error))
        })
}

fn decrypt_master(encrypted_data: Vec<u8>) -> Result<Master, String> {
    deserialize_and_decrypt(ENC_PASS, encrypted_data.as_slice())
        .map_err(|error| format!("Cannot decrypt file, Error: {:?}", error))
        .and_then(|decrypted| String::from_utf8(decrypted)
            .map_err(|error| format!("Could not read encrypted data, Error: {}", error)))
        .and_then(|json_string| {
            serde_json::from_str(&json_string)
                .map_err(|error| format!("Could not read encrypted data, Error: {}", error))
        })
}

fn read_from_file(file_path: &str) -> Result<Master, io::Error> {
    let mut file = File::open(file_path)?;
    let mut buffer: Vec<u8> = Vec::new();
    file.read_to_end(&mut buffer)?;
    if buffer.is_empty() {
        let mut master = Master::default();
        write_to_file(file_path, &mut master)?;
        Ok(master)
    } else {
        let decrypted_data = decrypt_master(buffer);
        if let Ok(decrypted) = decrypted_data {
            Ok(decrypted)
        } else {
            Err(io::Error::new(ErrorKind::Other, format!("Error decrypting file: {}",
                                                         decrypted_data.err().unwrap())))
        }
    }
}

pub fn write_to_file(file_path: &str, master: &Master) -> Result<(), io::Error> {
    let encrypted_data = encrypt_master(&master);
    if let Ok(encrypted_data) = encrypted_data {
        let mut file = OpenOptions::new().write(true).create(true).open(file_path)?;
        file.write_all(&encrypted_data)?;
        Ok(())
    } else {
        Err(io::Error::new(ErrorKind::Other, format!("Error writing encrypted data, Error: {}",
                                                     encrypted_data.err().unwrap())))
    }
}

pub fn get_file(file_path: &str) -> Result<Master, io::Error> {
    if !Path::new(file_path).exists() {
        //  create new
        OpenOptions::new().create_new(true).write(true).open(file_path)?;
    }
    read_from_file(file_path)
}

pub fn get_json_from_file(file_path: String) -> Result<Value, String> {
    let file_res = File::open(file_path);
    let file = match file_res {
        Ok(f) => f,
        Err(e) => return Err(format!("Failed to open file: {}", e.to_string())),
    };

    //  no need to check file extension, if serde fails, then definitely not json
    let reader = BufReader::new(file);
    match serde_json::from_reader(reader) {
        Ok(value) => Ok(value),
        Err(_err) => Err(String::from("Invalid JSON format.")),
    }
}

//  validate node file
pub fn is_valid_node_file(json: &Value) -> bool {
    match json.as_array() {
        Some(json_arr) => {
            if json_arr.is_empty() { return false; }
            let mut labels = HashSet::with_capacity(json_arr.len());
            let bad_nodes_present = json_arr.iter().any(|j| {
                if NodePayload::is_valid(j) {
                    let obj = j.as_object().unwrap();
                    let label_val = obj.get(LABEL).unwrap().as_str().unwrap();
                    !labels.insert(label_val)
                } else { true } //  bad
            });
            !bad_nodes_present
        }
        None => false,
    }
}

//  validate edge file
pub fn is_valid_edge_file(json: &Value) -> bool {
    match json.as_array() {
        Some(json_arr) => {
            if json_arr.is_empty() { return false; }
            let bad_nodes_present = json_arr.iter().any(|j| is_bad_edge(j));
            !bad_nodes_present
        }
        None => false,
    }
}

pub fn is_bad_edge(json: &Value) -> bool {
    if !json.is_object() {
        return true; // bad
    }
    let obj = json.as_object().unwrap();

    let mut from_present = false;
    let mut to_present = false;
    let mut label_present = false;

    obj.keys().for_each(|k| {
        match k.as_str() {
            FROM => from_present = obj.get(k).unwrap().is_string(),
            TO => to_present = obj.get(k).unwrap().is_string(),
            DATA => {
                let data_val = obj.get(k).unwrap();
                data_val.is_object()
                    .then(||
                        data_val.get(LABEL).is_some() &&
                            data_val.get(LABEL).unwrap().is_string())
                    .and_then(|res| {
                        label_present = res;
                        Some(res)
                    });
            }
            _ => {}
        }
    });
    !(from_present && to_present && label_present) //   not bad if satisfied
}

pub fn label_from_valid_edge(data_attr: &Value) -> String {
    data_attr.get(LABEL).unwrap().to_string()
}