#![allow(dead_code, unused_imports, deprecated)] // temporary

use crate::types::store::Store;
use crate::utils::misc;
use serde_json::Value;
use tracing::{debug, info};

mod types {
    pub mod colours;
    pub mod constants;
    pub mod master;
    pub mod query;
    pub mod store;
}

mod utils {
    pub mod misc;
    pub mod query_utils;
}

mod helpers {
    pub mod collection;
    pub mod executor;
    pub mod file;
    pub mod lock_helper;
    pub mod node;
}

pub use types::master::NodePayload;

pub fn init() {
    misc::init_tracing();
    misc::show_intro();
    let _store: &Store = Store::get();
}

pub fn print_volume() {
    let store: &Store = Store::get();
    let volume_lock = store.volume.lock().unwrap();
    info!("Volume located at: {}", &volume_lock);
}

pub fn lookup() {
    helpers::executor::lookup();
}

//  collection......................................................................................
pub fn create_collection(collection_name: String) -> bool {
    helpers::collection::create_collection(collection_name)
}

pub fn clear_collection(collection_name: String) {
    helpers::collection::clear_collection(collection_name);
}

pub fn delete_collection(collection_name: String) -> bool {
    helpers::collection::delete_collection(collection_name)
}
//  nodes...........................................................................................
//  -> add..........................................................................................
pub fn add_nodes(collection_name: String, nodes: Vec<Value>) -> Result<u8, String> {
    helpers::node::add_nodes(collection_name, nodes)
}

pub fn add_nodes_from_file(collection_name: String, json_path: String) -> Result<u8, String> {
    helpers::node::add_nodes_from_file(collection_name, json_path)
}
//  -> remove.......................................................................................
pub fn remove_nodes(collection_name: String, labels: Vec<String>) -> Result<u8, String> {
    helpers::node::remove_nodes(collection_name, labels)
}
//  -> update.......................................................................................
pub fn update_node(collection_name: String, label: String, node: Value) -> Result<bool, String> {
    helpers::node::update_node(collection_name, label, node)
}
//  -> query........................................................................................
pub fn query_node(collection_name: String, query: String) -> Result<Vec<NodePayload>, String> {
    helpers::node::query(collection_name, query)
}
//  ................................................................................................

#[cfg(test)]
mod tests {
    // use super::*;

    use crate::utils::misc;
    use crate::{NodePayload, helpers};
    use serde_json::{Value, from_str};
    use tracing::{debug, info};

    #[test]
    fn init() {
        misc::init_tracing();
        super::init();
    }

    #[test]
    fn print_volume() {
        misc::init_tracing();
        super::print_volume();
    }

    #[test]
    fn lookup() {
        misc::init_tracing();
        super::lookup();
    }

    #[test]
    pub fn create_collection() {
        misc::init_tracing();
        super::create_collection("people".to_string());
    }

    #[test]
    pub fn clear_collection() {
        misc::init_tracing();
        super::clear_collection("people".to_string());
    }

    #[test]
    pub fn delete_collection() {
        misc::init_tracing();
        super::delete_collection("people".to_string());
    }

    #[test]
    pub fn add_nodes() {
        misc::init_tracing();
        let json_str = r#"{"id": "001", "label": "James Taylor (CA)", "type": "person", "name": "James Taylor", "age": 31}"#;
        let v: Value = from_str(json_str).unwrap();
        super::add_nodes("people".to_string(), vec![v]).unwrap();
    }

    #[test]
    pub fn add_nodes_from_file() {
        misc::init_tracing();
        super::add_nodes_from_file(
            "people".to_string(),
            "/Users/janyajoshi/projects/ffi-test/tests/nodes.json".to_string(),
        )
        .unwrap();
    }

    #[test]
    pub fn remove_nodes() {
        misc::init_tracing();
        let label = "James Taylor (CA)".to_string();
        super::remove_nodes("people".to_string(), vec![label]).unwrap();
    }

    #[test]
    pub fn update_node() {
        misc::init_tracing();
        let label = "James Taylor (CA)".to_string();
        let json_str = r#"{"id": "001", "label": "James Taylor (CA)", "type": "person", "name": "James Taylor", "age": 40, "city": "Bengaluru"}"#;
        let v: Value = from_str(json_str).unwrap();
        let is_updated = super::update_node("people".to_string(), label, v).unwrap();
        info!("is_updated: {}", is_updated);
    }

    #[test]
    fn query_node() {
        misc::init_tracing();
        // todo!("Add a like '%this%' for string");
        // todo!("Refactor code for types > query, utils > query_utils, helpers > node");

        // todo!("Check for IN")
        // let query_str = "Eq(data.age , 40) && Eq(data.id, '001')";
        // let query_str = "Eq(data.city , 'Bengaluru')";
        // let query_str = "In(data.age , (40))";
        let query_str = "Lte(data.age , 40)";
        let matches = super::query_node("people".to_string(), query_str.to_string());
        debug!("{:#?}", matches);
    }
}
