use crate::helpers::file::label_from_valid_edge;
use crate::types::constants::{DATA, FROM, LABEL, TO, TYPE};
use crate::utils::misc::get_mac_add;
use chrono::{DateTime, Utc};
use mac_address::MacAddress;
use serde::{Deserialize, Serialize};
use serde_json::Value;
use std::collections::{HashMap, HashSet};
use tracing::{debug, error};
use uuid::Uuid;

//  ................................................................................................
#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct Master {
    pub version: u8,
    pub host: MacAddress,
    pub collections: Vec<Collection>,
    pub created_date: DateTime<Utc>,
    pub updated_date: DateTime<Utc>,
}

impl Master {
    pub(crate) fn new(version: u8, host: MacAddress, collections: Vec<Collection>) -> Self {
        Self {
            version,
            host,
            collections,
            created_date: Utc::now(),
            updated_date: Utc::now(),
        }
    }

    pub(crate) fn default() -> Self {
        Self::new(1, get_mac_add(), Vec::new())
    }

    pub fn get_collection(&self, name: String) -> Result<&Collection, String> {
        if let Some(collection) = self.collections.iter().find(|c| c.name == name) {
            Ok(collection)
        } else {
            Err(format!("collection not found - {}", name))
        }
    }

    pub fn get_collection_mut(&mut self, name: &String) -> Result<&mut Collection, String> {
        if let Some(c) = self.collections.iter_mut().find(|c| name.eq(&c.name)) {
            Ok(c)
        } else {
            Err(format!("collection not found - {}", name))
        }
    }
}
//  ................................................................................................
#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct Collection {
    pub id: Uuid,
    pub name: String,
    pub data: Vec<NodePayload>,
    pub label_id_map: HashMap<String, String>,
    pub adjacency_map: HashMap<String, HashSet<EdgePayload>>,
    pub created_date: DateTime<Utc>,
    pub updated_date: DateTime<Utc>,
}

impl Collection {
    pub fn new(name: String) -> Collection {
        Self {
            id: Uuid::new_v4(),
            name,
            data: vec![],
            label_id_map: HashMap::new(),
            adjacency_map: HashMap::new(),
            created_date: Utc::now(),
            updated_date: Utc::now(),
        }
    }

    pub fn clear(&mut self) {
        self.data.clear();
        self.label_id_map.clear();
        self.adjacency_map.clear();
        self.updated_date = Utc::now();
    }

    //  NODES.......................................................................................
    //  ->  Usage of raw transactional functions is discouraged, can panic if not handled properly.
    //  ->  Such functions are consciously kept private, Need to use through attempt_* functions.
    //  ->  these work on assumptions that any serde_json::Value passed is already valid.
    //  misc........................................................................................
    fn node_exists(&self, label: &String) -> bool {
        self.label_id_map.contains_key(label)
    }
    fn _get_node(&self, label: &String) -> Option<&NodePayload> {
        if self.node_exists(label) {
            Some(self.data.iter().find(|n| n.label.eq(label))?)
        } else {
            None
        }
    }
    fn get_node_mut(&mut self, label: &String) -> Option<&mut NodePayload> {
        if self.node_exists(label) {
            Some(self.data.iter_mut().find(|n| n.label.eq(label))?)
        } else {
            None
        }
    }
    //  addition....................................................................................
    fn add_node(&mut self, label: String, node: Value) {
        let potential_node = NodePayload::new(label.clone(), node);
        let node_id = potential_node.id.clone();
        self.data.push(potential_node);
        self.adjacency_map.insert(node_id.clone(), HashSet::new());
        self.label_id_map.insert(label, node_id);
    }
    fn add_nodes(&mut self, node_map: HashMap<String, Value>) {
        let mut nodes: Vec<NodePayload> = Vec::with_capacity(node_map.len());
        let mut delta_am: HashMap<String, HashSet<EdgePayload>> =
            HashMap::with_capacity(node_map.len());
        let mut delta_label_id_map: HashMap<String, String> =
            HashMap::with_capacity(node_map.len());

        node_map.iter().for_each(|nm| {
            let potential_node = NodePayload::new(nm.0.to_string(), nm.1.clone());
            let node_id = potential_node.id.clone();
            nodes.push(potential_node);
            delta_am.insert(node_id.clone(), HashSet::new());
            delta_label_id_map.insert(nm.0.to_string(), node_id);
        });

        self.data.extend(nodes);
        self.adjacency_map.extend(delta_am);
        self.label_id_map.extend(delta_label_id_map);
    }
    //  attempt addition............................................................................
    pub fn attempt_add_node(&mut self, potential_node: Value) -> Result<(), String> {
        let label = String::from(
            potential_node
                .as_object()
                .unwrap()
                .get(LABEL)
                .unwrap()
                .as_str()
                .unwrap(),
        );
        if self.node_exists(&label) {
            Err(format!("node {} already exists", label))
        } else {
            self.add_node(label, potential_node);
            Ok(())
        }
    }
    pub fn attempt_add_nodes(&mut self, potential_nodes: &Vec<Value>) -> Result<u64, String> {
        let mut potential_nm: HashMap<String, Value> =
            HashMap::with_capacity(potential_nodes.len());
        let mut count_nodes_added: u64 = 0;
        for n in potential_nodes.iter() {
            let label = String::from(n.as_object().unwrap().get(LABEL).unwrap().as_str().unwrap());
            if self.node_exists(&label) {
                error!("Node already exist, label: {}", label);
                // return Err(format!("node {} already exists", label));
            } else {
                potential_nm.insert(label, n.to_owned());
                count_nodes_added += 1;
            }
        }
        self.add_nodes(potential_nm);
        Ok(count_nodes_added)
    }
    //  removal.....................................................................................
    fn remove_node(&mut self, label: String) {
        let target = self.get_node_mut(&label).unwrap();
        let target_id = target.id.clone();
        self.data.retain(|n| n.id != target_id);
        self.adjacency_map.remove(&target_id);
        self.adjacency_map
            .iter_mut()
            .for_each(|(_, v)| v.retain(|e| e.to != target_id));
        self.label_id_map.remove(&label);
    }
    pub fn attempt_remove_node(&mut self, label: String) -> Result<(), String> {
        if self.node_exists(&label) {
            self.remove_node(label);
            Ok(())
        } else {
            Err(format!("node with label: {} does not exist", label))
        }
    }
    //  update......................................................................................
    fn update_node(&mut self, label: String, node: &Value) {
        let target = self.get_node_mut(&label).unwrap();
        target.data = node.to_owned();
    }
    pub fn attempt_update_node(
        &mut self,
        label: String,
        potential_node: &Value,
    ) -> Result<(), String> {
        if self.node_exists(&label) {
            self.update_node(label, potential_node);
            Ok(())
        } else {
            Err(format!("node with label: {} does not exist", label))
        }
    }
    //  ............................................................................................

    //  EDGES.......................................................................................
    //  ->  Usage of raw transactional functions is discouraged, can panic if not handled properly.
    //  ->  Such functions are consciously kept private, Need to use through attempt_* functions.
    //  ->  these work on assumptions that any serde_json::Value passed is already valid.

    //noinspection DuplicatedCode
    fn add_edge(
        &mut self,
        from_label: String,
        to_label: String,
        data_attr: Value,
    ) -> Result<(), String> {
        let default_err: String = format!("Cannot add edge from {} to {}", from_label, to_label);
        let from_id = self.label_id_map.get(&from_label).ok_or(format!(
            "{}, node {} does not exist",
            default_err, from_label
        ))?;
        let to_id = self.label_id_map.get(&to_label).ok_or(format!(
            "{}, node {} does not exist",
            default_err, from_label
        ))?;

        let to_set = self.adjacency_map.get_mut(from_id).unwrap();
        let edge_exists = to_set.iter().any(|e| e.to.eq(to_id));

        if edge_exists {
            Err(format!(
                "{}, edge exists - {}",
                default_err,
                label_from_valid_edge(&data_attr)
            ))
        } else {
            to_set.insert(EdgePayload::new(to_id.clone(), data_attr));
            Ok(())
        }
    }

    pub fn add_edges(&mut self, json_inputs: &Vec<Value>) -> Result<usize, String> {
        let mut count: usize = 0;
        json_inputs.iter().for_each(|json| {
            let obj = json.as_object().unwrap();
            match self.add_edge(
                String::from(obj.get(FROM).unwrap().as_str().unwrap()),
                String::from(obj.get(TO).unwrap().as_str().unwrap()),
                json.get(DATA).unwrap().to_owned(),
            ) {
                Ok(..) => count += 1,
                Err(error) => error!(error),
            }
        });
        Ok(count)
    }

    //noinspection DuplicatedCode
    pub fn remove_edge(&mut self, from_label: String, to_label: String) -> Result<(), String> {
        let default_err: String = format!("Cannot remove edge from {} to {}", from_label, to_label);
        let from_id = self.label_id_map.get(&from_label).ok_or(format!(
            "{}, node {} does not exist",
            default_err, from_label
        ))?;
        let to_id = self
            .label_id_map
            .get(&to_label)
            .ok_or(format!("{}, node {} does not exist", default_err, to_label))?;

        let to_set = self.adjacency_map.get_mut(from_id).unwrap();
        let edge_exists = to_set.iter().any(|e| e.to.eq(to_id));
        if !edge_exists {
            Err(String::from("Edge does not exist !!"))
        } else {
            to_set.retain(|edge_payload: &EdgePayload| !edge_payload.to.eq(to_id));
            Ok(())
        }
    }
}
//  ................................................................................................
#[derive(Clone, Debug, Serialize, Deserialize)]
pub struct NodePayload {
    pub id: String,
    pub label: String,
    pub data: Value,
}

impl NodePayload {
    pub fn new(label: String, data: Value) -> Self {
        Self {
            id: Uuid::new_v4().to_string(),
            label,
            data,
        }
    }

    pub fn is_valid(json: &Value) -> bool {
        if !json.is_object() {
            false
        } else {
            let obj = json.as_object().unwrap();
            obj.contains_key(LABEL)
                && obj.get(LABEL).unwrap().is_string()
                && obj.contains_key(TYPE)
                && obj.get(TYPE).unwrap().is_string()
        }
    }
}
//  ................................................................................................
#[derive(Clone, Debug, Serialize, Deserialize, Eq, PartialEq, Hash)]
pub struct EdgePayload {
    pub to: String, //  this will already be a part of from, hence only to is required.
    pub label: String,
    pub data: Value,
}
impl EdgePayload {
    pub fn new(to: String, data: Value) -> Self {
        Self {
            to,
            label: label_from_valid_edge(&data),
            data,
        }
    }
}
//  ................................................................................................
// impl Collection {
//     // Node query methods
//     pub fn get_node_by_id(&self, id: &String) -> Option<&NodePayload> {
//         self.data.iter().find(|n| n.id == *id)
//     }
//
//     pub fn get_nodes_by_property(&self, property: &str, value: &Value) -> Vec<&NodePayload> {
//         self.data.iter()
//             .filter(|n| n.data.get(property).map_or(false, |v| v == value))
//             .collect()
//     }
//
//     // Edge query methods
//     pub fn get_edge(&self, from_label: &String, to_label: &String) -> Option<&EdgePayload> {
//         let from_id = self.label_id_map.get(from_label)?;
//         let to_id = self.label_id_map.get(to_label)?;
//         self.adjacency_map.get(from_id)?.iter()
//             .find(|e| e.to == *to_id)
//     }
//
//     pub fn get_outgoing_edges(&self, from_label: &String) -> Option<&HashSet<EdgePayload>> {
//         let from_id = self.label_id_map.get(from_label)?;
//         self.adjacency_map.get(from_id)
//     }
//
//     // Bulk operations
//     pub fn remove_nodes_by_property(&mut self, property: &str, value: &Value) -> usize {
//         let labels_to_remove: Vec<String> = self.data.iter()
//             .filter(|n| n.data.get(property).map_or(false, |v| v == value))
//             .map(|n| n.label.clone())
//             .collect();
//
//         let count = labels_to_remove.len();
//         labels_to_remove.into_iter().for_each(|label| {
//             let _ = self.attempt_remove_node(label);
//         });
//         count
//     }
//
//     pub fn update_node_property(&mut self, label: &String, property: &str, value: Value) -> Result<(), String> {
//         let node = self.get_node_mut(label)
//             .ok_or_else(|| format!("Node {} not found", label))?;
//
//         if let Some(obj) = node.data.as_object_mut() {
//             obj.insert(property.to_string(), value);
//             Ok(())
//         } else {
//             Err("Node data is not an object".to_string())
//         }
//     }
// }

// impl NodePayload {
//     pub fn update_data(&mut self, data: Value) -> Result<(), String> {
//         if !data.is_object() {
//             return Err("Data must be an object".to_string());
//         }
//         self.data = data;
//         Ok(())
//     }
//
//     pub fn get_property(&self, key: &str) -> Option<&Value> {
//         self.data.as_object()?.get(key)
//     }
// }

// impl EdgePayload {
//     pub fn update_data(&mut self, data: Value) -> Result<(), String> {
//         if !data.is_object() {
//             return Err("Data must be an object".to_string());
//         }
//         self.data = data.clone();
//         self.label = label_from_valid_edge(&data);
//         Ok(())
//     }
//
//     pub fn get_property(&self, key: &str) -> Option<&Value> {
//         self.data.as_object()?.get(key)
//     }
//
//     pub fn is_valid(json: &Value) -> bool {
//         json.is_object() && json.as_object().unwrap().contains_key(LABEL)
//     }
// }
