use std::collections::HashSet;
use std::env;
use std::sync::Once;
use colored::Colorize;
use mac_address::MacAddress;
use serde_json::Value;
use tracing_subscriber::fmt::format::FmtSpan;
use tracing_subscriber::EnvFilter;
use crate::NodePayload;
use crate::types::colours::{GREEN, ORANGE, RESET};
use crate::types::constants::LABEL;

static LOGGER_INIT: Once = Once::new();

pub fn get_mac_add() -> MacAddress {
    match mac_address::get_mac_address() {
        Ok(o_ma) => o_ma.expect("cannot get mac-address"),
        Err(_err) => MacAddress::default()
    }
}

pub fn init_tracing() {
    LOGGER_INIT.call_once(|| {
        let log_level = env::var("RUST_LOG").unwrap_or_else(|_| "debug".to_string());
        tracing_subscriber::fmt()
            // .with_env_filter(EnvFilter::from_default_env())
            .with_env_filter(EnvFilter::new(log_level))
            .with_span_events(FmtSpan::FULL) // Shows full span events
            .pretty() // Enables pretty-printed, human-readable output
            .with_writer(std::io::stdout)
            .with_ansi(true) // Enables colored output
            .init();
    });
}

pub fn serde_value_to_node_payload(val: Vec<Value>) -> bool {
    val.iter().any(|v| !NodePayload::is_valid(v))
}

pub fn show_intro() {
    let intro = format!(r#"
{}                                   __           __                             __        {}
{}    _~^~^~_    {}   _______  _______/ /____  ____/ /     ____ __________ _____  / /_  _____{}
{}\) /  o o  \ (/{}  / ___/ / / / ___/ __/ _ \/ __  /_____/ __ `/ ___/ __ `/ __ \/ __ \/ ___/{}
{}  '_   -   _'  {} / /  / /_/ (__  ) /_/  __/ /_/ /_____/ /_/ / /  / /_/ / /_/ / / / (__  ) {}
{}  / '-----' \  {}/_/   \__,_/____/\__/\___/\__,_/      \__, /_/   \__,_/ .___/_/ /_/____/  {}
{}                                                    /____/          /_/                  {}
"#, GREEN, RESET,
                        ORANGE, GREEN, RESET,
                        ORANGE, GREEN, RESET,
                        ORANGE, GREEN, RESET,
                        ORANGE, GREEN, RESET,
                        GREEN, RESET,);

    println!("{}", intro.bold());
    println!("{}", "Welcome to Rusted Graphs".to_string().bold().yellow());
    let tracked_message = format!("Author: Janya Joshi, version {}",
                                  env!("CARGO_PKG_VERSION")).bold().yellow();
    println!("{}", tracked_message);
    println!("{}", std::iter::repeat("-").take(tracked_message.len()).collect::<String>()
        .bold().yellow());
}