use serde_json::Value::Bool;
use crate::types::query::{Expr, Operator, Parser, Token};

pub fn parse_query(input: &str) -> Result<Expr, String> {
    let trimmed = input.trim();
    if trimmed.is_empty() {
        // return a trivial condition that always matches
        return Ok(Expr::Condition {
            op: Operator::Eq,
            path: "".into(),
            values: Bool(true),
        });
    }

    //
    let toks = tokenize(trimmed)?;
    let mut parser = Parser::new(toks);
    let expr = parser.parse_expr()?;
    // ensure EOF
    if parser.peek() != &Token::EOF {
        return Err(format!("Unexpected trailing tokens: {:?}", parser.peek()));
    }
    Ok(expr)
}

fn is_ident_char(c: char) -> bool {
    c.is_ascii_alphabetic() || c == '_'
}

fn is_path_char(c: char) -> bool {
    c.is_ascii_alphanumeric() || c == '_' || c == '.' // allow dots in paths
}

fn tokenize(input: &str) -> Result<Vec<Token>, String> {
    let mut tokens = Vec::new();
    let mut it = input.chars().peekable();
    while let Some(&c) = it.peek() {
        match c {
            ch if ch.is_whitespace() => {
                it.next();
            }
            '&' => {
                it.next();
                if it.peek() == Some(&'&') {
                    it.next();
                    tokens.push(Token::And);
                } else {
                    return Err("Single '&' unsupported; use '&&'".into());
                }
            }
            '|' => {
                it.next();
                if it.peek() == Some(&'|') {
                    it.next();
                    tokens.push(Token::Or);
                } else {
                    return Err("Single '|' unsupported; use '||'".into());
                }
            }
            '(' => {
                it.next();
                tokens.push(Token::ParenOpen);
            }
            ')' => {
                it.next();
                tokens.push(Token::ParenClose);
            }
            ',' => {
                it.next();
                tokens.push(Token::Comma);
            }
            '\'' | '"' => {
                // parse quoted string
                let quote = it.next().unwrap();
                let mut s = String::new();
                while let Some(&nc) = it.peek() {
                    it.next();
                    if nc == quote {
                        break;
                    } else if nc == '\\' {
                        // handle simple escapes
                        if let Some(&esc) = it.peek() {
                            it.next();
                            match esc {
                                'n' => s.push('\n'),
                                't' => s.push('\t'),
                                'r' => s.push('\r'),
                                '\\' => s.push('\\'),
                                '\'' => s.push('\''),
                                '"' => s.push('"'),
                                other => s.push(other),
                            }
                        }
                    } else {
                        s.push(nc);
                    }
                }
                tokens.push(Token::String(s));
            }
            ch if ch.is_ascii_digit()
                || (ch == '-'
                    && it
                        .clone()
                        .nth(1)
                        .map(|c| c.is_ascii_digit())
                        .unwrap_or(false)) =>
            {
                // number (integer or float)
                let mut num = String::new();
                num.push(it.next().unwrap());
                while let Some(&nc) = it.peek() {
                    if nc.is_ascii_digit()
                        || nc == '.'
                        || nc == 'e'
                        || nc == 'E'
                        || nc == '+'
                        || nc == '-'
                    {
                        num.push(it.next().unwrap());
                    } else {
                        break;
                    }
                }
                tokens.push(Token::Number(num));
            }
            ch if is_ident_char(ch) => {
                let mut ident = String::new();
                ident.push(it.next().unwrap());
                while let Some(&nc) = it.peek() {
                    if is_ident_char(nc) || nc.is_ascii_digit() {
                        ident.push(it.next().unwrap());
                    } else {
                        break;
                    }
                }
                // ident might be a boolean literal "true"/"false" or a function name or part of path
                let id_lower = ident.to_lowercase();
                if id_lower == "true" {
                    tokens.push(Token::Bool(true));
                } else if id_lower == "false" {
                    tokens.push(Token::Bool(false));
                } else {
                    // Could be a function name like Eq or a path starting with data
                    // Lookahead: if next is a '.' or next char is '.', or identifier contains a dot? We didn't allow dot in ident parse.
                    // But path contains dots; if next is '.' then we parse rest as path
                    if it.peek() == Some(&'.') {
                        // parse the rest of the path
                        let mut path = ident;
                        while let Some(&nc) = it.peek() {
                            if is_path_char(nc) {
                                path.push(it.next().unwrap());
                            } else {
                                break;
                            }
                        }
                        tokens.push(Token::Path(path));
                    } else {
                        // Could be standalone path without dot (e.g., data) or function name
                        // Check further: if ident == "data" and next is '.' then parse '.' and subsequent
                        if ident == "data" && it.peek() == Some(&'.') {
                            let mut path = ident;
                            while let Some(&nc) = it.peek() {
                                if is_path_char(nc) {
                                    path.push(it.next().unwrap());
                                } else {
                                    break;
                                }
                            }
                            tokens.push(Token::Path(path));
                        } else {
                            // function name or short path
                            // but short path without dot is rare; treat as Path too
                            tokens.push(Token::Path(ident));
                        }
                    }
                }
            }
            ch if is_path_char(ch) => {
                // Path starting with underscore or digits maybe - capture contiguous path chars
                let mut path = String::new();
                path.push(it.next().unwrap());
                while let Some(&nc) = it.peek() {
                    if is_path_char(nc) {
                        path.push(it.next().unwrap());
                    } else {
                        break;
                    }
                }
                tokens.push(Token::Path(path));
            }
            other => {
                return Err(format!("Unexpected character in input: '{}'", other));
            }
        }
    }
    tokens.push(Token::EOF);
    Ok(tokens)
}
