use crate::NodePayload;
use crate::helpers::file;
use crate::helpers::lock_helper::{
    get_collection_lock, get_master_lock, perform_operation, write_master,
};
use crate::types::constants::{LABEL, TYPE};
use crate::types::query::{Expr, Operator};
use crate::utils::query_utils::parse_query;
use serde_json::Value;
use tracing::{debug, error};

pub fn add_nodes(collection_name: String, nodes: Vec<Value>) -> Result<u8, String> {
    debug!(
        "Adding {} node(s) to collection: {}",
        nodes.len(),
        collection_name
    );

    if let Some(invalid_node) = nodes.iter().find(|v| !NodePayload::is_valid(v)) {
        Err(format!("Invalid node: {:?}", invalid_node))
    } else {
        let mut count_added_nodes: u8 = 0;
        get_master_lock(true, |master, _| {
            get_collection_lock(master, &collection_name, true, |collection| {
                count_added_nodes = collection.attempt_add_nodes(&nodes).unwrap() as u8;
                debug!(
                    "Added {} node(s) to collection: {}",
                    count_added_nodes, collection_name
                );
            })
        })?;
        Ok(count_added_nodes)
    }
}

pub fn add_nodes_from_file(collection_name: String, json_path: String) -> Result<u8, String> {
    debug!(
        "Adding node(s) from file to collection, collection: {}, file: {}",
        collection_name, json_path
    );

    file::get_json_from_file(json_path).and_then(|json: Value| {
        if !file::is_valid_node_file(&json) {
            Err(format!(
                "Error: Invalid node format. Make sure .json file contains only \
                an array of objects and each of those objects contains a \"{}\" and \"{}\" key \
                with String value",
                LABEL, TYPE
            ))
        } else {
            let mut count_added_nodes: u8 = 0;
            get_master_lock(true, |master, _| {
                get_collection_lock(master, &collection_name, true, |collection| {
                    let potential_nodes = json.as_array().unwrap();
                    count_added_nodes =
                        collection.attempt_add_nodes(potential_nodes).unwrap() as u8;
                    debug!(
                        "Added {} node(s) to collection: {}",
                        count_added_nodes, collection_name
                    );
                })
            })?;
            Ok(count_added_nodes)
        }
    })
}

pub fn remove_nodes(collection_name: String, labels: Vec<String>) -> Result<u8, String> {
    debug!(
        "Removing {} node(s) from collection: {}",
        labels.len(),
        collection_name
    );

    let mut count_added_nodes: u8 = 0;
    get_master_lock(true, |master, _| {
        get_collection_lock(master, &collection_name, true, |collection| {
            for label in &labels {
                if let Err(err) = collection.attempt_remove_node(label.clone()) {
                    error!("Failed to remove node: {:?}", err);
                } else {
                    count_added_nodes += 1;
                }
            }
            debug!(
                "Removed {} node(s) from collection: {}",
                count_added_nodes, collection_name
            );
        })
    })?;
    Ok(count_added_nodes)
}

pub fn update_node(
    collection_name: String,
    label: String,
    potential_node: Value,
) -> Result<bool, String> {
    debug!("Updating node with label: {}", label);

    if !NodePayload::is_valid(&potential_node) {
        return Err(format!("Invalid node: {}", potential_node));
    }

    let mut is_updated = false;
    get_master_lock(true, |master, _| {
        get_collection_lock(master, &collection_name, true, |collection| {
            if let Err(err) = collection.attempt_update_node(label.clone(), &potential_node) {
                error!(
                    "Failed to update node with label: {:?}, err: {:?}",
                    label, err
                );
            } else {
                is_updated = true;
                debug!("Updated node with label: {}", label);
            }
        })
    })?;
    Ok(is_updated)
}

pub fn query(collection_name: String, query: String) -> Result<Vec<NodePayload>, String> {
    let parsed_expression = parse_query(query.as_str())?;
    debug!("Parsed expression: {:#?}", parsed_expression);

    let mut matching_nodes: Vec<NodePayload> = vec![];

    get_master_lock(true, |master, _| {
        get_collection_lock(master, &collection_name, true, |collection| {
            collection
                .data
                .iter()
                .filter(|node| eval_expr(&parsed_expression, &node.data))
                .for_each(|x| matching_nodes.push(x.to_owned()));
        })
    })?;

    debug!("{} nodes matched", matching_nodes.len());
    Ok(matching_nodes)
}

fn eval_expr(expr: &Expr, node_data: &Value) -> bool {
    match expr {
        Expr::Condition { op, path, values } => {
            // match all in case of empty query.
            if Operator::Eq.eq(op) && path.is_empty() {
                true
            } else {
                let resolved = resolve_path(node_data, path);
                match op {
                    Operator::Eq => resolved.map_or(false, |v| value_eq(v, values)),
                    Operator::Ne => resolved.map_or(false, |v| !value_eq(v, values)),
                    Operator::Lt => resolved.map_or(false, |v| value_lt(v, values)),
                    Operator::Lte => resolved.map_or(false, |v| value_le(v, values)),
                    Operator::Gt => resolved.map_or(false, |v| value_gt(v, values)),
                    Operator::Gte => resolved.map_or(false, |v| value_ge(v, values)),
                    Operator::In => {
                        if let Value::Array(arr) = values {
                            resolved.map_or(false, |v| value_in(v, arr))
                        } else {
                            false
                        }
                    }
                }
            }
        }
        Expr::And(l, r) => eval_expr(l, node_data) && eval_expr(r, node_data),
        Expr::Or(l, r) => eval_expr(l, node_data) || eval_expr(r, node_data),
    }
}

// comparison
fn value_eq(a: &Value, b: &Value) -> bool {
    match (a, b) {
        (Value::Number(na), Value::Number(nb)) => na.as_f64() == nb.as_f64(),
        (Value::String(sa), Value::String(sb)) => sa == sb,
        (Value::Bool(ab), Value::Bool(bb)) => ab == bb,
        (Value::Array(aa), Value::Array(bb)) => aa == bb,
        _ => false,
    }
}
fn value_lt(a: &Value, b: &Value) -> bool {
    match (a, b) {
        (Value::Number(na), Value::Number(nb)) => na.as_f64() < nb.as_f64(),
        (Value::String(sa), Value::String(sb)) => sa < sb,
        _ => false,
    }
}
fn value_le(a: &Value, b: &Value) -> bool {
    value_lt(a, b) || value_eq(a, b)
}
fn value_gt(a: &Value, b: &Value) -> bool {
    match (a, b) {
        (Value::Number(na), Value::Number(nb)) => na.as_f64() > nb.as_f64(),
        (Value::String(sa), Value::String(sb)) => sa > sb,
        _ => false,
    }
}
fn value_ge(a: &Value, b: &Value) -> bool {
    value_gt(a, b) || value_eq(a, b)
}
fn value_in(node_val: &Value, set: &[Value]) -> bool {
    if let Value::Array(arr) = node_val {
        for el in arr {
            for candidate in set {
                if value_eq(el, candidate) {
                    return true;
                }
            }
        }
        return false;
    }
    for candidate in set {
        if value_eq(node_val, candidate) {
            return true;
        }
    }
    false
}

//  path resolver
fn resolve_path<'a>(data: &'a Value, path: &str) -> Option<&'a Value> {
    let path = if path.starts_with("data.") {
        &path[5..]
    } else {
        path
    };
    if path.is_empty() {
        return Some(data);
    }
    let mut current = data;
    for seg in path.split('.') {
        match current {
            Value::Object(map) => {
                current = map.get(seg)?;
            }
            Value::Array(arr) => {
                let idx = seg.parse::<usize>().ok()?;
                current = arr.get(idx)?;
            }
            _ => return None,
        }
    }
    Some(current)
}
