use crate::NodePayload;
use crate::helpers::file;
use crate::helpers::lock_helper::{
    get_collection_lock, get_master_lock, perform_operation, write_master,
};
use crate::types::constants::{DATA, FROM, LABEL, TO, TYPE};
use crate::types::master::EdgePayload;
use crate::types::query::{Expr, Operator};
use crate::utils::query_utils::parse_query;
use serde_json::Value;
use tracing::{debug, error, info};

pub fn add_edges(collection_name: String, edges: Vec<Value>) -> Result<u8, String> {
    debug!(
        "Adding {} edge(s) to collection: {}",
        edges.len(),
        collection_name
    );

    if let Some(invalid_edge) = edges.iter().find(|v| !EdgePayload::is_valid(v)) {
        Err(format!("Invalid edge: {:?}", invalid_edge))
    } else {
        let mut count_added_edges: u8 = 0;
        get_master_lock(true, |master, _| {
            get_collection_lock(master, &collection_name, true, |collection| {
                count_added_edges = collection.add_edges(&edges).unwrap() as u8;
                debug!(
                    "Added {} edges(s) to collection: {}",
                    count_added_edges, collection_name
                );
            })
        })?;
        Ok(count_added_edges)
    }
}

pub fn add_edges_from_file(collection_name: String, json_path: String) -> Result<u8, String> {
    debug!(
        "Adding edge(s) from file to collection, collection: {}, file: {}",
        collection_name, json_path
    );

    file::get_json_from_file(json_path).and_then(|json: Value| {
        if !file::is_valid_edge_file(&json) {
            Err(format!("Error: Invalid edge format. Make sure .json file contains only \
                an array of objects and each of those objects contains \"{}\", \"{}\" and \"{}.{}\" keys \
                with String values", FROM, TO, DATA, LABEL))
        } else {
            let mut count_added_edges: u8 = 0;
            get_master_lock(true, |master, _| {
                get_collection_lock(master, &collection_name, true, |collection| {
                    let potential_edges = json.as_array().unwrap();
                    count_added_edges =
                        collection.add_edges(potential_edges).unwrap() as u8;
                    debug!(
                        "Added {} edges(s) to collection: {}",
                        count_added_edges, collection_name
                    );
                })
            })?;
            Ok(count_added_edges)
        }
    })
}

pub fn remove_edge(
    collection_name: String,
    from_label: String,
    to_label: String,
) -> Result<u8, String> {
    debug!(
        "Removing edge from collection: {}, from node: {}, to node: {}",
        collection_name, from_label, to_label
    );

    let mut res: Result<u8, String> = Ok(1);
    get_master_lock(true, |master, _| {
        get_collection_lock(master, &collection_name, true, |collection| {
            if let Err(err) = collection.remove_edge(from_label.clone(), to_label.clone()) {
                res = Err(err);
            } else {
                debug!("Removed edge(s) from collection: {}", collection_name);
            }
        })
    })?;

    res
}
