use serde_json::{Number, Value};
use std::fmt::{Display, Formatter, Result as FmtResult};

#[derive(Clone, Debug, PartialEq)]
pub enum Operator {
    Eq,
    Ne,
    Lt,
    Lte,
    Gt,
    Gte,
    In,
}

#[derive(Clone, Debug)]
pub enum Expr {
    Condition {
        op: Operator,
        path: String,
        values: Value, // one or many (In uses many)
    },
    And(Box<Expr>, Box<Expr>),
    Or(Box<Expr>, Box<Expr>),
}

impl Display for Operator {
    fn fmt(&self, f: &mut Formatter<'_>) -> FmtResult {
        use Operator::*;
        write!(
            f,
            "{}",
            match self {
                Eq => "Eq",
                Ne => "Ne",
                Lt => "Lt",
                Lte => "Lte",
                Gt => "Gt",
                Gte => "Gte",
                In => "In",
            }
        )
    }
}

#[derive(Clone, Debug, PartialEq)]
pub enum Token {
    Ident(String),  // Eq, Lte, In
    Path(String),   // data.prop.sub
    Number(String), // numeric literal as string -> to parse later
    String(String), // quoted string without quotes
    Bool(bool),     // true / false
    ParenOpen,
    ParenClose,
    Comma,
    And, // &&
    Or,  // ||
    EOF,
}

pub struct Parser {
    tokens: Vec<Token>,
    pos: usize,
}

impl Parser {
    pub(crate) fn new(tokens: Vec<Token>) -> Self {
        Self { tokens, pos: 0 }
    }

    pub(crate) fn peek(&self) -> &Token {
        &self.tokens[self.pos]
    }

    fn next(&mut self) -> &Token {
        if self.pos < self.tokens.len() {
            self.pos += 1;
        }
        &self.tokens[self.pos - 1]
    }

    fn expect(&mut self, expected: &Token) -> Result<(), String> {
        if self.peek() == expected {
            self.next();
            Ok(())
        } else {
            Err(format!("Expected {:?}, found {:?}", expected, self.peek()))
        }
    }

    pub(crate) fn parse_expr(&mut self) -> Result<Expr, String> {
        self.parse_or()
    }

    // or_expr := and_expr ( '||' and_expr )*
    fn parse_or(&mut self) -> Result<Expr, String> {
        let mut node = self.parse_and()?;
        while self.peek() == &Token::Or {
            self.next();
            let right = self.parse_and()?;
            node = Expr::Or(Box::new(node), Box::new(right));
        }
        Ok(node)
    }

    // and_expr := primary ( '&&' primary )*
    fn parse_and(&mut self) -> Result<Expr, String> {
        let mut node = self.parse_primary()?;
        while self.peek() == &Token::And {
            self.next();
            let right = self.parse_primary()?;
            node = Expr::And(Box::new(node), Box::new(right));
        }
        Ok(node)
    }

    // primary := func_call | '(' expr ')'
    fn parse_primary(&mut self) -> Result<Expr, String> {
        match self.peek() {
            Token::Ident(_name) | Token::Path(_name) => {
                // If it's an Ident for function name, parse function
                // Some tokens produced paths for "Eq" as Path if capitalized; handle both.
                let func_name = match self.next() {
                    Token::Ident(s) => s.clone(),
                    Token::Path(s) => s.clone(),
                    other => {
                        return Err(format!(
                            "Unexpected token when parsing primary: {:?}",
                            other
                        ));
                    }
                };
                self.parse_function_call(&func_name)
            }
            Token::ParenOpen => {
                self.next(); // consume '('
                let expr = self.parse_expr()?;
                if self.peek() != &Token::ParenClose {
                    return Err("Expected ')'".into());
                }
                self.next();
                Ok(expr)
            }
            other => Err(format!("Unexpected token in primary: {:?}", other)),
        }
    }

    fn parse_function_call(&mut self, func_name: &str) -> Result<Expr, String> {
        if self.peek() != &Token::ParenOpen {
            return Err(format!("Expected '(' after '{}'", func_name));
        }
        self.next();

        let path = match self.next() {
            Token::Path(p) | Token::Ident(p) => p.clone(),
            other => return Err(format!("Expected path, found {:?}", other)),
        };

        if self.peek() != &Token::Comma {
            return Err(format!("Expected ',' after path"));
        }
        self.next();

        #[allow(unused_assignments)]
        let mut val = Value::Null;

        match self.peek() {
            Token::ParenOpen => {
                val = Value::Array(Vec::new());
                self.next();
                loop {
                    match self.peek() {
                        Token::String(s) => {
                            val.as_array_mut().unwrap().push(Value::String(s.clone()));
                            self.next();
                        }
                        Token::Number(n) => {
                            let number = parse_number_literal(n)?;
                            val.as_array_mut().unwrap().push(Value::Number(number));
                            self.next();
                        }
                        Token::Bool(b) => {
                            val.as_array_mut().unwrap().push(Value::Bool(*b));
                            self.next();
                        }
                        other => {
                            return Err(format!("Unexpected token in list literal: {:?}", other));
                        }
                    }
                    match self.peek() {
                        Token::Comma => {
                            self.next();
                            continue;
                        }
                        Token::ParenClose => {
                            self.next();
                            break;
                        }
                        other => {
                            return Err(format!(
                                "Expected ',' or ')' in list literal, found {:?}",
                                other
                            ));
                        }
                    }
                }
            }
            Token::String(s) => {
                val = Value::String(s.clone());
                self.next();
            }
            Token::Number(n) => {
                val = Value::Number(parse_number_literal(n)?);
                self.next();
            }
            Token::Bool(b) => {
                val = Value::Bool(*b);
                self.next();
            }
            other => return Err(format!("Unexpected token for second argument: {:?}", other)),
        }

        if self.peek() != &Token::ParenClose {
            return Err(format!("Expected ')' to close '{}'", func_name));
        }
        self.next();

        let op = match func_name.to_lowercase().as_str() {
            "eq" => Operator::Eq,
            "ne" => Operator::Ne,
            "lt" => Operator::Lt,
            "lte" | "le" => Operator::Lte,
            "gt" => Operator::Gt,
            "gte" | "ge" => Operator::Gte,
            "in" => Operator::In,
            other => return Err(format!("Unknown operator '{}'", other)),
        };

        Ok(Expr::Condition {
            op,
            path,
            values: val,
        })
    }
}

fn parse_number_literal(s: &str) -> Result<Number, String> {
    // parse as f64 then convert to serde_json::Number
    let n: f64 = s
        .parse()
        .map_err(|e| format!("Failed to parse number '{}': {}", s, e))?;
    Number::from_f64(n).ok_or_else(|| format!("Invalid JSON number from '{}'", s))
}
