#![allow(deprecated, unused_imports)] // will fix later

use pyo3::exceptions::PyValueError;
use pyo3::prelude::*;
use pyo3::types::{PyDict, PyList};
use pyo3_stub_gen::{PyStubType, TypeInfo, define_stub_info_gatherer, derive::gen_stub_pyfunction};
use rusted_graphs;
use rusted_graphs::NodePayload;
use serde_json::Value;
use serde_pyobject::from_pyobject;
use std::collections::HashSet;

mod utils {
    pub mod adaptors;
}

#[pyclass]
pub struct PyNodePayload(pub NodePayload);

impl PyStubType for PyNodePayload {
    fn type_output() -> TypeInfo {
        TypeInfo {
            name: "PyNodePayload".to_string(),
            import: HashSet::new(),
        }
    }

    fn type_input() -> TypeInfo {
        Self::type_output()
    }
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn init() {
    rusted_graphs::init();
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn print_volume() {
    rusted_graphs::print_volume();
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn lookup() {
    rusted_graphs::lookup();
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn create_collection(collection_name: String) -> PyResult<bool> {
    Ok(rusted_graphs::create_collection(collection_name))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn clear_collection(collection_name: String) {
    rusted_graphs::clear_collection(collection_name)
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn delete_collection(collection_name: String) -> PyResult<bool> {
    Ok(rusted_graphs::delete_collection(collection_name))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn add_nodes(collection_name: String, nodes: &Bound<'_, PyList>) -> PyResult<u8> {
    let mut vals: Vec<Value> = Vec::with_capacity(nodes.len());
    for node in nodes.iter() {
        let n: Value = from_pyobject(node)?;
        vals.push(n);
    }

    rusted_graphs::add_nodes(collection_name, vals).map_err(|e| PyValueError::new_err(e))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn add_nodes_from_file(collection_name: String, json_path: String) -> PyResult<u8> {
    rusted_graphs::add_nodes_from_file(collection_name, json_path)
        .map_err(|e| PyValueError::new_err(e))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn remove_nodes(collection_name: String, labels: Vec<String>) -> PyResult<u8> {
    rusted_graphs::remove_nodes(collection_name, labels).map_err(|e| PyValueError::new_err(e))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn update_node(collection_name: String, label: String, node: Bound<PyAny>) -> PyResult<bool> {
    let new_node: Value = from_pyobject(node)?;
    rusted_graphs::update_node(collection_name, label, new_node)
        .map_err(|e| PyValueError::new_err(e))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn query_node(py: Python<'_>, collection_name: String, query: String) -> PyResult<PyObject> {
    let nodes =
        rusted_graphs::query_node(collection_name, query).map_err(|e| PyValueError::new_err(e))?;

    let json_values: Vec<Value> = nodes
        .into_iter()
        .map(|n| serde_json::to_value(n).unwrap())
        .collect();

    let py_objects = json_values
        .into_iter()
        .map(|v| crate::utils::adaptors::serde_value_to_pyobject(py, v))
        .collect::<PyResult<Vec<_>>>()?;

    let py_list = PyList::new(py, py_objects)?;
    Ok(PyList::new(py, py_list)?.into())
    // Ok(py_list.into_py(py))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn add_edges(collection_name: String, edges: &Bound<'_, PyList>) -> PyResult<u8> {
    let mut vals: Vec<Value> = Vec::with_capacity(edges.len());
    for edge in edges.iter() {
        let n: Value = from_pyobject(edge)?;
        vals.push(n);
    }

    rusted_graphs::add_edges(collection_name, vals).map_err(|e| PyValueError::new_err(e))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn add_edges_from_file(collection_name: String, json_path: String) -> PyResult<u8> {
    rusted_graphs::add_edges_from_file(collection_name, json_path)
        .map_err(|e| PyValueError::new_err(e))
}

#[gen_stub_pyfunction]
#[pyfunction]
pub fn remove_edge(collection_name: String, from_label: String, to_label: String) -> PyResult<u8> {
    rusted_graphs::remove_edge(collection_name, from_label, to_label)
        .map_err(|e| PyValueError::new_err(e))
}

define_stub_info_gatherer!(stub_info);

#[pymodule]
fn rusted_graphs_python(_py: Python, m: &Bound<PyModule>) -> PyResult<()> {
    m.add_function(wrap_pyfunction!(init, m)?)?;
    m.add_function(wrap_pyfunction!(print_volume, m)?)?;
    m.add_function(wrap_pyfunction!(lookup, m)?)?;
    m.add_function(wrap_pyfunction!(create_collection, m)?)?;
    m.add_function(wrap_pyfunction!(clear_collection, m)?)?;
    m.add_function(wrap_pyfunction!(delete_collection, m)?)?;
    m.add_function(wrap_pyfunction!(add_nodes, m)?)?;
    m.add_function(wrap_pyfunction!(add_nodes_from_file, m)?)?;
    m.add_function(wrap_pyfunction!(remove_nodes, m)?)?;
    m.add_function(wrap_pyfunction!(update_node, m)?)?;
    m.add_function(wrap_pyfunction!(query_node, m)?)?;
    m.add_function(wrap_pyfunction!(add_edges, m)?)?;
    m.add_function(wrap_pyfunction!(add_edges_from_file, m)?)?;
    m.add_function(wrap_pyfunction!(remove_edge, m)?)?;
    Ok(())
}
