use pyo3::types::{PyBool, PyDict, PyFloat, PyInt, PyList, PyString};
use pyo3::{IntoPyObject, IntoPyObjectExt, PyObject, PyResult, Python};
use pyo3::prelude::PyDictMethods;
use serde_json::Value;

// convert serde_json::Value to PyObject
pub fn serde_value_to_pyobject(py: Python<'_>, v: Value) -> PyResult<PyObject> {
    use serde_json::Value::*;

    Ok(match v {
        Null => py.None(),
        Bool(b) => b.into_py_any(py)?,
        Number(n) => {
            if let Some(i) = n.as_i64() {
                PyInt::new(py, i).into()
            } else if let Some(f) = n.as_f64() {
                PyFloat::new(py, f).into()
            } else {
                // Handle non-representable numbers as None or raise an error
                py.None()
            }
        }
        String(s) => PyString::new(py, s.as_str()).into(),
        Array(a) => {
            let inner = a
                .into_iter()
                .map(|x| serde_value_to_pyobject(py, x))
                .collect::<PyResult<Vec<_>>>()?;
            inner.into_py_any(py)?
        }
        Object(o) => {
            let dict = PyDict::new(py);
            for (k, v) in o {
                dict.set_item(k, serde_value_to_pyobject(py, v)?)?;
            }
            dict.into()
        }
    })
}