#!/bin/sh

# Executes the pytest framework in both JPype and Jep modes.
#
# Usage examples:
#   bin/test.sh
#   bin/test.sh tests/test_basics.py
#   bin/test.sh tests/test_convert.py::TestConvert::test2DStringArray

set -e

dir=$(dirname "$0")
cd "$dir/.."

echo
echo "-------------------------------------------"
echo "| Testing JPype mode (Java inside Python) |"
echo "-------------------------------------------"

if [ $# -gt 0 ]
then
  python -m pytest -p no:faulthandler $@
else
  python -m pytest -p no:faulthandler tests/
fi
jpypeCode=$?

echo
echo "-------------------------------------------"
echo "| Running integration tests (JPype only)  |"
echo "-------------------------------------------"
itCode=0
for t in tests/it/*.py
do
  python "$t"
  code=$?
  printf -- "--> %s " "$t"
  if [ "$code" -eq 0 ]
  then
    echo "[OK]"
  else
    echo "[FAILED]"
    itCode=$code
  fi
done

echo
echo "-------------------------------------------"
echo "|  Testing Jep mode (Python inside Java)  |"
echo "-------------------------------------------"

# Discern the Jep installation.
site_packages=$(python -c 'import sys; print(next(p for p in sys.path if p.endswith("site-packages")))')
test -d "$site_packages/jep" || {
  echo "[ERROR] Failed to detect Jep installation in current environment!" 1>&2
  exit 1
}

# We execute the pytest framework through Jep via jgo, so that
# the surrounding JVM includes scijava-table on the classpath.
#
# Arguments to the shell script are translated into an argument
# list to the pytest.main function. A weak attempt at handling
# special characters, e.g. single quotation marks and backslashes,
# is made, but there are surely other non-working cases.

if [ $# -gt 0 ]
then
  a=$(echo "$@" | sed 's/\\/\\\\/g')     # escape backslashes
  a=$(echo "$a" | sed 's/'\''/\\'\''/g') # escape single quotes
  a=$(echo "$a" | sed 's/ /'\'','\''/g') # replace space with ','
  argString="['-v', '$a']"
else
  argString=""
fi
if [ "$(uname -s)" = "Darwin" ]
then
  echo "Skipping jep tests on macOS due to flakiness"
  jepCode=0
else
  echo "# AUTOGENERATED test file for jep; safe to delete.
import logging, sys, pytest, scyjava
scyjava._logger.addHandler(logging.StreamHandler(sys.stderr))
scyjava._logger.setLevel(logging.INFO)
scyjava.config.set_verbose(2)
result = pytest.main($argString)
if result:
  sys.exit(result)
" > jep_test.py
  jgo -vv \
    -r scijava.public=https://maven.scijava.org/content/groups/public \
    -Djava.library.path="$site_packages/jep" \
    black.ninia:jep:jep.Run+org.scijava:scijava-table \
    jep_test.py
  jepCode=$?
  rm -f jep_test.py
fi

test "$jpypeCode" -ne 0 && exit "$jpypeCode"
test "$itCode" -ne 0 && exit "$itCode"
test "$jepCode" -ne 0 && exit "$jepCode"
exit 0
