# -*- coding: utf-8 -*-
"""sidsassignment

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1etQvUJAPurTYsIwa6MRs3rRwkN48K4Z1
"""

import string

class TextProcessor:
    """
    A utility class for performing basic text preprocessing steps.

    The steps included are:
    1. Lowercasing
    2. Punctuation removal
    3. Tokenization (splitting into words)
    4. Stop word removal (using a common, predefined set)
    """

    # A basic, static set of common English stop words
    STOP_WORDS = set([
        "a", "an", "and", "are", "as", "at", "be", "by", "for", "from", "has",
        "he", "in", "is", "it", "its", "of", "on", "that", "the", "to", "was",
        "were", "will", "with"
    ])

    def __init__(self):
        """Initializes the processor."""
        # Create a translation table to quickly remove punctuation
        self.punctuation_translator = str.maketrans('', '', string.punctuation)
        print("TextProcessor initialized.")

    def _clean_text(self, text: str) -> str:
        """Lowercases and removes punctuation from the text."""
        # 1. Lowercasing
        text = text.lower()
        # 2. Punctuation removal
        text = text.translate(self.punctuation_translator)
        return text

    def _tokenize(self, text: str) -> list[str]:
        """Splits the text into tokens (words)."""
        # Simple split by whitespace
        return text.split()

    def _remove_stop_words(self, tokens: list[str]) -> list[str]:
        """Removes common stop words from the list of tokens."""
        # 4. Stop word removal
        return [token for token in tokens if token not in self.STOP_WORDS]

    def preprocess(self, text: str) -> list[str]:
        """
        Performs the complete sequence of preprocessing steps on the input text.

        Args:
            text: The raw input string to be processed.

        Returns:
            A list of cleaned, tokenized, and stop-word-removed strings.
        """
        if not isinstance(text, str):
            raise TypeError("Input must be a string.")

        # Step 1 & 2: Clean text
        cleaned_text = self._clean_text(text)

        # Step 3: Tokenize
        tokens = self._tokenize(cleaned_text)

        # Step 4: Remove stop words
        processed_tokens = self._remove_stop_words(tokens)

        return processed_tokens

if __name__ == '__main__':
    # --- Demonstration ---
    processor = TextProcessor()

    sample_text = (
        "The quick brown fox jumps over the lazy dog. "
        "It was a sunny day, and the dogs were all playing in the park."
    )
    print("\n--- Original Text ---")
    print(sample_text)

    # Process the text
    final_tokens = processor.preprocess(sample_text)

    print("\n--- Processed Tokens ---")
    print(final_tokens)
    print(f"\nTotal tokens (before cleaning): {len(sample_text.split())}")
    print(f"Final tokens: {len(final_tokens)}")

    # Example with a sentence containing only stop words
    short_text = "It is an and the"
    print("\n--- Short Text Example ---")
    print(f"Original: '{short_text}'")
    final_tokens_short = processor.preprocess(short_text)
    print(f"Processed: {final_tokens_short}")