from enum import Enum
from typing import Any, Dict

from ...net.headers.base_header import BaseHeader


class DefaultHeadersKeys(Enum):
    """
    An enumeration of default headers.

    :ivar str ACCESS_AUTH: The access token authentication header.
    :ivar str BASIC_AUTH: The basic authentication header.
    :ivar str API_KEY_AUTH: The API key authentication header.
    """

    ACCESS_AUTH = "access_token_auth"
    BASIC_AUTH = "basic_auth"
    API_KEY_AUTH = "api_key_auth"


class DefaultHeaders:
    """
    A class to manage default headers.

    :ivar Dict[str, BaseHeader] _default_headers: The default headers.
    """

    def __init__(self):
        self._default_headers: Dict[str, BaseHeader] = {}

    def set_header(self, key: DefaultHeadersKeys, value: BaseHeader) -> None:
        """
        Set a default header.

        :param DefaultHeadersKeys key: The key of the header to set.
        :param Any value: The value to set for the header.
        """
        self._default_headers[key.value] = value

    def get_header(self, key: DefaultHeadersKeys) -> BaseHeader:
        """
        Get a default header.

        :param DefaultHeadersKeys key: The key of the header to get.
        :return: The value of the header.
        :rtype: Any
        """
        return self._default_headers[key.value]

    def get_headers(self) -> list:
        """
        Get the default headers.

        :return: A list of the default headers.
        :rtype: list
        """
        return list(self._default_headers.values())
