"""
________________________________________________________________________

:PROJECT: SiLA2_python

*SiLA2 Code Generator for using Packages*

:details: Base class to generate prototypes of a SiLA2 Client/Server

:file:    PrototypeGenerator.py
:authors: Timm Severin (timm.severin@tum.de)
          mark doerr  (mark.doerr@uni-greifswald.de)

:date: (creation)          2019-06-21
:date: (last modification) 2020-08-13

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

# import the general packages required
import os
import logging
import textwrap
from datetime import datetime
from sila2lib.smart_template.smart_template import SmartTemplate
from sila2lib.smart_template.hooks.code import indent as hook_indent, trim as hook_trim, wrap as hook_wrap
from sila2lib.smart_template.hooks.formatting import upper as hook_upper
from sila2lib.utils.utils import safe_write

# import modules from this package
from ..service_descriptor_parser import ServiceDescriptorParser

# import packages from the SiLA2 library
from sila2lib.fdl_parser.fdl_parser import FDLParser
from sila2lib.fdl_parser.command import Command

# meta information packages
from typing import Union, Any
from typing import Dict, List
from ..__meta__ import __version__, __app_name__


class PrototypeGenerator:
    """
    Base class for prototype generation
    """
    _template_dir: str
    _output_dir: str
    service_dict: Dict[str, Any]
    service_description: ServiceDescriptorParser
    fdl_parsers: Dict[str, FDLParser]

    # file extension of the C++ template files
    CPP_TEMPLATE_EXTENSION = "cppt"

    def __init__(self,
                 service_description: Union[ServiceDescriptorParser, None] = None,
                 output_dir: Union[str, None] = None,
                 template_dir: Union[str, None] = None,
                 fdl_path: Union[str, None] = None,
                 ignore_overwrite_warning: bool = False ):
        """
        Class initialiser

        :param service_description: Object of the service description. If None, an empty object will be created.
        :param output_dir: General output directory for the prototype files. If None, the method will try to derive it
                           from the service_description, if not possible it will be set to '.'.
        :param template_dir: Directory from where to load the templates. If None, the package provided template
                             directory "run-inline" will be used.
        :param fdl_path: Path to where **all** FDL input files are stored
        """
        self.ignore_overwrite_warning = ignore_overwrite_warning

        if service_description is None:
            self.service_description = ServiceDescriptorParser(input_file=None)
        else:
            self.service_description = service_description

        self.service_dict = self.service_description.description_dictionary

        if fdl_path:
            # Generate FDLParser objects for all sila features registered
            self.fdl_parsers = {}
            for feature_id in self.service_dict['SiLA_feature_list']:
                fdl_filename = os.path.join(fdl_path, feature_id) + '.sila.xml'
                self.fdl_parsers[feature_id] = FDLParser(fdl_filename=fdl_filename)

        # prepare a dictionary which can be filled by sub-classes and will be used in the template substitution process
        self.substitution_dict = {
            'note': 'Code generated by {app_name} {version}'.format(version=__version__, app_name=__app_name__),
            'creation_date': datetime.utcnow().isoformat(),
            'sila_framework': 'silaFW_pb2'
        }

        # auto generate a reasonable output directory (if not given)
        if output_dir is None:
            if service_description is None:
                output_dir = '.'
            else:
                output_dir = service_description.description_dictionary['service_name'] + '_build'
            logging.info('Set primary output directory to "{output_dir}"'.format(output_dir=output_dir))
        self._output_dir = output_dir

        if template_dir is None:
            template_dir = os.path.join(os.path.dirname(__file__), 'templates', 'run-inline')
            logging.info('Template directory for prototype generation set to "{template_dir}"'.format(
                template_dir=template_dir))
        self._template_dir = template_dir

    # noinspection PyIncorrectDocstring
    def write_from_template(self, *args, **kwargs) -> bool:
        """
        Writes the data stored in a dictionary into the given template.
            Template replacements will be read from self.service_dict.

        :param input_template: Name of the template file to use (without .pyt and path).
        :param template_ext: File extension of the input template file (default: 'pyt').
        :param output_filename: Name of the output file (can contain subdirectories, will be written to
                                self._output_dir).
        :param args: Further dictionaries can be passed that are used to substitute template variables.
                     These dictionaries take precedence over the internal service_dictionary from the
                     service_description in the order they are given.
        :param kwargs: Additional keyword/value pairs can be passed, which are also used as substitution variables
                       for the template.

        :returns: Result of the substitution and write process.

        .. note:: Further available template variables:

                  ================ ============================================
                  name             value
                  ================ ============================================
                  creation_date    Current date (UTC ISO)
                  output_filename  The basename of the written file
                  input_template   The name of the input template file
                  ================ ============================================

        """

        template_code = self.generate_from_template(module_name=os.path.splitext(kwargs['output_filename'])[0],
                                                    *args, **kwargs)
        # template_code = self.generate_from_template(input_template=kwargs['input_template'], *args, **kwargs)

        # update the output filename to include the output_dir
        output_filename = os.path.join(self._output_dir, kwargs['output_filename'])

        if template_code is not None:
            if self.ignore_overwrite_warning:
                with open(output_filename, 'w') as output_file:
                    output_file.write(template_code)
            else:
                safe_write(output_filename, template_code)

        else:
            return False

        return True

    # noinspection PyIncorrectDocstring
    def generate_from_template(self, *args, **kwargs) -> str:
        """
        Generate code from a template file and return that code.

        :param input_template: Name of the template file to use (without .pyt and path).
        :param template_ext: File extension of the input template file (default: 'pyt').
        :param args: Further dictionaries can be passed that are used to substitute template variables.
                     These dictionaries take precedence over the internal service_dictionary from the
                     service_description in the order they are given.
        :param kwargs: Additional keyword/value pairs can be passed, which are also used as substitution variables
                       for the template. They are applied before any other dictionary.

        :return: Generated code with all template substitutions included.

        .. note:: Further available template variables:

                  =============== ============================================
                  name            value
                  =============== ============================================
                  creation_date   Current date (UTC ISO)
                  input_template  The name of the input template file
                  =============== ============================================

        """

        # generate complete template path
        template_ext = kwargs.get("template_ext", "pyt")
        input_template = os.path.join(
            self._template_dir, (
                kwargs['input_template'] + os.extsep + template_ext
            )
        )

        with open(input_template, 'r') as template_file:
            template_object = SmartTemplate(template_file.read())

            # register hooks
            template_object.add_function_hook(name='indent', func=hook_indent, args=[int])
            template_object.add_function_hook(
                name='trim',
                func=hook_trim,
                args=[PrototypeGenerator._string_to_bool, PrototypeGenerator._string_to_bool]
            )
            template_object.add_function_hook(name="wrap", func=hook_wrap, args=[int])
            template_object.add_function_hook(name="upper", func=hook_upper, args=[int])
            template_object.add_function_hook(name="doc_wrap", func=self._doc_wrap, args=[int])

            # substitute in order of precedence: dictionaries in args, internal service_dict
            #   this merging process requires python >= 3.5
            merged_dict = {**self.service_dict, **self.substitution_dict}
            for arg_dict in args:
                merged_dict = {**merged_dict, **arg_dict}

            # we can directly pass on the kwargs, they are treated in the Template class
            template_code = template_object.substitute(merged_dict, **kwargs)

        return template_code

    def _generate_parameters_responses_description(self, command: Command) -> str:
        """
        Generates a description for the command's parameters and responses for the command's doc comment
        """
        def gen_description(items: tuple) -> str:
            description = ""
            for identifier, item in items:
                if identifier in ('EmptyParameter', 'EmptyResponse'):
                    continue
                temp = self.generate_from_template({
                        'identifier': identifier,
                        'description': item.description,
                    },
                    input_template='parameter_response_description',
                    template_ext=self.CPP_TEMPLATE_EXTENSION
                )
                description += self._doc_wrap(temp, 76) + '\n'

            description = description[:-1] if description else ' * None'
            return description

        return (gen_description(command.parameters.items()),
                gen_description(command.responses.items()))

    @staticmethod
    def _doc_wrap(input_string: str, width: int = 120) -> str:
        r"""
        Wraps a C++ doc string text to a maximum width.

        :param input_string: The string which should be wrapped.
        :param width: The maximum width allowed for a line in the text.

        :return: The input string with wrapped lines.
        """
        return hook_wrap(input_string, width - 3).replace('\n', '\n * ')

    @staticmethod
    def _string_to_bool(input_string: str) -> bool:
        """
        Convert a string to a boolean while evaluating the strings contents.
            This function will convert the input string into a boolean value, while evaluating the contents of the
            string. Thus it will do the following checks:

            * If the `input_string` is 'False' or '0', it will return `False`.
            * Otherwise it will return bool(input_string), which can still return `False` if e.g. the string is empty.

        :param input_string: The string to evaluate.

        :returns: The boolean value of the string.
        """
        if input_string.lower() == 'false' or input_string == '0':
            return False
        else:
            return bool(input_string)

    @staticmethod
    def get_template_dir(template_name: str = "run-inline") -> str:
        """
        Returns the complete path to a template directory inside this library.
            The method uses the name of a template to build a path to the template directory stored inside this package.

        :param template_name: Name of the template for which the path is to be generated.
        :returns: The path to the template directory. If the template does not exist, and empty string will be returned.

        .. note:: Currently defined template names are:

                  * run-inline
        """
        template_path = os.path.join(os.path.dirname(__file__), "..", "templates", template_name)
        if os.path.exists(template_path):
            return template_path
        else:
            return ""

    @staticmethod
    def get_templates() -> List[str]:
        """
        Returns a list of all templates available.

        :return: A list of all template directories found inside this package.
        """

        template_list = []

        for dir_entry in os.scandir(path=os.path.join(os.path.dirname(__file__), "..", "templates")):
            if dir_entry.is_dir:
                template_list.append(dir_entry.name)

        return template_list
