import { aws_iam as iam, aws_bedrockagentcore as acore } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * Properties for SimpleAgentCoreRuntime construct.
 */
export interface SimpleAgentCoreRuntimeProps {
    /**
     * The name of the Bedrock AgentCore runtime.
     * This will be used as the runtime name and default ECR repository name.
     */
    readonly agentName: string;
    /**
     * Path to the agent source code directory containing Dockerfile.
     */
    readonly agentSrcPath: string;
    /**
     * ECR repository name for the agent container image.
     * @default - Creates a new repository named with agentName
     */
    readonly ecrRepositoryName?: string;
    /**
     * IAM role for the AgentCore runtime execution.
     * @default - Creates a new role with required Bedrock AgentCore permissions
     */
    readonly runtimeExecutionRole?: iam.Role;
    /**
     * Network configuration for the AgentCore runtime.
     * @default - PUBLIC network mode: { networkMode: 'PUBLIC' }
     */
    readonly networkConfiguration?: acore.CfnRuntime.NetworkConfigurationProperty;
    /**
     * Environment variables to pass to the agent container.
     * @default - No environment variables
     */
    readonly environmentVariables?: Record<string, string>;
    /**
     * Description of the agent runtime.
     * @default - No description
     */
    readonly agentDescription?: string;
}
/**
 * A construct that creates an AWS Bedrock AgentCore runtime.
 *
 * This construct handles:
 * - Building and deploying Docker container images to ECR
 * - Creating IAM roles and service-linked roles
 * - Configuring Bedrock AgentCore runtime with network and environment settings
 *
 * @example
 * new SimpleAgentCoreRuntime(this, 'MyAgent', {
 *   agentName: 'my-bedrock-agent',
 *   agentSrcPath: './agent-code',
 * });
 */
export declare class SimpleAgentCoreRuntime extends Construct {
    /**
     * The IAM role used by the AgentCore runtime.
     */
    readonly runtimeExecutionRole: iam.Role;
    /**
     * The unique identifier of the AgentCore runtime.
     */
    readonly runtimeId: string;
    /**
     * The version of the AgentCore runtime.
     */
    readonly runtimeVersion: string;
    /**
     * The ARN of the AgentCore runtime.
     */
    readonly runtimeArn: string;
    constructor(scope: Construct, id: string, props: SimpleAgentCoreRuntimeProps);
    private getOrCreateEcrRepository;
    private buildDockerImage;
}
