"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleAgentCoreRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_ecr_deployment_1 = require("cdk-ecr-deployment");
const constructs_1 = require("constructs");
const iam_role_factory_1 = require("./iam-role-factory");
/**
 * A construct that creates an AWS Bedrock AgentCore runtime.
 *
 * This construct handles:
 * - Building and deploying Docker container images to ECR
 * - Creating IAM roles and service-linked roles
 * - Configuring Bedrock AgentCore runtime with network and environment settings
 *
 * @example
 * new SimpleAgentCoreRuntime(this, 'MyAgent', {
 *   agentName: 'my-bedrock-agent',
 *   agentSrcPath: './agent-code',
 * });
 */
class SimpleAgentCoreRuntime extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Validate agentName
        if (!props.agentName || props.agentName.trim().length === 0) {
            throw new Error('agentName must be a non-empty string');
        }
        if (props.agentName.length > 40) {
            throw new Error('agentName must be 40 characters or less');
        }
        if (!/^[a-z0-9_]+$/.test(props.agentName)) {
            throw new Error('agentName must contain only lowercase letters, numbers, and underscores (snake_case)');
        }
        // Validate agentSrcPath
        if (!props.agentSrcPath || props.agentSrcPath.trim().length === 0) {
            throw new Error('agentSrcPath must be a non-empty string');
        }
        const stack = aws_cdk_lib_1.Stack.of(this);
        // Create required service-linked roles for Bedrock AgentCore
        iam_role_factory_1.IamRoleFactory.createServiceLinkedRoles(this, id);
        // Set up runtime execution role
        this.runtimeExecutionRole = props.runtimeExecutionRole ?? iam_role_factory_1.IamRoleFactory.createRuntimeExecutionRole(this, id, props.agentName);
        // Build agent container image
        const image = this.buildDockerImage(id, props.agentSrcPath);
        // Deploy Container image to ECR
        const ecrRepoName = this.getOrCreateEcrRepository(id, props.agentName, props.ecrRepositoryName);
        const containerImageUri = `${stack.account}.dkr.ecr.${stack.region}.amazonaws.com/${ecrRepoName}:latest`;
        const ecrDeploy = new cdk_ecr_deployment_1.ECRDeployment(this, `${id}EcrDeploy`, {
            src: new cdk_ecr_deployment_1.DockerImageName(image.imageUri),
            dest: new cdk_ecr_deployment_1.DockerImageName(containerImageUri),
        });
        // Set up network configuration
        const networkConfiguration = props.networkConfiguration ?? { networkMode: 'PUBLIC' };
        // Create the Bedrock AgentCore runtime
        const cfnRuntime = new aws_cdk_lib_1.aws_bedrockagentcore.CfnRuntime(this, `${id}Runtime`, {
            agentRuntimeName: props.agentName,
            agentRuntimeArtifact: {
                containerConfiguration: {
                    containerUri: containerImageUri,
                },
            },
            networkConfiguration,
            roleArn: this.runtimeExecutionRole.roleArn,
            environmentVariables: props.environmentVariables,
        });
        cfnRuntime.node.addDependency(ecrDeploy);
        // Set up deployed attributes
        this.runtimeId = cfnRuntime.attrAgentRuntimeId;
        this.runtimeVersion = cfnRuntime.attrAgentRuntimeVersion;
        this.runtimeArn = cfnRuntime.attrAgentRuntimeArn;
    }
    getOrCreateEcrRepository(id, agentName, ecrRepositoryName) {
        if (ecrRepositoryName) {
            return ecrRepositoryName;
        }
        const ecrRepo = new aws_cdk_lib_1.aws_ecr.Repository(this, `${id}Ecr`, {
            repositoryName: agentName,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            emptyOnDelete: true,
        });
        return ecrRepo.repositoryName;
    }
    buildDockerImage(id, agentSrcPath) {
        return new aws_cdk_lib_1.aws_ecr_assets.DockerImageAsset(this, `${id}Image`, {
            directory: agentSrcPath,
            platform: aws_cdk_lib_1.aws_ecr_assets.Platform.LINUX_ARM64,
        });
    }
}
exports.SimpleAgentCoreRuntime = SimpleAgentCoreRuntime;
_a = JSII_RTTI_SYMBOL_1;
SimpleAgentCoreRuntime[_a] = { fqn: "simple-agentcore-runtime-patterns.SimpleAgentCoreRuntime", version: "0.0.0" };
//# sourceMappingURL=data:application/json;base64,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