import { aws_apigatewayv2 as apigwv2 } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * Configuration for a runtime invocation route.
 */
export interface RuntimeInvokeConfig {
    /**
     * ARN of the AgentCore runtime to invoke.
     */
    readonly runtimeArn: string;
    /**
     * HTTP route path for this runtime.
     */
    readonly routePath: string;
}
/**
 * Properties for HttpApiAgentCoreRuntimePattern construct.
 */
export interface HttpApiAgentCoreRuntimePatternProps {
    /**
     * Array of runtime configurations to expose via HTTP API.
     */
    readonly runtimes: RuntimeInvokeConfig[];
    /**
     * API key for simple authentication. Creates a Lambda authorizer.
     * Cannot be used with `authorizer`.
     * @default - No authentication
     */
    readonly authApiKey?: string;
    /**
     * Custom HTTP route authorizer.
     * Cannot be used with `authApiKey`.
     * @default - No authentication
     */
    readonly authorizer?: apigwv2.IHttpRouteAuthorizer;
}
/**
 * A construct that creates an HTTP API Gateway to invoke AgentCore runtimes.
 *
 * This construct handles:
 * - Creating HTTP API Gateway with CORS configuration
 * - Setting up Lambda functions to invoke AgentCore runtimes
 * - Optional API key or custom authentication
 * - Multiple runtime endpoints
 *
 * @example
 * new HttpApiAgentCoreRuntimePattern(this, 'HttpApi', {
 *   runtimes: [
 *     { runtimeArn: runtime.runtimeArn, routePath: '/agent' }
 *   ],
 *   authApiKey: 'my-secret-key',
 * });
 */
export declare class HttpApiAgentCoreRuntimePattern extends Construct {
    /**
     * The HTTP API Gateway instance.
     */
    readonly httpApi: apigwv2.HttpApi;
    /**
     * The URL of the HTTP API.
     */
    readonly apiUrl: string;
    /**
     * IAM role used by Lambda functions to invoke AgentCore runtimes.
     */
    private readonly lambdaRole;
    constructor(scope: Construct, id: string, props: HttpApiAgentCoreRuntimePatternProps);
    /**
     * Adds a new runtime invocation endpoint to the HTTP API.
     *
     * @param id - Unique identifier for CDK resource naming
     * @param config - Runtime configuration including ARN and route path
     */
    addRuntimeInvoke(id: string, config: RuntimeInvokeConfig): void;
}
