"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpApiAgentCoreRuntimePattern = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_lambda_python_alpha_1 = require("@aws-cdk/aws-lambda-python-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * A construct that creates an HTTP API Gateway to invoke AgentCore runtimes.
 *
 * This construct handles:
 * - Creating HTTP API Gateway with CORS configuration
 * - Setting up Lambda functions to invoke AgentCore runtimes
 * - Optional API key or custom authentication
 * - Multiple runtime endpoints
 *
 * @example
 * new HttpApiAgentCoreRuntimePattern(this, 'HttpApi', {
 *   runtimes: [
 *     { runtimeArn: runtime.runtimeArn, routePath: '/agent' }
 *   ],
 *   authApiKey: 'my-secret-key',
 * });
 */
class HttpApiAgentCoreRuntimePattern extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.authApiKey && props.authorizer) {
            throw new Error('Cannot specify both authApiKey and authorizer. Please provide only one.');
        }
        this.lambdaRole = new aws_cdk_lib_1.aws_iam.Role(this, `${id}LambdaRole`, {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        this.lambdaRole.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: ['bedrock-agentcore:InvokeAgentRuntime'],
            resources: ['arn:aws:bedrock-agentcore:*:*:runtime/*'],
        }));
        let defaultAuthorizer;
        if (props.authApiKey) {
            const authorizerFunction = new aws_lambda_python_alpha_1.PythonFunction(this, `${id}AuthFunction`, {
                entry: path.join(__dirname, '../lambda/simple-api-auth'),
                runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_13,
                architecture: aws_cdk_lib_1.aws_lambda.Architecture.ARM_64,
                timeout: aws_cdk_lib_1.Duration.seconds(30),
                environment: {
                    VALID_API_KEY: props.authApiKey,
                },
                logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, `${id}AuthFunctionLogGroup`, {
                    retention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
                }),
            });
            defaultAuthorizer = new aws_cdk_lib_1.aws_apigatewayv2_authorizers.HttpLambdaAuthorizer(`${id}Authorizer`, authorizerFunction, {
                authorizerName: `${id}-authorizer`,
                responseTypes: [aws_cdk_lib_1.aws_apigatewayv2_authorizers.HttpLambdaResponseType.SIMPLE],
                identitySource: ['$request.header.x-api-key'],
            });
        }
        else if (props.authorizer) {
            defaultAuthorizer = props.authorizer;
        }
        this.httpApi = new aws_cdk_lib_1.aws_apigatewayv2.HttpApi(this, `${id}HttpApi`, {
            apiName: `${id}-http-api`,
            defaultAuthorizer,
            corsPreflight: {
                allowOrigins: ['*'],
                allowMethods: [aws_cdk_lib_1.aws_apigatewayv2.CorsHttpMethod.POST, aws_cdk_lib_1.aws_apigatewayv2.CorsHttpMethod.OPTIONS],
                allowHeaders: ['*'],
            },
        });
        props.runtimes.forEach((runtime, index) => {
            this.addRuntimeInvoke(`Runtime${index}`, runtime);
        });
        this.apiUrl = this.httpApi.apiEndpoint;
    }
    /**
     * Adds a new runtime invocation endpoint to the HTTP API.
     *
     * @param id - Unique identifier for CDK resource naming
     * @param config - Runtime configuration including ARN and route path
     */
    addRuntimeInvoke(id, config) {
        const fnInvoke = new aws_lambda_python_alpha_1.PythonFunction(this, `${id}Function`, {
            entry: path.join(__dirname, '../lambda/agentcore-runtime-http-api'),
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_13,
            architecture: aws_cdk_lib_1.aws_lambda.Architecture.ARM_64,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: this.lambdaRole,
            environment: {
                ARN_AGENTCORE_RUNTIME: config.runtimeArn,
            },
            logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, `${id}FunctionLogGroup`, {
                retention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
            }),
        });
        const integration = new aws_cdk_lib_1.aws_apigatewayv2_integrations.HttpLambdaIntegration(`${id}Integration`, fnInvoke, {
            payloadFormatVersion: aws_cdk_lib_1.aws_apigatewayv2.PayloadFormatVersion.VERSION_2_0,
        });
        this.httpApi.addRoutes({
            path: config.routePath,
            methods: [aws_cdk_lib_1.aws_apigatewayv2.HttpMethod.POST],
            integration: integration,
        });
    }
}
exports.HttpApiAgentCoreRuntimePattern = HttpApiAgentCoreRuntimePattern;
_a = JSII_RTTI_SYMBOL_1;
HttpApiAgentCoreRuntimePattern[_a] = { fqn: "simple-agentcore-runtime-patterns.HttpApiAgentCoreRuntimePattern", version: "0.0.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaHR0cC1hcGktYWdlbnRjb3JlLXJ1bnRpbWUtcGF0dGVybi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9odHRwLWFwaS1hZ2VudGNvcmUtcnVudGltZS1wYXR0ZXJuLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkJBQTZCO0FBQzdCLDhFQUFrRTtBQUNsRSw2Q0FRcUI7QUFDckIsMkNBQXVDO0FBeUN2Qzs7Ozs7Ozs7Ozs7Ozs7OztHQWdCRztBQUNILE1BQWEsOEJBQStCLFNBQVEsc0JBQVM7SUFnQjNELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMEM7UUFDbEYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDO1lBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMseUVBQXlFLENBQUMsQ0FBQztRQUM3RixDQUFDO1FBRUQsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLHFCQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsWUFBWSxFQUFFO1lBQ3RELFNBQVMsRUFBRSxJQUFJLHFCQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDM0QsZUFBZSxFQUFFO2dCQUNmLHFCQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDBDQUEwQyxDQUFDO2FBQ3ZGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztZQUNsRCxNQUFNLEVBQUUscUJBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQztZQUNqRCxTQUFTLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQztTQUN2RCxDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksaUJBQTJELENBQUM7UUFFaEUsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDckIsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLHdDQUFjLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxjQUFjLEVBQUU7Z0JBQ3ZFLEtBQUssRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwyQkFBMkIsQ0FBQztnQkFDeEQsT0FBTyxFQUFFLHdCQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7Z0JBQ25DLFlBQVksRUFBRSx3QkFBTSxDQUFDLFlBQVksQ0FBQyxNQUFNO2dCQUN4QyxPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2dCQUM3QixXQUFXLEVBQUU7b0JBQ1gsYUFBYSxFQUFFLEtBQUssQ0FBQyxVQUFVO2lCQUNoQztnQkFDRCxRQUFRLEVBQUUsSUFBSSxzQkFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsR0FBRyxFQUFFLHNCQUFzQixFQUFFO29CQUM3RCxTQUFTLEVBQUUsc0JBQUksQ0FBQyxhQUFhLENBQUMsUUFBUTtpQkFDdkMsQ0FBQzthQUNILENBQUMsQ0FBQztZQUVILGlCQUFpQixHQUFHLElBQUksMENBQVksQ0FBQyxvQkFBb0IsQ0FBQyxHQUFHLEVBQUUsWUFBWSxFQUFFLGtCQUFrQixFQUFFO2dCQUMvRixjQUFjLEVBQUUsR0FBRyxFQUFFLGFBQWE7Z0JBQ2xDLGFBQWEsRUFBRSxDQUFDLDBDQUFZLENBQUMsc0JBQXNCLENBQUMsTUFBTSxDQUFDO2dCQUMzRCxjQUFjLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQzthQUM5QyxDQUFDLENBQUM7UUFDTCxDQUFDO2FBQU0sSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDNUIsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUN2QyxDQUFDO1FBRUQsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLDhCQUFPLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsU0FBUyxFQUFFO1lBQ3ZELE9BQU8sRUFBRSxHQUFHLEVBQUUsV0FBVztZQUN6QixpQkFBaUI7WUFDakIsYUFBYSxFQUFFO2dCQUNiLFlBQVksRUFBRSxDQUFDLEdBQUcsQ0FBQztnQkFDbkIsWUFBWSxFQUFFLENBQUMsOEJBQU8sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLDhCQUFPLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQztnQkFDM0UsWUFBWSxFQUFFLENBQUMsR0FBRyxDQUFDO2FBQ3BCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsS0FBSyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUU7WUFDeEMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsS0FBSyxFQUFFLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDcEQsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGdCQUFnQixDQUFDLEVBQVUsRUFBRSxNQUEyQjtRQUM3RCxNQUFNLFFBQVEsR0FBRyxJQUFJLHdDQUFjLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxVQUFVLEVBQUU7WUFDekQsS0FBSyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNDQUFzQyxDQUFDO1lBQ25FLE9BQU8sRUFBRSx3QkFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLFlBQVksRUFBRSx3QkFBTSxDQUFDLFlBQVksQ0FBQyxNQUFNO1lBQ3hDLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsSUFBSSxFQUFFLElBQUksQ0FBQyxVQUFVO1lBQ3JCLFdBQVcsRUFBRTtnQkFDWCxxQkFBcUIsRUFBRSxNQUFNLENBQUMsVUFBVTthQUN6QztZQUNELFFBQVEsRUFBRSxJQUFJLHNCQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQ3pELFNBQVMsRUFBRSxzQkFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRO2FBQ3ZDLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsR0FBRyxJQUFJLDJDQUFtQixDQUFDLHFCQUFxQixDQUMvRCxHQUFHLEVBQUUsYUFBYSxFQUNsQixRQUFRLEVBQ1I7WUFDRSxvQkFBb0IsRUFBRSw4QkFBTyxDQUFDLG9CQUFvQixDQUFDLFdBQVc7U0FDL0QsQ0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUM7WUFDckIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxTQUFTO1lBQ3RCLE9BQU8sRUFBRSxDQUFDLDhCQUFPLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQztZQUNsQyxXQUFXLEVBQUUsV0FBVztTQUN6QixDQUFDLENBQUM7SUFDTCxDQUFDOztBQWhISCx3RUFpSEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgUHl0aG9uRnVuY3Rpb24gfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhLXB5dGhvbi1hbHBoYSc7XG5pbXBvcnQge1xuICBEdXJhdGlvbixcbiAgYXdzX2lhbSBhcyBpYW0sXG4gIGF3c19sYW1iZGEgYXMgbGFtYmRhLFxuICBhd3NfbG9ncyBhcyBsb2dzLFxuICBhd3NfYXBpZ2F0ZXdheXYyIGFzIGFwaWd3djIsXG4gIGF3c19hcGlnYXRld2F5djJfaW50ZWdyYXRpb25zIGFzIGFwaWd3djJfaW50ZWdyYXRpb24sXG4gIGF3c19hcGlnYXRld2F5djJfYXV0aG9yaXplcnMgYXMgYXBpZ3d2Ml9hdXRoLFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBhIHJ1bnRpbWUgaW52b2NhdGlvbiByb3V0ZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSdW50aW1lSW52b2tlQ29uZmlnIHtcbiAgLyoqXG4gICAqIEFSTiBvZiB0aGUgQWdlbnRDb3JlIHJ1bnRpbWUgdG8gaW52b2tlLlxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBIVFRQIHJvdXRlIHBhdGggZm9yIHRoaXMgcnVudGltZS5cbiAgICovXG4gIHJlYWRvbmx5IHJvdXRlUGF0aDogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIEh0dHBBcGlBZ2VudENvcmVSdW50aW1lUGF0dGVybiBjb25zdHJ1Y3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSHR0cEFwaUFnZW50Q29yZVJ1bnRpbWVQYXR0ZXJuUHJvcHMge1xuICAvKipcbiAgICogQXJyYXkgb2YgcnVudGltZSBjb25maWd1cmF0aW9ucyB0byBleHBvc2UgdmlhIEhUVFAgQVBJLlxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZXM6IFJ1bnRpbWVJbnZva2VDb25maWdbXTtcblxuICAvKipcbiAgICogQVBJIGtleSBmb3Igc2ltcGxlIGF1dGhlbnRpY2F0aW9uLiBDcmVhdGVzIGEgTGFtYmRhIGF1dGhvcml6ZXIuXG4gICAqIENhbm5vdCBiZSB1c2VkIHdpdGggYGF1dGhvcml6ZXJgLlxuICAgKiBAZGVmYXVsdCAtIE5vIGF1dGhlbnRpY2F0aW9uXG4gICAqL1xuICByZWFkb25seSBhdXRoQXBpS2V5Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDdXN0b20gSFRUUCByb3V0ZSBhdXRob3JpemVyLlxuICAgKiBDYW5ub3QgYmUgdXNlZCB3aXRoIGBhdXRoQXBpS2V5YC5cbiAgICogQGRlZmF1bHQgLSBObyBhdXRoZW50aWNhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgYXV0aG9yaXplcj86IGFwaWd3djIuSUh0dHBSb3V0ZUF1dGhvcml6ZXI7XG59XG5cbi8qKlxuICogQSBjb25zdHJ1Y3QgdGhhdCBjcmVhdGVzIGFuIEhUVFAgQVBJIEdhdGV3YXkgdG8gaW52b2tlIEFnZW50Q29yZSBydW50aW1lcy5cbiAqXG4gKiBUaGlzIGNvbnN0cnVjdCBoYW5kbGVzOlxuICogLSBDcmVhdGluZyBIVFRQIEFQSSBHYXRld2F5IHdpdGggQ09SUyBjb25maWd1cmF0aW9uXG4gKiAtIFNldHRpbmcgdXAgTGFtYmRhIGZ1bmN0aW9ucyB0byBpbnZva2UgQWdlbnRDb3JlIHJ1bnRpbWVzXG4gKiAtIE9wdGlvbmFsIEFQSSBrZXkgb3IgY3VzdG9tIGF1dGhlbnRpY2F0aW9uXG4gKiAtIE11bHRpcGxlIHJ1bnRpbWUgZW5kcG9pbnRzXG4gKlxuICogQGV4YW1wbGVcbiAqIG5ldyBIdHRwQXBpQWdlbnRDb3JlUnVudGltZVBhdHRlcm4odGhpcywgJ0h0dHBBcGknLCB7XG4gKiAgIHJ1bnRpbWVzOiBbXG4gKiAgICAgeyBydW50aW1lQXJuOiBydW50aW1lLnJ1bnRpbWVBcm4sIHJvdXRlUGF0aDogJy9hZ2VudCcgfVxuICogICBdLFxuICogICBhdXRoQXBpS2V5OiAnbXktc2VjcmV0LWtleScsXG4gKiB9KTtcbiAqL1xuZXhwb3J0IGNsYXNzIEh0dHBBcGlBZ2VudENvcmVSdW50aW1lUGF0dGVybiBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgSFRUUCBBUEkgR2F0ZXdheSBpbnN0YW5jZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBodHRwQXBpOiBhcGlnd3YyLkh0dHBBcGk7XG5cbiAgLyoqXG4gICAqIFRoZSBVUkwgb2YgdGhlIEhUVFAgQVBJLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFwaVVybDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJQU0gcm9sZSB1c2VkIGJ5IExhbWJkYSBmdW5jdGlvbnMgdG8gaW52b2tlIEFnZW50Q29yZSBydW50aW1lcy5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgbGFtYmRhUm9sZTogaWFtLlJvbGU7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEh0dHBBcGlBZ2VudENvcmVSdW50aW1lUGF0dGVyblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmIChwcm9wcy5hdXRoQXBpS2V5ICYmIHByb3BzLmF1dGhvcml6ZXIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHNwZWNpZnkgYm90aCBhdXRoQXBpS2V5IGFuZCBhdXRob3JpemVyLiBQbGVhc2UgcHJvdmlkZSBvbmx5IG9uZS4nKTtcbiAgICB9XG5cbiAgICB0aGlzLmxhbWJkYVJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgYCR7aWR9TGFtYmRhUm9sZWAsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZScpLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHRoaXMubGFtYmRhUm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbJ2JlZHJvY2stYWdlbnRjb3JlOkludm9rZUFnZW50UnVudGltZSddLFxuICAgICAgcmVzb3VyY2VzOiBbJ2Fybjphd3M6YmVkcm9jay1hZ2VudGNvcmU6KjoqOnJ1bnRpbWUvKiddLFxuICAgIH0pKTtcblxuICAgIGxldCBkZWZhdWx0QXV0aG9yaXplcjogYXBpZ3d2Mi5JSHR0cFJvdXRlQXV0aG9yaXplciB8IHVuZGVmaW5lZDtcblxuICAgIGlmIChwcm9wcy5hdXRoQXBpS2V5KSB7XG4gICAgICBjb25zdCBhdXRob3JpemVyRnVuY3Rpb24gPSBuZXcgUHl0aG9uRnVuY3Rpb24odGhpcywgYCR7aWR9QXV0aEZ1bmN0aW9uYCwge1xuICAgICAgICBlbnRyeTogcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2xhbWJkYS9zaW1wbGUtYXBpLWF1dGgnKSxcbiAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfMTMsXG4gICAgICAgIGFyY2hpdGVjdHVyZTogbGFtYmRhLkFyY2hpdGVjdHVyZS5BUk1fNjQsXG4gICAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoMzApLFxuICAgICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICAgIFZBTElEX0FQSV9LRVk6IHByb3BzLmF1dGhBcGlLZXksXG4gICAgICAgIH0sXG4gICAgICAgIGxvZ0dyb3VwOiBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCBgJHtpZH1BdXRoRnVuY3Rpb25Mb2dHcm91cGAsIHtcbiAgICAgICAgICByZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICAgICAgfSksXG4gICAgICB9KTtcblxuICAgICAgZGVmYXVsdEF1dGhvcml6ZXIgPSBuZXcgYXBpZ3d2Ml9hdXRoLkh0dHBMYW1iZGFBdXRob3JpemVyKGAke2lkfUF1dGhvcml6ZXJgLCBhdXRob3JpemVyRnVuY3Rpb24sIHtcbiAgICAgICAgYXV0aG9yaXplck5hbWU6IGAke2lkfS1hdXRob3JpemVyYCxcbiAgICAgICAgcmVzcG9uc2VUeXBlczogW2FwaWd3djJfYXV0aC5IdHRwTGFtYmRhUmVzcG9uc2VUeXBlLlNJTVBMRV0sXG4gICAgICAgIGlkZW50aXR5U291cmNlOiBbJyRyZXF1ZXN0LmhlYWRlci54LWFwaS1rZXknXSxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSBpZiAocHJvcHMuYXV0aG9yaXplcikge1xuICAgICAgZGVmYXVsdEF1dGhvcml6ZXIgPSBwcm9wcy5hdXRob3JpemVyO1xuICAgIH1cblxuICAgIHRoaXMuaHR0cEFwaSA9IG5ldyBhcGlnd3YyLkh0dHBBcGkodGhpcywgYCR7aWR9SHR0cEFwaWAsIHtcbiAgICAgIGFwaU5hbWU6IGAke2lkfS1odHRwLWFwaWAsXG4gICAgICBkZWZhdWx0QXV0aG9yaXplcixcbiAgICAgIGNvcnNQcmVmbGlnaHQ6IHtcbiAgICAgICAgYWxsb3dPcmlnaW5zOiBbJyonXSxcbiAgICAgICAgYWxsb3dNZXRob2RzOiBbYXBpZ3d2Mi5Db3JzSHR0cE1ldGhvZC5QT1NULCBhcGlnd3YyLkNvcnNIdHRwTWV0aG9kLk9QVElPTlNdLFxuICAgICAgICBhbGxvd0hlYWRlcnM6IFsnKiddLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHByb3BzLnJ1bnRpbWVzLmZvckVhY2goKHJ1bnRpbWUsIGluZGV4KSA9PiB7XG4gICAgICB0aGlzLmFkZFJ1bnRpbWVJbnZva2UoYFJ1bnRpbWUke2luZGV4fWAsIHJ1bnRpbWUpO1xuICAgIH0pO1xuXG4gICAgdGhpcy5hcGlVcmwgPSB0aGlzLmh0dHBBcGkuYXBpRW5kcG9pbnQ7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIG5ldyBydW50aW1lIGludm9jYXRpb24gZW5kcG9pbnQgdG8gdGhlIEhUVFAgQVBJLlxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBVbmlxdWUgaWRlbnRpZmllciBmb3IgQ0RLIHJlc291cmNlIG5hbWluZ1xuICAgKiBAcGFyYW0gY29uZmlnIC0gUnVudGltZSBjb25maWd1cmF0aW9uIGluY2x1ZGluZyBBUk4gYW5kIHJvdXRlIHBhdGhcbiAgICovXG4gIHB1YmxpYyBhZGRSdW50aW1lSW52b2tlKGlkOiBzdHJpbmcsIGNvbmZpZzogUnVudGltZUludm9rZUNvbmZpZyk6IHZvaWQge1xuICAgIGNvbnN0IGZuSW52b2tlID0gbmV3IFB5dGhvbkZ1bmN0aW9uKHRoaXMsIGAke2lkfUZ1bmN0aW9uYCwge1xuICAgICAgZW50cnk6IHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9sYW1iZGEvYWdlbnRjb3JlLXJ1bnRpbWUtaHR0cC1hcGknKSxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzEzLFxuICAgICAgYXJjaGl0ZWN0dXJlOiBsYW1iZGEuQXJjaGl0ZWN0dXJlLkFSTV82NCxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgcm9sZTogdGhpcy5sYW1iZGFSb2xlLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgQVJOX0FHRU5UQ09SRV9SVU5USU1FOiBjb25maWcucnVudGltZUFybixcbiAgICAgIH0sXG4gICAgICBsb2dHcm91cDogbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgYCR7aWR9RnVuY3Rpb25Mb2dHcm91cGAsIHtcbiAgICAgICAgcmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX1dFRUssXG4gICAgICB9KSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGludGVncmF0aW9uID0gbmV3IGFwaWd3djJfaW50ZWdyYXRpb24uSHR0cExhbWJkYUludGVncmF0aW9uKFxuICAgICAgYCR7aWR9SW50ZWdyYXRpb25gLFxuICAgICAgZm5JbnZva2UsXG4gICAgICB7XG4gICAgICAgIHBheWxvYWRGb3JtYXRWZXJzaW9uOiBhcGlnd3YyLlBheWxvYWRGb3JtYXRWZXJzaW9uLlZFUlNJT05fMl8wLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgdGhpcy5odHRwQXBpLmFkZFJvdXRlcyh7XG4gICAgICBwYXRoOiBjb25maWcucm91dGVQYXRoLFxuICAgICAgbWV0aG9kczogW2FwaWd3djIuSHR0cE1ldGhvZC5QT1NUXSxcbiAgICAgIGludGVncmF0aW9uOiBpbnRlZ3JhdGlvbixcbiAgICB9KTtcbiAgfVxufVxuIl19