"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IamRoleFactory = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
/**
 * Factory class for creating IAM roles and service-linked roles required by Bedrock AgentCore.
 */
class IamRoleFactory {
    /**
     * Creates an IAM role for Bedrock AgentCore runtime execution.
     *
     * The role includes permissions for:
     * - ECR image access
     * - CloudWatch Logs
     * - X-Ray tracing
     * - CloudWatch metrics
     * - Bedrock model invocation
     * - AgentCore workload identity
     *
     * @param ctx - The parent construct
     * @param id - The construct ID
     * @param agentName - The name of the agent runtime
     * @returns The created IAM role
     */
    static createRuntimeExecutionRole(ctx, id, agentName) {
        const stack = aws_cdk_lib_1.Stack.of(ctx);
        const role = new aws_cdk_lib_1.aws_iam.Role(ctx, `${id}ExecRole`, {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('bedrock-agentcore.amazonaws.com').withConditions({
                StringEquals: {
                    'aws:SourceAccount': stack.account,
                },
                ArnLike: {
                    'aws:SourceArn': `arn:aws:bedrock-agentcore:${stack.region}:${stack.account}:*`,
                },
            }),
            inlinePolicies: {
                AgentCoreExecutionPolicy: this.createExecutionPolicy(stack, agentName),
            },
        });
        this.addNagSuppressions(role, stack, agentName);
        return role;
    }
    /**
     * Creates service-linked roles required by Bedrock AgentCore.
     *
     * This method creates two service-linked roles:
     * - Network SLR for network configuration
     * - Runtime Identity SLR for workload identity management
     *
     * If the roles already exist, the errors are ignored.
     *
     * @param ctx - The parent construct
     * @param id - The construct ID
     */
    static createServiceLinkedRoles(ctx, id) {
        // Create Network SLR with error handling
        new aws_cdk_lib_1.custom_resources.AwsCustomResource(ctx, `${id}NetworkSLR`, {
            onCreate: {
                service: 'IAM',
                action: 'createServiceLinkedRole',
                parameters: {
                    AWSServiceName: 'network.bedrock-agentcore.amazonaws.com',
                    Description: 'Service-linked role for Bedrock AgentCore Network',
                },
                physicalResourceId: aws_cdk_lib_1.custom_resources.PhysicalResourceId.of('bedrock-agentcore-network-slr'),
                ignoreErrorCodesMatching: 'InvalidInput',
            },
            policy: aws_cdk_lib_1.custom_resources.AwsCustomResourcePolicy.fromSdkCalls({
                resources: aws_cdk_lib_1.custom_resources.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // Create Runtime Identity SLR with error handling
        new aws_cdk_lib_1.custom_resources.AwsCustomResource(ctx, `${id}RuntimeSLR`, {
            onCreate: {
                service: 'IAM',
                action: 'createServiceLinkedRole',
                parameters: {
                    AWSServiceName: 'runtime-identity.bedrock-agentcore.amazonaws.com',
                    Description: 'Service-linked role for Bedrock AgentCore Runtime Identity',
                },
                physicalResourceId: aws_cdk_lib_1.custom_resources.PhysicalResourceId.of('bedrock-agentcore-runtime-slr'),
                ignoreErrorCodesMatching: 'InvalidInput',
            },
            policy: aws_cdk_lib_1.custom_resources.AwsCustomResourcePolicy.fromSdkCalls({
                resources: aws_cdk_lib_1.custom_resources.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
    }
    static createExecutionPolicy(stack, agentName) {
        return new aws_cdk_lib_1.aws_iam.PolicyDocument({
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    sid: 'ECRImageAccess',
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['ecr:BatchGetImage', 'ecr:GetDownloadUrlForLayer'],
                    resources: [`arn:aws:ecr:${stack.region}:${stack.account}:repository/*`],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['logs:DescribeLogStreams', 'logs:CreateLogGroup'],
                    resources: [`arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/bedrock-agentcore/runtimes/*`],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['logs:DescribeLogGroups'],
                    resources: [`arn:aws:logs:${stack.region}:${stack.account}:log-group:*`],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['logs:CreateLogStream', 'logs:PutLogEvents'],
                    resources: [`arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/bedrock-agentcore/runtimes/*:log-stream:*`],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    sid: 'ECRTokenAccess',
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['ecr:GetAuthorizationToken'],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords', 'xray:GetSamplingRules', 'xray:GetSamplingTargets'],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['cloudwatch:PutMetricData'],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'cloudwatch:namespace': 'bedrock-agentcore',
                        },
                    },
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    sid: 'GetAgentAccessToken',
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        'bedrock-agentcore:GetWorkloadAccessToken',
                        'bedrock-agentcore:GetWorkloadAccessTokenForJWT',
                        'bedrock-agentcore:GetWorkloadAccessTokenForUserId',
                    ],
                    resources: [
                        `arn:aws:bedrock-agentcore:${stack.region}:${stack.account}:workload-identity-directory/default`,
                        `arn:aws:bedrock-agentcore:${stack.region}:${stack.account}:workload-identity-directory/default/workload-identity/${agentName}-*`,
                    ],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    sid: 'BedrockModelInvocation',
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['bedrock:InvokeModel', 'bedrock:InvokeModelWithResponseStream'],
                    resources: [
                        'arn:aws:bedrock:*::foundation-model/*',
                        `arn:aws:bedrock:${stack.region}:${stack.account}:*`,
                    ],
                }),
            ],
        });
    }
    static addNagSuppressions(role, stack, agentName) {
        cdk_nag_1.NagSuppressions.addResourceSuppressions(role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'ECR repository wildcard required for AgentCore to access multiple container images',
                appliesTo: [`Resource::arn:aws:ecr:${stack.region}:${stack.account}:repository/*`],
            },
            {
                id: 'AwsSolutions-IAM5',
                reason: 'CloudWatch Logs wildcard required for AgentCore runtime logging',
                appliesTo: [`Resource::arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/bedrock-agentcore/runtimes/*`],
            },
            {
                id: 'AwsSolutions-IAM5',
                reason: 'CloudWatch Logs describe operation requires wildcard access',
                appliesTo: [`Resource::arn:aws:logs:${stack.region}:${stack.account}:log-group:*`],
            },
            {
                id: 'AwsSolutions-IAM5',
                reason: 'CloudWatch Logs stream wildcard required for AgentCore runtime logging',
                appliesTo: [`Resource::arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/bedrock-agentcore/runtimes/*:log-stream:*`],
            },
            {
                id: 'AwsSolutions-IAM5',
                reason: 'ECR GetAuthorizationToken, X-Ray, and CloudWatch operations require global wildcard access',
                appliesTo: ['Resource::*'],
            },
            {
                id: 'AwsSolutions-IAM5',
                reason: 'Bedrock AgentCore workload identity wildcard required for agent runtime access',
                appliesTo: [`Resource::arn:aws:bedrock-agentcore:${stack.region}:${stack.account}:workload-identity-directory/default/workload-identity/${agentName}-*`],
            },
            {
                id: 'AwsSolutions-IAM5',
                reason: 'Bedrock foundation model wildcard required for cross-region model access',
                appliesTo: ['Resource::arn:aws:bedrock:*::foundation-model/*'],
            },
            {
                id: 'AwsSolutions-IAM5',
                reason: 'Bedrock service wildcard required for AgentCore operations',
                appliesTo: [`Resource::arn:aws:bedrock:${stack.region}:${stack.account}:*`],
            },
        ]);
    }
}
exports.IamRoleFactory = IamRoleFactory;
_a = JSII_RTTI_SYMBOL_1;
IamRoleFactory[_a] = { fqn: "simple-agentcore-runtime-patterns.IamRoleFactory", version: "0.0.1" };
//# sourceMappingURL=data:application/json;base64,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