"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebsocketAgentCoreRuntimePattern = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_lambda_python_alpha_1 = require("@aws-cdk/aws-lambda-python-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigatewayv2_integrations_1 = require("aws-cdk-lib/aws-apigatewayv2-integrations");
const constructs_1 = require("constructs");
/**
 * A construct that creates a WebSocket API Gateway to invoke AgentCore runtimes with streaming support.
 *
 * This construct handles:
 * - Creating WebSocket API Gateway with connect/disconnect/message routes
 * - Setting up Lambda function to invoke AgentCore runtime
 * - Optional API key or custom authentication
 * - Streaming response support via WebSocket
 *
 * @example
 * new WebsocketAgentCoreRuntimePattern(this, 'WebSocket', {
 *   runtimeArn: runtime.runtimeArn,
 *   authApiKey: 'my-secret-key',
 * });
 */
class WebsocketAgentCoreRuntimePattern extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.authApiKey && props.authorizer) {
            throw new Error('Cannot specify both authApiKey and authorizer. Please provide only one.');
        }
        const lambdaRole = new aws_cdk_lib_1.aws_iam.Role(this, `${id}LambdaRole`, {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        lambdaRole.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: ['bedrock-agentcore:InvokeAgentRuntime'],
            resources: ['arn:aws:bedrock-agentcore:*:*:runtime/*'],
        }));
        const fnWebSocket = new aws_lambda_python_alpha_1.PythonFunction(this, `${id}Function`, {
            entry: path.join(__dirname, '../lambda/agentcore-runtime-websocket-api'),
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_13,
            architecture: aws_cdk_lib_1.aws_lambda.Architecture.ARM_64,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: lambdaRole,
            environment: {
                ARN_AGENTCORE_RUNTIME: props.runtimeArn,
            },
            logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, `${id}FunctionLogGroup`, {
                retention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
            }),
        });
        let wsAuthorizer;
        if (props.authApiKey) {
            const authorizerFunction = new aws_lambda_python_alpha_1.PythonFunction(this, `${id}AuthFunction`, {
                entry: path.join(__dirname, '../lambda/simple-api-auth'),
                runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_13,
                architecture: aws_cdk_lib_1.aws_lambda.Architecture.ARM_64,
                timeout: aws_cdk_lib_1.Duration.seconds(30),
                environment: {
                    VALID_API_KEY: props.authApiKey,
                },
                logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, `${id}AuthFunctionLogGroup`, {
                    retention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
                }),
            });
            wsAuthorizer = new aws_cdk_lib_1.aws_apigatewayv2_authorizers.WebSocketLambdaAuthorizer(`${id}Authorizer`, authorizerFunction, {
                authorizerName: `${id}-authorizer`,
                identitySource: ['route.request.header.x-api-key'],
            });
        }
        else if (props.authorizer) {
            wsAuthorizer = props.authorizer;
        }
        this.webSocketApi = new aws_cdk_lib_1.aws_apigatewayv2.WebSocketApi(this, `${id}WebSocketApi`, {
            apiName: `${id}-websocket-api`,
            connectRouteOptions: {
                integration: new aws_apigatewayv2_integrations_1.WebSocketLambdaIntegration(`${id}ConnectIntegration`, fnWebSocket),
                authorizer: wsAuthorizer,
            },
            disconnectRouteOptions: {
                integration: new aws_apigatewayv2_integrations_1.WebSocketLambdaIntegration(`${id}DisconnectIntegration`, fnWebSocket),
            },
        });
        this.webSocketApi.addRoute(props.routePath || 'message', {
            integration: new aws_apigatewayv2_integrations_1.WebSocketLambdaIntegration(`${id}MessageIntegration`, fnWebSocket),
        });
        this.webSocketStage = new aws_cdk_lib_1.aws_apigatewayv2.WebSocketStage(this, `${id}WebSocketStage`, {
            webSocketApi: this.webSocketApi,
            stageName: 'prod',
            autoDeploy: true,
        });
        fnWebSocket.grantInvoke(new aws_cdk_lib_1.aws_iam.ServicePrincipal('apigateway.amazonaws.com'));
        fnWebSocket.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: ['execute-api:ManageConnections'],
            resources: ['arn:aws:execute-api:*:*:*'],
        }));
        this.webSocketUrl = this.webSocketStage.url;
    }
}
exports.WebsocketAgentCoreRuntimePattern = WebsocketAgentCoreRuntimePattern;
_a = JSII_RTTI_SYMBOL_1;
WebsocketAgentCoreRuntimePattern[_a] = { fqn: "simple-agentcore-runtime-patterns.WebsocketAgentCoreRuntimePattern", version: "0.0.1" };
//# sourceMappingURL=data:application/json;base64,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