r'''
# Simple AgentCore Runtime Patterns

An AWS CDK construct library for deploying AWS Bedrock AgentCore runtimes.

This library helps you deploy containerized AI agents to AWS Bedrock AgentCore using AWS CDK. It handles Docker image building, ECR deployment, IAM roles, and runtime configuration automatically.

## Requirements

* Node.js 22 or later
* Docker or Finch (for building container images)
* AWS CDK v2.221.0 or later

## Quick Start

### AWS CDK(TypeScript)

#### Install

```bash
npm install simple-agentcore-runtime-patterns
```

#### Example

```python
import {
  SimpleAgentCoreRuntime,
  HttpApiAgentCoreRuntimePattern,
  WebsocketAgentCoreRuntimePattern,
  LambdaUrlStreamingAgentCoreRuntimePattern,
} from "simple-agentcore-runtime-patterns";

// Create AgentCore Runtime
const acr = new SimpleAgentCoreRuntime(stack, "MyAgent", {
  agentName: "my_bedrock_agent", // Required: snake_case, max 40 chars
  agentSrcPath: "./my-agent-code", // Required: path to your agent code
});

// [Pattern 1] HTTP API
const httpApi = new HttpApiAgentCoreRuntimePattern(this, "HttpApiPattern", {
  runtimes: [{ runtimeArn: acr.runtimeArn, routePath: "/sync" }],
  authApiKey: "prototype",
});

// [Pattern 2] WebSocket for streaming response
const websocket = new WebsocketAgentCoreRuntimePattern(
  this,
  "WebsocketPattern",
  {
    runtimeArn: acr.runtimeArn,
    authApiKey: "prototype",
  }
);

// [Pattern 3] Lambda URL for streaming response
const lambdaUrl = new LambdaUrlStreamingAgentCoreRuntimePattern(
  this,
  "LambdaUrlPattern",
  {
    runtimeArn: acr.runtimeArn,
  }
);
```

### AWS CDK(Python)

#### Install

```bash
pip install simple-agentcore-runtime-patterns
```

#### Example

```python
from simple_agentcore_runtime_patterns import SimpleAgentCoreRuntime
from simple_agentcore_runtime_patterns import HttpApiAgentCoreRuntimePattern, WebsocketAgentCoreRuntimePattern, LambdaUrlStreamingAgentCoreRuntimePattern

# Create AgentCore Runtime
acr = SimpleAgentCoreRuntime(stack, "MyAgent",
    agent_name="my_bedrock_agent",      # Required: snake_case, max 40 chars
    agent_src_path="./my-agent-code",   # Required: path to your agent code
)

# [Pattern 1] HTTP API
http_api = HttpApiAgentCoreRuntimePattern(self, "HttpApiPattern", runtimes=[{"runtimeArn": acr.runtime_arn, "routePath": "/sync"}], auth_api_key="prototype")

# [Pattern 2] WebSocket for streaming response
websocket = WebsocketAgentCoreRuntimePattern(self, "WebsocketPattern", runtime_arn=acr.runtime_arn, auth_api_key="prototype")

# [Pattern 3] Lambda URL for streaming response
lambda_url = LambdaUrlStreamingAgentCoreRuntimePattern(self, "LambdaUrlPattern", runtime_arn=acr.runtime_arn)
```

## Architecture

```
Input Properties                                                         Outputs
─────────────────                                                        ───────
• agentName                                                              • runtimeId
• agentSrcPath       ┌────────────────────────────────────────────┐     • runtimeVersion
                 ───▶│ SimpleAgentCoreRuntime Construct           │────▶• runtimeArn
                     │                                             │     • runtimeExecutionRole
                     │  ┌──────────────────────────────────────┐  │
                     │  │ IAM Role                             │  │
                     │  │ (AgentCoreRuntimeExecutionRole)      │  │
                     │  │  • ECR access                        │  │
                     │  │  • CloudWatch Logs                   │  │
                     │  │  • Bedrock model invocation          │  │
                     │  └──────────────────┬───────────────────┘  │
                     │                     │                       │
                     │  ┌──────────────────▼───────────────────┐  │
Docker Image ────────┼─▶│ ECR Repository                       │  │
(from agentSrcPath)  │  │  • Stores container image            │  │
                     │  │  • Tag: latest                       │  │
                     │  └──────────────────┬───────────────────┘  │
                     │                     │                       │
                     │  ┌──────────────────▼───────────────────┐  │
                     │  │ Bedrock AgentCore Runtime            │  │
                     │  │  • Runs your agent container         │  │
                     │  │  • Network: PUBLIC (default)         │  │
                     │  │  • Environment variables             │  │
                     │  └──────────────────────────────────────┘  │
                     │                                             │
                     └────────────────────────────────────────────┘
                                      │
                                      │ checks & creates if needed
                                      ▼
                     ┌────────────────────────────────────────────┐
                     │ Service-Linked Roles (Outside Construct)   │
                     │  • Network SLR                             │
                     │  • Runtime Identity SLR                    │
                     └────────────────────────────────────────────┘
```

## Documentation

* [API Documentation](./API.md) - Complete API reference
* [AGENTS.md](./AGENTS.md) - Guide for AI coding assistants

## License

MIT-0
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from ._jsii import *

import aws_cdk.aws_apigatewayv2 as _aws_cdk_aws_apigatewayv2_ceddda9d
import aws_cdk.aws_bedrockagentcore as _aws_cdk_aws_bedrockagentcore_ceddda9d
import aws_cdk.aws_iam as _aws_cdk_aws_iam_ceddda9d
import aws_cdk.aws_lambda as _aws_cdk_aws_lambda_ceddda9d
import constructs as _constructs_77d1e7e8


class HttpApiAgentCoreRuntimePattern(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="simple-agentcore-runtime-patterns.HttpApiAgentCoreRuntimePattern",
):
    '''A construct that creates an HTTP API Gateway to invoke AgentCore runtimes.

    This construct handles:

    - Creating HTTP API Gateway with CORS configuration
    - Setting up Lambda functions to invoke AgentCore runtimes
    - Optional API key or custom authentication
    - Multiple runtime endpoints

    Example::

        new HttpApiAgentCoreRuntimePattern(this, 'HttpApi', {
          runtimes: [
            { runtimeArn: runtime.runtimeArn, routePath: '/agent' }
          ],
          authApiKey: 'my-secret-key',
        });
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        runtimes: typing.Sequence[typing.Union["RuntimeInvokeConfig", typing.Dict[builtins.str, typing.Any]]],
        auth_api_key: typing.Optional[builtins.str] = None,
        authorizer: typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IHttpRouteAuthorizer] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param runtimes: Array of runtime configurations to expose via HTTP API.
        :param auth_api_key: API key for simple authentication. Creates a Lambda authorizer. Cannot be used with ``authorizer``. Default: - No authentication
        :param authorizer: Custom HTTP route authorizer. Cannot be used with ``authApiKey``. Default: - No authentication
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce1e1475c1fb357decfc51968d5193d7ab6742691435310aa5fedf4912f62ea4)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = HttpApiAgentCoreRuntimePatternProps(
            runtimes=runtimes, auth_api_key=auth_api_key, authorizer=authorizer
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addRuntimeInvoke")
    def add_runtime_invoke(
        self,
        id: builtins.str,
        *,
        route_path: builtins.str,
        runtime_arn: builtins.str,
    ) -> None:
        '''Adds a new runtime invocation endpoint to the HTTP API.

        :param id: - Unique identifier for CDK resource naming.
        :param route_path: HTTP route path for this runtime.
        :param runtime_arn: ARN of the AgentCore runtime to invoke.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ac58a940f716c59867c1ae0600d788ec135549b7bc9300da1ef8d33a2bece3a1)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        config = RuntimeInvokeConfig(route_path=route_path, runtime_arn=runtime_arn)

        return typing.cast(None, jsii.invoke(self, "addRuntimeInvoke", [id, config]))

    @builtins.property
    @jsii.member(jsii_name="apiUrl")
    def api_url(self) -> builtins.str:
        '''The URL of the HTTP API.'''
        return typing.cast(builtins.str, jsii.get(self, "apiUrl"))

    @builtins.property
    @jsii.member(jsii_name="httpApi")
    def http_api(self) -> _aws_cdk_aws_apigatewayv2_ceddda9d.HttpApi:
        '''The HTTP API Gateway instance.'''
        return typing.cast(_aws_cdk_aws_apigatewayv2_ceddda9d.HttpApi, jsii.get(self, "httpApi"))


@jsii.data_type(
    jsii_type="simple-agentcore-runtime-patterns.HttpApiAgentCoreRuntimePatternProps",
    jsii_struct_bases=[],
    name_mapping={
        "runtimes": "runtimes",
        "auth_api_key": "authApiKey",
        "authorizer": "authorizer",
    },
)
class HttpApiAgentCoreRuntimePatternProps:
    def __init__(
        self,
        *,
        runtimes: typing.Sequence[typing.Union["RuntimeInvokeConfig", typing.Dict[builtins.str, typing.Any]]],
        auth_api_key: typing.Optional[builtins.str] = None,
        authorizer: typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IHttpRouteAuthorizer] = None,
    ) -> None:
        '''Properties for HttpApiAgentCoreRuntimePattern construct.

        :param runtimes: Array of runtime configurations to expose via HTTP API.
        :param auth_api_key: API key for simple authentication. Creates a Lambda authorizer. Cannot be used with ``authorizer``. Default: - No authentication
        :param authorizer: Custom HTTP route authorizer. Cannot be used with ``authApiKey``. Default: - No authentication
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bde83ddc1d4cc36d5a68ccb6d9a775d23e2cbae4ae7a6a2c914cb296422a09d8)
            check_type(argname="argument runtimes", value=runtimes, expected_type=type_hints["runtimes"])
            check_type(argname="argument auth_api_key", value=auth_api_key, expected_type=type_hints["auth_api_key"])
            check_type(argname="argument authorizer", value=authorizer, expected_type=type_hints["authorizer"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "runtimes": runtimes,
        }
        if auth_api_key is not None:
            self._values["auth_api_key"] = auth_api_key
        if authorizer is not None:
            self._values["authorizer"] = authorizer

    @builtins.property
    def runtimes(self) -> typing.List["RuntimeInvokeConfig"]:
        '''Array of runtime configurations to expose via HTTP API.'''
        result = self._values.get("runtimes")
        assert result is not None, "Required property 'runtimes' is missing"
        return typing.cast(typing.List["RuntimeInvokeConfig"], result)

    @builtins.property
    def auth_api_key(self) -> typing.Optional[builtins.str]:
        '''API key for simple authentication.

        Creates a Lambda authorizer.
        Cannot be used with ``authorizer``.

        :default: - No authentication
        '''
        result = self._values.get("auth_api_key")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer(
        self,
    ) -> typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IHttpRouteAuthorizer]:
        '''Custom HTTP route authorizer.

        Cannot be used with ``authApiKey``.

        :default: - No authentication
        '''
        result = self._values.get("authorizer")
        return typing.cast(typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IHttpRouteAuthorizer], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpApiAgentCoreRuntimePatternProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class IamRoleFactory(
    metaclass=jsii.JSIIMeta,
    jsii_type="simple-agentcore-runtime-patterns.IamRoleFactory",
):
    '''Factory class for creating IAM roles and service-linked roles required by Bedrock AgentCore.'''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="createRuntimeExecutionRole")
    @builtins.classmethod
    def create_runtime_execution_role(
        cls,
        ctx: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        agent_name: builtins.str,
    ) -> _aws_cdk_aws_iam_ceddda9d.Role:
        '''Creates an IAM role for Bedrock AgentCore runtime execution.

        The role includes permissions for:

        - ECR image access
        - CloudWatch Logs
        - X-Ray tracing
        - CloudWatch metrics
        - Bedrock model invocation
        - AgentCore workload identity

        :param ctx: - The parent construct.
        :param id: - The construct ID.
        :param agent_name: - The name of the agent runtime.

        :return: The created IAM role
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d97fea57a1f7e730231edd98d82cf19af380408be0f75f1b820f10fc7d4be8e)
            check_type(argname="argument ctx", value=ctx, expected_type=type_hints["ctx"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument agent_name", value=agent_name, expected_type=type_hints["agent_name"])
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Role, jsii.sinvoke(cls, "createRuntimeExecutionRole", [ctx, id, agent_name]))

    @jsii.member(jsii_name="createServiceLinkedRoles")
    @builtins.classmethod
    def create_service_linked_roles(
        cls,
        ctx: _constructs_77d1e7e8.Construct,
        id: builtins.str,
    ) -> None:
        '''Creates service-linked roles required by Bedrock AgentCore.

        This method creates two service-linked roles:

        - Network SLR for network configuration
        - Runtime Identity SLR for workload identity management

        If the roles already exist, the errors are ignored.

        :param ctx: - The parent construct.
        :param id: - The construct ID.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b468f9087dd9ea96202992660a2d86378e391e3c4d6b16850c481b9a2e83d51c)
            check_type(argname="argument ctx", value=ctx, expected_type=type_hints["ctx"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        return typing.cast(None, jsii.sinvoke(cls, "createServiceLinkedRoles", [ctx, id]))


class LambdaUrlStreamingAgentCoreRuntimePattern(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="simple-agentcore-runtime-patterns.LambdaUrlStreamingAgentCoreRuntimePattern",
):
    '''A construct that creates a Lambda function with streaming URL to invoke AgentCore runtimes.

    This construct handles:

    - Creating Lambda function with response streaming support
    - Setting up Lambda function URL with CORS configuration
    - IAM role with AgentCore invocation permissions
    - Streaming response support via Lambda URL

    Example::

        new LambdaUrlStreamingAgentCoreRuntimePattern(this, 'LambdaUrl', {
          runtimeArn: runtime.runtimeArn,
          authType: lambda.FunctionUrlAuthType.NONE,
        });
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        runtime_arn: builtins.str,
        auth_type: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FunctionUrlAuthType] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param runtime_arn: ARN of the AgentCore runtime to invoke.
        :param auth_type: Authentication type for the Lambda function URL. Default: AWS_IAM
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b5a0e406279589c8ec9edc902805879f307e4ff5b2533c80b0d917ea55336c0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = LambdaUrlStreamingAgentCoreRuntimePatternProps(
            runtime_arn=runtime_arn, auth_type=auth_type
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="function")
    def function(self) -> _aws_cdk_aws_lambda_ceddda9d.Function:
        '''The Lambda function instance.'''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.Function, jsii.get(self, "function"))

    @builtins.property
    @jsii.member(jsii_name="functionUrl")
    def function_url(self) -> _aws_cdk_aws_lambda_ceddda9d.FunctionUrl:
        '''The Lambda function URL.'''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.FunctionUrl, jsii.get(self, "functionUrl"))

    @builtins.property
    @jsii.member(jsii_name="url")
    def url(self) -> builtins.str:
        '''The URL string of the Lambda function.'''
        return typing.cast(builtins.str, jsii.get(self, "url"))


@jsii.data_type(
    jsii_type="simple-agentcore-runtime-patterns.LambdaUrlStreamingAgentCoreRuntimePatternProps",
    jsii_struct_bases=[],
    name_mapping={"runtime_arn": "runtimeArn", "auth_type": "authType"},
)
class LambdaUrlStreamingAgentCoreRuntimePatternProps:
    def __init__(
        self,
        *,
        runtime_arn: builtins.str,
        auth_type: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FunctionUrlAuthType] = None,
    ) -> None:
        '''Properties for LambdaUrlStreamingAgentCoreRuntimePattern construct.

        :param runtime_arn: ARN of the AgentCore runtime to invoke.
        :param auth_type: Authentication type for the Lambda function URL. Default: AWS_IAM
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5c0645022dcfc1d7e613e62817813dc6842fe964fe7b3bcb2501693120ae7dd1)
            check_type(argname="argument runtime_arn", value=runtime_arn, expected_type=type_hints["runtime_arn"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "runtime_arn": runtime_arn,
        }
        if auth_type is not None:
            self._values["auth_type"] = auth_type

    @builtins.property
    def runtime_arn(self) -> builtins.str:
        '''ARN of the AgentCore runtime to invoke.'''
        result = self._values.get("runtime_arn")
        assert result is not None, "Required property 'runtime_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def auth_type(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FunctionUrlAuthType]:
        '''Authentication type for the Lambda function URL.

        :default: AWS_IAM
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FunctionUrlAuthType], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LambdaUrlStreamingAgentCoreRuntimePatternProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="simple-agentcore-runtime-patterns.RuntimeInvokeConfig",
    jsii_struct_bases=[],
    name_mapping={"route_path": "routePath", "runtime_arn": "runtimeArn"},
)
class RuntimeInvokeConfig:
    def __init__(self, *, route_path: builtins.str, runtime_arn: builtins.str) -> None:
        '''Configuration for a runtime invocation route.

        :param route_path: HTTP route path for this runtime.
        :param runtime_arn: ARN of the AgentCore runtime to invoke.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__662c560129a82b54b24c568102d6e2d685578d07e4078ce7fa08bb3dbb33bea9)
            check_type(argname="argument route_path", value=route_path, expected_type=type_hints["route_path"])
            check_type(argname="argument runtime_arn", value=runtime_arn, expected_type=type_hints["runtime_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "route_path": route_path,
            "runtime_arn": runtime_arn,
        }

    @builtins.property
    def route_path(self) -> builtins.str:
        '''HTTP route path for this runtime.'''
        result = self._values.get("route_path")
        assert result is not None, "Required property 'route_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def runtime_arn(self) -> builtins.str:
        '''ARN of the AgentCore runtime to invoke.'''
        result = self._values.get("runtime_arn")
        assert result is not None, "Required property 'runtime_arn' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RuntimeInvokeConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SimpleAgentCoreRuntime(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="simple-agentcore-runtime-patterns.SimpleAgentCoreRuntime",
):
    '''A construct that creates an AWS Bedrock AgentCore runtime.

    This construct handles:

    - Building and deploying Docker container images to ECR
    - Creating IAM roles and service-linked roles
    - Configuring Bedrock AgentCore runtime with network and environment settings

    Example::

        new SimpleAgentCoreRuntime(this, 'MyAgent', {
          agentName: 'my-bedrock-agent',
          agentSrcPath: './agent-code',
        });
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        agent_name: builtins.str,
        agent_src_path: builtins.str,
        agent_description: typing.Optional[builtins.str] = None,
        ecr_repository_name: typing.Optional[builtins.str] = None,
        environment_variables: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        network_configuration: typing.Optional[typing.Union[_aws_cdk_aws_bedrockagentcore_ceddda9d.CfnRuntime.NetworkConfigurationProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        runtime_execution_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param agent_name: The name of the Bedrock AgentCore runtime. This will be used as the runtime name and default ECR repository name.
        :param agent_src_path: Path to the agent source code directory containing Dockerfile.
        :param agent_description: Description of the agent runtime. Default: - No description
        :param ecr_repository_name: ECR repository name for the agent container image. Default: - Creates a new repository named with agentName
        :param environment_variables: Environment variables to pass to the agent container. Default: - No environment variables
        :param network_configuration: Network configuration for the AgentCore runtime. Default: - PUBLIC network mode: { networkMode: 'PUBLIC' }
        :param runtime_execution_role: IAM role for the AgentCore runtime execution. Default: - Creates a new role with required Bedrock AgentCore permissions
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__481f3e7badf4deb5e6f8d05944da84821bdf4897da5e5834d13c94e0ec8e3a68)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = SimpleAgentCoreRuntimeProps(
            agent_name=agent_name,
            agent_src_path=agent_src_path,
            agent_description=agent_description,
            ecr_repository_name=ecr_repository_name,
            environment_variables=environment_variables,
            network_configuration=network_configuration,
            runtime_execution_role=runtime_execution_role,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="runtimeArn")
    def runtime_arn(self) -> builtins.str:
        '''The ARN of the AgentCore runtime.'''
        return typing.cast(builtins.str, jsii.get(self, "runtimeArn"))

    @builtins.property
    @jsii.member(jsii_name="runtimeExecutionRole")
    def runtime_execution_role(self) -> _aws_cdk_aws_iam_ceddda9d.Role:
        '''The IAM role used by the AgentCore runtime.'''
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Role, jsii.get(self, "runtimeExecutionRole"))

    @builtins.property
    @jsii.member(jsii_name="runtimeId")
    def runtime_id(self) -> builtins.str:
        '''The unique identifier of the AgentCore runtime.'''
        return typing.cast(builtins.str, jsii.get(self, "runtimeId"))

    @builtins.property
    @jsii.member(jsii_name="runtimeVersion")
    def runtime_version(self) -> builtins.str:
        '''The version of the AgentCore runtime.'''
        return typing.cast(builtins.str, jsii.get(self, "runtimeVersion"))


@jsii.data_type(
    jsii_type="simple-agentcore-runtime-patterns.SimpleAgentCoreRuntimeProps",
    jsii_struct_bases=[],
    name_mapping={
        "agent_name": "agentName",
        "agent_src_path": "agentSrcPath",
        "agent_description": "agentDescription",
        "ecr_repository_name": "ecrRepositoryName",
        "environment_variables": "environmentVariables",
        "network_configuration": "networkConfiguration",
        "runtime_execution_role": "runtimeExecutionRole",
    },
)
class SimpleAgentCoreRuntimeProps:
    def __init__(
        self,
        *,
        agent_name: builtins.str,
        agent_src_path: builtins.str,
        agent_description: typing.Optional[builtins.str] = None,
        ecr_repository_name: typing.Optional[builtins.str] = None,
        environment_variables: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        network_configuration: typing.Optional[typing.Union[_aws_cdk_aws_bedrockagentcore_ceddda9d.CfnRuntime.NetworkConfigurationProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        runtime_execution_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role] = None,
    ) -> None:
        '''Properties for SimpleAgentCoreRuntime construct.

        :param agent_name: The name of the Bedrock AgentCore runtime. This will be used as the runtime name and default ECR repository name.
        :param agent_src_path: Path to the agent source code directory containing Dockerfile.
        :param agent_description: Description of the agent runtime. Default: - No description
        :param ecr_repository_name: ECR repository name for the agent container image. Default: - Creates a new repository named with agentName
        :param environment_variables: Environment variables to pass to the agent container. Default: - No environment variables
        :param network_configuration: Network configuration for the AgentCore runtime. Default: - PUBLIC network mode: { networkMode: 'PUBLIC' }
        :param runtime_execution_role: IAM role for the AgentCore runtime execution. Default: - Creates a new role with required Bedrock AgentCore permissions
        '''
        if isinstance(network_configuration, dict):
            network_configuration = _aws_cdk_aws_bedrockagentcore_ceddda9d.CfnRuntime.NetworkConfigurationProperty(**network_configuration)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f94cb6c5267966c60a4bfbf21b4d81b6b1810fb8ca35b8eae96d3a64376453a1)
            check_type(argname="argument agent_name", value=agent_name, expected_type=type_hints["agent_name"])
            check_type(argname="argument agent_src_path", value=agent_src_path, expected_type=type_hints["agent_src_path"])
            check_type(argname="argument agent_description", value=agent_description, expected_type=type_hints["agent_description"])
            check_type(argname="argument ecr_repository_name", value=ecr_repository_name, expected_type=type_hints["ecr_repository_name"])
            check_type(argname="argument environment_variables", value=environment_variables, expected_type=type_hints["environment_variables"])
            check_type(argname="argument network_configuration", value=network_configuration, expected_type=type_hints["network_configuration"])
            check_type(argname="argument runtime_execution_role", value=runtime_execution_role, expected_type=type_hints["runtime_execution_role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "agent_name": agent_name,
            "agent_src_path": agent_src_path,
        }
        if agent_description is not None:
            self._values["agent_description"] = agent_description
        if ecr_repository_name is not None:
            self._values["ecr_repository_name"] = ecr_repository_name
        if environment_variables is not None:
            self._values["environment_variables"] = environment_variables
        if network_configuration is not None:
            self._values["network_configuration"] = network_configuration
        if runtime_execution_role is not None:
            self._values["runtime_execution_role"] = runtime_execution_role

    @builtins.property
    def agent_name(self) -> builtins.str:
        '''The name of the Bedrock AgentCore runtime.

        This will be used as the runtime name and default ECR repository name.
        '''
        result = self._values.get("agent_name")
        assert result is not None, "Required property 'agent_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def agent_src_path(self) -> builtins.str:
        '''Path to the agent source code directory containing Dockerfile.'''
        result = self._values.get("agent_src_path")
        assert result is not None, "Required property 'agent_src_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def agent_description(self) -> typing.Optional[builtins.str]:
        '''Description of the agent runtime.

        :default: - No description
        '''
        result = self._values.get("agent_description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ecr_repository_name(self) -> typing.Optional[builtins.str]:
        '''ECR repository name for the agent container image.

        :default: - Creates a new repository named with agentName
        '''
        result = self._values.get("ecr_repository_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment_variables(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Environment variables to pass to the agent container.

        :default: - No environment variables
        '''
        result = self._values.get("environment_variables")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def network_configuration(
        self,
    ) -> typing.Optional[_aws_cdk_aws_bedrockagentcore_ceddda9d.CfnRuntime.NetworkConfigurationProperty]:
        '''Network configuration for the AgentCore runtime.

        :default: - PUBLIC network mode: { networkMode: 'PUBLIC' }
        '''
        result = self._values.get("network_configuration")
        return typing.cast(typing.Optional[_aws_cdk_aws_bedrockagentcore_ceddda9d.CfnRuntime.NetworkConfigurationProperty], result)

    @builtins.property
    def runtime_execution_role(self) -> typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role]:
        '''IAM role for the AgentCore runtime execution.

        :default: - Creates a new role with required Bedrock AgentCore permissions
        '''
        result = self._values.get("runtime_execution_role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SimpleAgentCoreRuntimeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class WebsocketAgentCoreRuntimePattern(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="simple-agentcore-runtime-patterns.WebsocketAgentCoreRuntimePattern",
):
    '''A construct that creates a WebSocket API Gateway to invoke AgentCore runtimes with streaming support.

    This construct handles:

    - Creating WebSocket API Gateway with connect/disconnect/message routes
    - Setting up Lambda function to invoke AgentCore runtime
    - Optional API key or custom authentication
    - Streaming response support via WebSocket

    Example::

        new WebsocketAgentCoreRuntimePattern(this, 'WebSocket', {
          runtimeArn: runtime.runtimeArn,
          authApiKey: 'my-secret-key',
        });
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        runtime_arn: builtins.str,
        auth_api_key: typing.Optional[builtins.str] = None,
        authorizer: typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IWebSocketRouteAuthorizer] = None,
        route_path: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param runtime_arn: ARN of the AgentCore runtime to invoke.
        :param auth_api_key: API key for simple authentication. Creates a Lambda authorizer. Cannot be used with ``authorizer``. Default: - No authentication
        :param authorizer: Custom WebSocket route authorizer. Cannot be used with ``authApiKey``. Default: - No authentication
        :param route_path: Custom route path for WebSocket messages. Default: 'message'
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ccf22e72567d570de184c9886f82ff126261552b993495dc756a7f903b172e51)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = WebsocketAgentCoreRuntimePatternProps(
            runtime_arn=runtime_arn,
            auth_api_key=auth_api_key,
            authorizer=authorizer,
            route_path=route_path,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="webSocketApi")
    def web_socket_api(self) -> _aws_cdk_aws_apigatewayv2_ceddda9d.WebSocketApi:
        '''The WebSocket API Gateway instance.'''
        return typing.cast(_aws_cdk_aws_apigatewayv2_ceddda9d.WebSocketApi, jsii.get(self, "webSocketApi"))

    @builtins.property
    @jsii.member(jsii_name="webSocketStage")
    def web_socket_stage(self) -> _aws_cdk_aws_apigatewayv2_ceddda9d.WebSocketStage:
        '''The WebSocket API stage.'''
        return typing.cast(_aws_cdk_aws_apigatewayv2_ceddda9d.WebSocketStage, jsii.get(self, "webSocketStage"))

    @builtins.property
    @jsii.member(jsii_name="webSocketUrl")
    def web_socket_url(self) -> builtins.str:
        '''The URL of the WebSocket API.'''
        return typing.cast(builtins.str, jsii.get(self, "webSocketUrl"))


@jsii.data_type(
    jsii_type="simple-agentcore-runtime-patterns.WebsocketAgentCoreRuntimePatternProps",
    jsii_struct_bases=[],
    name_mapping={
        "runtime_arn": "runtimeArn",
        "auth_api_key": "authApiKey",
        "authorizer": "authorizer",
        "route_path": "routePath",
    },
)
class WebsocketAgentCoreRuntimePatternProps:
    def __init__(
        self,
        *,
        runtime_arn: builtins.str,
        auth_api_key: typing.Optional[builtins.str] = None,
        authorizer: typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IWebSocketRouteAuthorizer] = None,
        route_path: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for WebsocketAgentCoreRuntimePattern construct.

        :param runtime_arn: ARN of the AgentCore runtime to invoke.
        :param auth_api_key: API key for simple authentication. Creates a Lambda authorizer. Cannot be used with ``authorizer``. Default: - No authentication
        :param authorizer: Custom WebSocket route authorizer. Cannot be used with ``authApiKey``. Default: - No authentication
        :param route_path: Custom route path for WebSocket messages. Default: 'message'
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__13d6b59f328530f1e8a3014c4081cf1912b751da78685a7460b3dd794cbaa116)
            check_type(argname="argument runtime_arn", value=runtime_arn, expected_type=type_hints["runtime_arn"])
            check_type(argname="argument auth_api_key", value=auth_api_key, expected_type=type_hints["auth_api_key"])
            check_type(argname="argument authorizer", value=authorizer, expected_type=type_hints["authorizer"])
            check_type(argname="argument route_path", value=route_path, expected_type=type_hints["route_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "runtime_arn": runtime_arn,
        }
        if auth_api_key is not None:
            self._values["auth_api_key"] = auth_api_key
        if authorizer is not None:
            self._values["authorizer"] = authorizer
        if route_path is not None:
            self._values["route_path"] = route_path

    @builtins.property
    def runtime_arn(self) -> builtins.str:
        '''ARN of the AgentCore runtime to invoke.'''
        result = self._values.get("runtime_arn")
        assert result is not None, "Required property 'runtime_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def auth_api_key(self) -> typing.Optional[builtins.str]:
        '''API key for simple authentication.

        Creates a Lambda authorizer.
        Cannot be used with ``authorizer``.

        :default: - No authentication
        '''
        result = self._values.get("auth_api_key")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer(
        self,
    ) -> typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IWebSocketRouteAuthorizer]:
        '''Custom WebSocket route authorizer.

        Cannot be used with ``authApiKey``.

        :default: - No authentication
        '''
        result = self._values.get("authorizer")
        return typing.cast(typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IWebSocketRouteAuthorizer], result)

    @builtins.property
    def route_path(self) -> typing.Optional[builtins.str]:
        '''Custom route path for WebSocket messages.

        :default: 'message'
        '''
        result = self._values.get("route_path")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "WebsocketAgentCoreRuntimePatternProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


__all__ = [
    "HttpApiAgentCoreRuntimePattern",
    "HttpApiAgentCoreRuntimePatternProps",
    "IamRoleFactory",
    "LambdaUrlStreamingAgentCoreRuntimePattern",
    "LambdaUrlStreamingAgentCoreRuntimePatternProps",
    "RuntimeInvokeConfig",
    "SimpleAgentCoreRuntime",
    "SimpleAgentCoreRuntimeProps",
    "WebsocketAgentCoreRuntimePattern",
    "WebsocketAgentCoreRuntimePatternProps",
]

publication.publish()

def _typecheckingstub__ce1e1475c1fb357decfc51968d5193d7ab6742691435310aa5fedf4912f62ea4(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    runtimes: typing.Sequence[typing.Union[RuntimeInvokeConfig, typing.Dict[builtins.str, typing.Any]]],
    auth_api_key: typing.Optional[builtins.str] = None,
    authorizer: typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IHttpRouteAuthorizer] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__ac58a940f716c59867c1ae0600d788ec135549b7bc9300da1ef8d33a2bece3a1(
    id: builtins.str,
    *,
    route_path: builtins.str,
    runtime_arn: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__bde83ddc1d4cc36d5a68ccb6d9a775d23e2cbae4ae7a6a2c914cb296422a09d8(
    *,
    runtimes: typing.Sequence[typing.Union[RuntimeInvokeConfig, typing.Dict[builtins.str, typing.Any]]],
    auth_api_key: typing.Optional[builtins.str] = None,
    authorizer: typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IHttpRouteAuthorizer] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__4d97fea57a1f7e730231edd98d82cf19af380408be0f75f1b820f10fc7d4be8e(
    ctx: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    agent_name: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__b468f9087dd9ea96202992660a2d86378e391e3c4d6b16850c481b9a2e83d51c(
    ctx: _constructs_77d1e7e8.Construct,
    id: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__3b5a0e406279589c8ec9edc902805879f307e4ff5b2533c80b0d917ea55336c0(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    runtime_arn: builtins.str,
    auth_type: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FunctionUrlAuthType] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__5c0645022dcfc1d7e613e62817813dc6842fe964fe7b3bcb2501693120ae7dd1(
    *,
    runtime_arn: builtins.str,
    auth_type: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FunctionUrlAuthType] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__662c560129a82b54b24c568102d6e2d685578d07e4078ce7fa08bb3dbb33bea9(
    *,
    route_path: builtins.str,
    runtime_arn: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__481f3e7badf4deb5e6f8d05944da84821bdf4897da5e5834d13c94e0ec8e3a68(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    agent_name: builtins.str,
    agent_src_path: builtins.str,
    agent_description: typing.Optional[builtins.str] = None,
    ecr_repository_name: typing.Optional[builtins.str] = None,
    environment_variables: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    network_configuration: typing.Optional[typing.Union[_aws_cdk_aws_bedrockagentcore_ceddda9d.CfnRuntime.NetworkConfigurationProperty, typing.Dict[builtins.str, typing.Any]]] = None,
    runtime_execution_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__f94cb6c5267966c60a4bfbf21b4d81b6b1810fb8ca35b8eae96d3a64376453a1(
    *,
    agent_name: builtins.str,
    agent_src_path: builtins.str,
    agent_description: typing.Optional[builtins.str] = None,
    ecr_repository_name: typing.Optional[builtins.str] = None,
    environment_variables: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    network_configuration: typing.Optional[typing.Union[_aws_cdk_aws_bedrockagentcore_ceddda9d.CfnRuntime.NetworkConfigurationProperty, typing.Dict[builtins.str, typing.Any]]] = None,
    runtime_execution_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__ccf22e72567d570de184c9886f82ff126261552b993495dc756a7f903b172e51(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    runtime_arn: builtins.str,
    auth_api_key: typing.Optional[builtins.str] = None,
    authorizer: typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IWebSocketRouteAuthorizer] = None,
    route_path: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__13d6b59f328530f1e8a3014c4081cf1912b751da78685a7460b3dd794cbaa116(
    *,
    runtime_arn: builtins.str,
    auth_api_key: typing.Optional[builtins.str] = None,
    authorizer: typing.Optional[_aws_cdk_aws_apigatewayv2_ceddda9d.IWebSocketRouteAuthorizer] = None,
    route_path: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass
