from fuzzywuzzy import fuzz
from base import devops_client_exception
from base import devops_base
from projects import devops_project

# _attribute_map = {
#        '_links': {'key': '_links', 'type': 'ReferenceLinks'},
#        'default_branch': {'key': 'defaultBranch', 'type': 'str'},
#        'id': {'key': 'id', 'type': 'str'},
#        'is_fork': {'key': 'isFork', 'type': 'bool'},
#        'name': {'key': 'name', 'type': 'str'},
#        'parent_repository': {'key': 'parentRepository', 'type': 'GitRepositoryRef'},
#        'project': {'key': 'project', 'type': 'TeamProjectReference'},
#        'remote_url': {'key': 'remoteUrl', 'type': 'str'},
#        'size': {'key': 'size', 'type': 'long'},
#        'ssh_url': {'key': 'sshUrl', 'type': 'str'},
#        'url': {'key': 'url', 'type': 'str'},
#        'valid_remote_urls': {'key': 'validRemoteUrls', 'type': '[str]'}
#    }


class devops_repos(devops_base):
    """ spin up connection from base and create client, and get list of repos"""
    def __init__(self, project):
        # init base class (brings up connections etc)
        super().__init__()
        if not isinstance(project, devops_project):
            raise devops_client_exception("Project Parameter should be an instance of devops_project")
        self._project = project
        try:
            self._git_client = self._connection.get_client("azure.devops.v5_1.git.git_client.GitClient")
        except Exception as could_not_get_client:
            print(f"Could Not Get Devops Client {str(could_not_get_client)}")

        try:
            self._repos = self._git_client.get_repositories(self._project.id)
        except Exception as could_not_get_repo_list:
            print(f"Could Not Get Project List {str(could_not_get_repo_list)}")

    def get_repo_list(self):
        """ return list of repos as devops_repo objects """
        repolist = []
        for ro in self._repos:
            repolist.append(devops_repo(self._project, ro.__dict__))
        return repolist

    def get_repo_from_id(self, repo_id):
        """return repo object from id"""
        for repo in self._repos:
            if repo.__dict__['id'] == repo_id:
                return devops_repo(self._project, repo.__dict__)
        raise devops_client_exception(f"Couldn't find Repo {repo_id}")

    def get_repo_from_name(self, repo_name):
        """return repo object from name"""
        for repo in self._repos:
            if repo.__dict__['name'] == repo_name:
                return devops_repo(self._project, repo.__dict__)
        raise devops_client_exception(f"Couldn't find Repo {repo_name}")


class devops_repo(devops_base):
    def __init__(self, project, repo_object):
        """ represents the object to interact with an individual devops repo"""
        super().__init__()
        self._repo = repo_object
        self._id = self._repo['id']
        self._name = self._repo['name']

        if not isinstance(project, devops_project):
            raise devops_client_exception("Project Parameter should be an instance of devops_project")

        self._project = project

    def connect_to_devops(self):
        """ create a devops git client object """
        try:
            self._git_client = self._connection.get_client("azure.devops.v5_1.git.git_client.GitClient")

        except Exception as could_not_get_client:
            print(f"Could Not Get Devops Client {str(could_not_get_client)}")

    def get_refs(self):
        """ get all refs(branches) from the project as devops_ref objects"""
        out_refs = []
        self.__connect_to_devops()
        try:
            refs = self._git_client.get_refs(self._id, project=str(self._project.id))
            for ref in refs:
                out_refs.append(devops_ref(self._project, self._repo, ref.__dict__))
        except Exception as could_not_get_refs:
            print(f"Could Not Get Refs {str(could_not_get_refs)}")
        return out_refs

    def get_ref_by_name(self, name):
        """ get ref(branch) object by it's name (using 90% fuzzy matching)"""
        refs = self.__get_refs()
        for ref in refs:
            ratio = fuzz.partial_ratio(ref.__dict__['name'], name)
            if ratio > 90:
                return devops_ref(self._project, self._repo, ref.__dict__)
        raise devops_client_exception(f"Could Not Find Ref Name {name} using fuzzy matching")

    def __str__(self):
        return f"Devops Repository: {self._repo['name']}"
    __repr__ = __str__
    __connect_to_devops = connect_to_devops
    __get_refs = get_refs


class devops_ref():
    def __init__(self, project, repo, ref):
        """ python object to abstract the autogenerated one, represents the info for a branch(ref)"""
        print(ref)
        if not isinstance(project, devops_project):
            raise devops_client_exception("Project Parameter should be an instance of devops_project")
        self._project = project
        self._repo = repo
        self._ref = ref
        self.name = ref['name']
        self.repo_name = self._repo['name']
        self.repo_id = self._repo['id']

    def __str__(self):
        return str(f"Devops Branch {self.name} in Repo {self.repo_name}")

    __repr__ = __str__
