
import base64
import hmac
import hashlib
import logging
import time
import urllib


log = logging.getLogger(__name__)


class SPFileHandlerException(Exception):
    pass


class SPFileHandler(object):

    def __init__(self, secret, access_key):
        self.secret = secret
        self.access_key = access_key

    def generate_signed_url(self, resource_url, expires=60, method='GET'):
        """
        Generates a signed, expiring url to the hosted resource specified.
        :param resource: file descriptor / handler
        :param expires: number of seconds before the url should expire
        :param method: GET or POST
        :return: signed, expiring url to the hosted resource
        :raises: SPFileHandlerException
        """

        expiry = int(time.time()) + expires

        c_string = self.build_canonical_string(method, expiry, resource_url)

        secret = str(self.secret)
        signature = base64.b64encode(hmac.new(
            secret,
            msg=c_string,
            digestmod=hashlib.sha1).digest()
        )

        q_dict = {'AccessKey': self.access_key, 'Signature': signature, 'Expires': expiry}
        q_string = urllib.urlencode(q_dict)
        signed_url = '{}?{}'.format(resource_url, q_string)

        return signed_url

    def is_valid_signature(self, signature):

        log.info('Validating signature: {}'.format(signature))

        to_hash = '{}{}'.format(self.access_key, self.secret)
        digest = hashlib.sha1(to_hash).digest()

        calculated_signature = base64.urlsafe_b64encode(digest)

        if calculated_signature == signature:
            return True

        return False

    def build_canonical_string(self, method, expires, auth_path):
        """
        Generates a canonical url similar to: http://s3.amazonaws.com/doc/s3-developer-guide/RESTAuthentication.html
        :param method:
        :param expires:
        :param auth_path:
        :return:
        """

        c_string = '{}\n\n\n{}\n{}'.format(method, expires, auth_path)
        return c_string

    def check_signed_url(self, url, method, auth_path, expires):
        """
        Compares the given url to the signed url generated by the given parameters (method, auth_path, expires)
        :param url:
        :param method:
        :param auth_path:
        :param expires:
        :return:
        """

        c_string = self.build_canonical_string(method, expires, auth_path)

        secret = str(self.secret)
        signature = base64.b64encode(hmac.new(
            secret,
            msg=c_string,
            digestmod=hashlib.sha1).digest()
        )

        q_dict = {'AccessKey': self.access_key, 'Signature': signature, 'Expires': expires}
        q_string = urllib.urlencode(q_dict)
        created_url = '{}?{}'.format(auth_path, q_string)

        return url == created_url










