# pylint: disable=C0111,R0902,R0913
# Smartsheet Python SDK.
#
# Copyright 2017 Smartsheet.com, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"): you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import logging
import warnings
from datetime import datetime
from typing import Optional, Union

from .util import fresh_operation
from .models import Error, IndexResult, Result, Share, Sight, SightPublish


class Sights:

    """Class for handling Sights operations."""

    def __init__(self, smartsheet_obj):
        """Init Sights with base Smartsheet object."""
        self._base = smartsheet_obj
        self._log = logging.getLogger(__name__)

    def list_sights(
        self,
        page_size: Optional[int] = None,
        page: Optional[int] = None,
        include_all: Optional[bool] = None,
        last_key: Optional[str] = None,
        max_items: Optional[int] = None,
        pagination_type: Optional[str] = None,
        modified_since: Optional[datetime] = None
    ) -> Union[IndexResult[Sight], Error]:
        """Get the list of all Sights the User has access to, in alphabetical
        order, by name.

        Args:
            page_size (int, optional): [DEPRECATED] The maximum number of items to
                return per page. Use pagination_type='token' with max_items instead.
            page (int, optional): [DEPRECATED] Which page to return.
                Use pagination_type='token' with last_key instead.
            include_all (bool, optional): [DEPRECATED] If true, include all results
                (i.e. do not paginate). Use pagination_type='token' instead.
            last_key (str, optional): Pagination cursor for next page (token pagination only).
            max_items (int, optional): Maximum items per page (token pagination only).
                Must be a positive integer.
            pagination_type (str, optional): Use 'token' for efficient cursor-based pagination.
                Defaults to legacy offset-based pagination if not specified.
            modified_since (datetime, optional): Return sights modified since datetime.

        Returns:
            Union[IndexResult[Sight], Error]: The result of the operation, or an Error object if the request fails.
                When using legacy pagination, contains paginated results with
                total_count, total_pages, etc.

        Raises:
            ValueError: If pagination_type is not 'token' or None, or if max_items <= 0
                when using token pagination.
        """
        # Parameter validation
        if pagination_type is not None and pagination_type not in ['token']:
            raise ValueError("pagination_type must be 'token' or None")
        if pagination_type == 'token' and max_items is not None and max_items <= 0:
            raise ValueError("max_items must be a positive integer")

        _op = fresh_operation("list_sights")
        _op["method"] = "GET"
        _op["path"] = "/sights"

        # Issue deprecation warnings for old parameters when used
        if page_size is not None:
            warnings.warn(
                "page_size parameter is deprecated. Use pagination_type='token' with max_items instead.",
                DeprecationWarning,
                stacklevel=2
            )
        if page is not None:
            warnings.warn(
                "page parameter is deprecated. Use pagination_type='token' with last_key instead.",
                DeprecationWarning,
                stacklevel=2
            )
        if include_all is not None:
            warnings.warn(
                "include_all parameter is deprecated. Use pagination_type='token' instead.",
                DeprecationWarning,
                stacklevel=2
            )

        if pagination_type == "token":
            _op["query_params"]["lastKey"] = last_key
            _op["query_params"]["maxItems"] = max_items
            _op["query_params"]["paginationType"] = pagination_type
        else:
            _op["query_params"]["pageSize"] = page_size
            _op["query_params"]["page"] = page
            _op["query_params"]["includeAll"] = include_all

        if isinstance(modified_since, datetime):
            _op["query_params"]["modifiedSince"] = modified_since.isoformat()

        expected = ["IndexResult", "Sight"]

        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def get_sight(self, sight_id, level=None, include=None) -> Union[Sight, Error]:
        """Get the specified Sight.

        Args:
            sight_id (int): Sight ID
            level (int): compatibility level
            include (list[str]): optional include parameters

        Returns:
            Union[Sight, Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("get_sight")
        _op["method"] = "GET"
        _op["path"] = "/sights/" + str(sight_id)
        _op["query_params"]["include"] = include
        _op["query_params"]["level"] = level

        expected = "Sight"
        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def update_sight(self, sight_id, sight_obj) -> Union[Result[Sight], Error]:
        """Updates the specified Sight.

        Args:
            sight_id (int): Sight ID
            sight_obj (Sight): Sight object.

        Returns:
            Union[Result[Sight], Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("update_sight")
        _op["method"] = "PUT"
        _op["path"] = "/sights/" + str(sight_id)
        _op["json"] = sight_obj

        expected = ["Result", "Sight"]

        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def delete_sight(self, sight_id) -> Union[Result[None], Error]:
        """Delete the specified Sight.

        Args:
            sight_id (int): Sight ID

        Returns:
            Union[Result[None], Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("delete_sight")
        _op["method"] = "DELETE"
        _op["path"] = "/sights/" + str(sight_id)

        expected = ["Result", None]
        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def copy_sight(self, sight_id, container_destination_obj) -> Union[Result[Sight], Error]:
        """Creates a copy of the specified Sight

        Args:
            sight_id (int): Sight ID
            container_destination_obj
                (ContainerDestination): Container Destination object.

        Returns:
            Union[Result[Sight], Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("copy_sight")
        _op["method"] = "POST"
        _op["path"] = "/sights/" + str(sight_id) + "/copy"
        _op["json"] = container_destination_obj

        expected = ["Result", "Sight"]

        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def move_sight(self, sight_id, container_destination_obj) -> Union[Result[Sight], Error]:
        """Creates a copy of the specified Sight

        Args:
            sight_id (int): Sight ID
            container_destination_obj
                (ContainerDestination): Container Destination object.

        Returns:
            Union[Result[Sight], Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("move_sight")
        _op["method"] = "POST"
        _op["path"] = "/sights/" + str(sight_id) + "/move"
        _op["json"] = container_destination_obj

        expected = ["Result", "Sight"]

        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def list_shares(
        self,
        sight_id,
        page_size=None,
        page=None,
        include_all=None,
        include_workspace_shares=False,
    ) -> Union[IndexResult[Share], Error]:
        """Get the list of all Users and Groups to whom the specified Sight is
        shared, and their access level.

        Args:
            sight_id (int): Sight ID
            page_size (int): The maximum number of items to
                return per page.
            page (int): Which page to return.
            include_all (bool): If true, include all results
                (i.e. do not paginate).
            include_workspace_shares(bool): Include Workspace shares

        Returns:
            Union[IndexResult[Share], Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("list_shares")
        _op["method"] = "GET"
        _op["path"] = "/sights/" + str(sight_id) + "/shares"
        _op["query_params"]["pageSize"] = page_size
        _op["query_params"]["page"] = page
        _op["query_params"]["includeAll"] = include_all
        if include_workspace_shares:
            _op["query_params"]["include"] = "workspaceShares"

        expected = ["IndexResult", "Share"]

        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def get_share(self, sight_id, share_id) -> Union[Share, Error]:
        """Get the specified Share.

        Args:
            sight_id (int): Sight ID
            share_id (str): Share ID

        Returns:
            Union[Share, Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("get_share")
        _op["method"] = "GET"
        _op["path"] = "/sights/" + str(sight_id) + "/shares/" + str(share_id)

        expected = "Share"
        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def share_sight(self, sight_id, share_obj, send_email=False) -> Union[Result[Share], Error]:
        """Share the specified Sight.

        Share the specified Sight with the specified Users and
        Groups.

        Args:
            sight_id (int): Sight ID
            share_obj (Share): Share object.
            send_email (bool): Either true or false to
                indicate whether or not to notify the user by email. Default
                is false.

        Returns:
            Union[Result[Share], Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("share_sight")
        _op["method"] = "POST"
        _op["path"] = "/sights/" + str(sight_id) + "/shares"
        _op["query_params"]["sendEmail"] = send_email
        _op["json"] = share_obj

        expected = ["Result", "Share"]

        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def update_share(self, sight_id, share_id, share_obj) -> Union[Result[Share], Error]:
        """Update the access level of a User or Group for the specified Sight.

        Args:
            sight_id (int): Sight ID
            share_id (str): Share ID
            share_obj (Share): Share object.

        Returns:
            Union[Result[Share], Error]: The result of the operation, or an Error object if the request fails.
        """
        if not all(val is not None for val in ["sight_id", "share_id", "share_obj"]):
            raise ValueError(
                ("One or more required values are missing from call to " + __name__)
            )

        _op = fresh_operation("update_share")
        _op["method"] = "PUT"
        _op["path"] = "/sights/" + str(sight_id) + "/shares/" + str(share_id)
        _op["json"] = share_obj

        expected = ["Result", "Share"]

        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def delete_share(self, sight_id, share_id) -> Union[Result[None], Error]:
        """Delete the specified Share.

        Args:
            sight_id (int): Sight ID
            share_id (str): Share ID

        Returns:
            Union[Result[None], Error]: The result of the operation, or an Error object if the request fails.
        """
        _op = fresh_operation("delete_share")
        _op["method"] = "DELETE"
        _op["path"] = "/sights/" + str(sight_id) + "/shares/" + str(share_id)

        expected = ["Result", None]
        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def get_publish_status(self, sight_id) -> Union[SightPublish, Error]:
        """Get the Publish status of the Sight.

        Get the status of the Publish settings of the Sight,
        including URLs of any enabled publishings.

        Args:
            sight_id (int): Sight ID

        Returns:
            Union[Sight, Error]: The result of the operation, or an Error object if the request fails.Publish
        """
        _op = fresh_operation("get_publish_status")
        _op["method"] = "GET"
        _op["path"] = "/sights/" + str(sight_id) + "/publish"

        expected = "SightPublish"
        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response

    def set_publish_status(self, sight_id, sight_publish_obj) -> Union[Result[SightPublish], Error]:
        """Set the publish status of the Sight and returns the new status,
        including the URLs of any enabled publishings.

        Args:
            sight_id (int): Sight ID
            sight_publish_obj (SightPublish): SightPublish object.

        Returns:
            Union[Result[SightPublish], Error]: The result of the operation, or an Error object if the request fails.
        """
        attributes = ["read_only_full_enabled", "read_only_full_accessible_by"]

        fetch_first = False
        # check for incompleteness, fill in from current status if necessary
        for attribute in attributes:
            val = getattr(sight_publish_obj, attribute, None)
            if val is None:
                fetch_first = True
                break

        if fetch_first:
            current_status = self.get_publish_status(sight_id).to_dict()
            current_status.update(sight_publish_obj.to_dict())
            sight_publish_obj = self._base.models.SightPublish(current_status)

        _op = fresh_operation("set_publish_status")
        _op["method"] = "PUT"
        _op["path"] = "/sights/" + str(sight_id) + "/publish"
        _op["json"] = sight_publish_obj

        expected = ["Result", "SightPublish"]

        prepped_request = self._base.prepare_request(_op)
        response = self._base.request(prepped_request, expected, _op)

        return response
