"""
SoIdea-update-python 备份与回滚模块
支持自动备份、历史管理、git commit/tag、回滚恢复
"""

import os
import shutil
import time

from .git_utils import ensure_git_repo, git_commit_all, git_tag


def backup_dir(src_dir, backup_root, logger=None, git_enable=True, tag=None):
    """
    备份指定目录到 backup_root，自动管理历史，支持 git commit/tag。
    :param src_dir: 源目录
    :param backup_root: 备份根目录
    :param logger: 日志对象
    :param git_enable: 是否启用 git 备份
    :param tag: 可选 git tag
    :return: 备份目录路径
    """
    timestamp = time.strftime('%Y%m%d_%H%M%S')
    backup_path = os.path.join(backup_root, f'backup_{timestamp}')
    if os.path.exists(backup_path):
        shutil.rmtree(backup_path)
    shutil.copytree(src_dir, backup_path)
    if logger:
        logger.info(f'已备份目录: {src_dir} -> {backup_path}')
    # 自动 git commit/tag
    if git_enable:
        ensure_git_repo(src_dir, logger)
        msg = f'自动备份 {timestamp}'
        git_commit_all(src_dir, msg, logger)
        if tag:
            git_tag(src_dir, tag, logger)
    # 自动清理历史备份（只保留最近5个）
    _cleanup_history(backup_root, keep=5, logger=logger)
    return backup_path

def _cleanup_history(backup_root, keep=5, logger=None):
    """
    清理历史备份，只保留最近 keep 个。
    """
    backups = [d for d in os.listdir(backup_root) if d.startswith('backup_')]
    backups.sort(reverse=True)
    for old in backups[keep:]:
        path = os.path.join(backup_root, old)
        shutil.rmtree(path)
        if logger:
            logger.info(f'自动清理历史备份: {path}')

def restore_backup(backup_path, target_dir, logger=None):
    """
    回滚：将备份目录恢复到目标目录。
    :param backup_path: 备份目录
    :param target_dir: 目标目录
    :param logger: 日志对象
    """
    if logger:
        logger.info(f'回滚：恢复备份 {backup_path} -> {target_dir}')
    # 清空目标目录
    for item in os.listdir(target_dir):
        path = os.path.join(target_dir, item)
        if os.path.isdir(path):
            shutil.rmtree(path)
        else:
            os.remove(path)
    # 复制备份内容到目标目录
    for item in os.listdir(backup_path):
        s = os.path.join(backup_path, item)
        d = os.path.join(target_dir, item)
        if os.path.isdir(s):
            shutil.copytree(s, d)
        else:
            parent = os.path.dirname(d)
            if parent and not os.path.exists(parent):
                os.makedirs(parent, exist_ok=True)
            shutil.copy2(s, d)
