

"""
SoIdea-update-python 配置管理模块
支持 toml/json/yaml 配置文件自动加载、参数优先级、模板生成。
依赖：toml, pyyaml
"""

import json
import logging
import os
from typing import Any, Dict, Optional

import toml
import yaml


def load_config(config_path: str, logger: Optional[logging.Logger] = None) -> Dict[str, Any]:
    """
    加载配置文件（支持 toml/json/yaml），返回 dict。

    Args:
        config_path (str): 配置文件路径。
        logger (Optional[logging.Logger]): 可选日志对象。
    Returns:
        Dict[str, Any]: 配置 dict。
    """
    config = {}
    if config_path and os.path.isfile(config_path):
        try:
            ext = os.path.splitext(config_path)[-1].lower()
            with open(config_path, 'r', encoding='utf-8') as f:
                if ext == '.toml':
                    config = toml.load(f)
                elif ext == '.json':
                    config = json.load(f)
                elif ext in ('.yaml', '.yml'):
                    config = yaml.safe_load(f)
                else:
                    config = toml.load(f)
            if logger:
                logger.info(f"已加载配置文件: {config_path}")
        except Exception as e:
            if logger:
                logger.warning(f"配置文件解析失败: {e}")
    elif config_path:
        if logger:
            logger.warning(f"配置文件不存在: {config_path}")
    return config

def get_param_with_priority(cmd_value: Any, api_value: Any, config_value: Any) -> Any:
    """
    参数优先级：命令行 > API > 配置文件。

    Args:
        cmd_value (Any): 命令行参数。
        api_value (Any): API 参数。
        config_value (Any): 配置文件参数。
    Returns:
        Any: 优先级最高的参数值。
    """
    return cmd_value if cmd_value is not None else (api_value if api_value is not None else config_value)

def generate_config_template(config_path: str) -> None:
    """
    自动生成标准升级配置模板，支持 toml/json/yaml。
    :param config_path: 模板文件路径
    """
    template = {
        'backup_dir': 'backup',
        'enable_git': True,
        'enable_tag': True,
        'enable_backup': True,
        'enable_sig': True,
        'enable_hash': True,
        'enable_multithread': False,
        'target_exe': 'ReportTablePro.exe',
        'update_zip': 'update.zip',
        'update_hash': '',
        'update_tag': 'v1.0.0',
        'extract_dir': 'update_tmp',
        'portable_dir': '',
        'proxy': '',
    }
    ext = os.path.splitext(config_path)[-1].lower()
    parent = os.path.dirname(config_path)
    if parent and not os.path.exists(parent):
        os.makedirs(parent, exist_ok=True)
    with open(config_path, 'w', encoding='utf-8') as f:
        if ext == '.toml':
            toml.dump(template, f)
        elif ext == '.json':
            json.dump(template, f, indent=2, ensure_ascii=False)
        elif ext in ('.yaml', '.yml'):
            yaml.safe_dump(template, f, allow_unicode=True)
        else:
            f.write(str(template))
