
"""
SoIdea-update-python 进程管理工具
支持跨平台主程序进程结束
"""
import logging
import os
import shutil
import time

import psutil


# 通过 lock 文件读取 PID 并杀进程
def kill_by_pid_from_lock(lock_path: str, logger=None) -> bool:
	"""
	读取 lock 文件中的 PID 并尝试 kill 进程。

	Args:
		lock_path (str): lock 文件路径。
		logger (logging.Logger, optional): 日志记录器。
	Returns:
		bool: 是否成功 kill。
	"""
	logger = logger or logging.getLogger()
	pid = get_pid_from_lock(lock_path, logger)
	if pid is None:
		logger.warning(f"lock文件无有效PID，跳过 kill_by_pid: {lock_path}")
		return False
	logger.info(f"解析得到 PID={pid}，尝试 kill...")
	return kill_process_by_pid(pid, logger)


def safe_copy2(src: str, dst: str, max_retry: int = 5, wait: int = 1, logger=None) -> bool:
	"""
	尝试多次复制文件，失败时输出目标文件可写状态，日志和控制台都详细记录。

	Args:
		src (str): 源文件路径。
		dst (str): 目标文件路径。
		max_retry (int, optional): 最大重试次数。
		wait (int, optional): 每次重试等待秒数。
		logger (logging.Logger, optional): 日志记录器。
	Returns:
		bool: 是否复制成功。
	"""
	# 日志诊断输出
	logger_info = f"[safe_copy2] logger={repr(logger)}"
	if logger and hasattr(logger, 'handlers'):
		logger_info += f", handlers={logger.handlers}"
		for h in logger.handlers:
			if hasattr(h, 'baseFilename'):
				logger_info += f", log_file={h.baseFilename}"
	print(logger_info)
	if logger:
		logger.info(logger_info)
	else:
		logging.getLogger().info(logger_info)

	max_retry = 10
	wait = 1
	for i in range(max_retry):
		try:
			shutil.copy2(src, dst)
			msg = f"第{i+1}次尝试复制成功: {src} -> {dst}"
			if logger:
				logger.info(msg)
			else:
				logging.getLogger().info(msg)
			print(msg)
			return True
		except PermissionError as e:
			can_write = os.access(dst, os.W_OK)
			msg = f"第{i+1}次尝试复制失败: {e}，目标文件可写: {can_write}"
			if logger:
				logger.warning(msg)
			else:
				logging.getLogger().warning(msg)
			print(msg)
			time.sleep(wait)
		except Exception as e:
			can_write = os.access(dst, os.W_OK)
			msg = f"复制文件异常: {e}，目标文件可写: {can_write}"
			if logger:
				logger.error(msg)
			else:
				logging.getLogger().error(msg)
			print(msg)
			time.sleep(wait)
	msg = f"多次尝试后仍无法复制 {src} 到 {dst}"
	if logger:
		logger.error(msg)
	else:
		logging.getLogger().error(msg)
	print(msg)
	return False


# 新增：通过lock文件获取PID
def get_pid_from_lock(lock_path: str, logger=None) -> int | None:
	"""
	读取lock文件获取PID，假定内容为PID字符串。

	Args:
		lock_path (str): lock 文件路径。
		logger (logging.Logger, optional): 日志记录器。
	Returns:
		int | None: 解析到的 PID，失败返回 None。
	"""
	logger = logger or logging.getLogger()
	try:
		with open(lock_path, 'r', encoding='utf-8') as f:
			pid_str = f.read().strip()
			if pid_str.isdigit():
				logger.info(f"读取lock文件成功: {lock_path}, PID={pid_str}")
				return int(pid_str)
			else:
				logger.warning(f"lock文件内容不是有效PID: {lock_path}, 内容='{pid_str}'")
	except Exception as e:
		logger.warning(f"读取lock文件失败: {lock_path} {e}")
	return None

# 新增：通过PID直接kill进程
def kill_process_by_pid(pid: int, logger=None) -> bool:
	"""
	直接通过PID结束进程，跨平台，推荐用psutil。

	Args:
		pid (int): 进程号。
		logger (logging.Logger, optional): 日志记录器。
	Returns:
		bool: 是否成功 kill。
	"""
	logger = logger or logging.getLogger()
	try:
		proc = psutil.Process(pid)
		exe_path = proc.exe() if hasattr(proc, 'exe') else ''
		proc.kill()
		logger.info(f"[psutil进程kill] 已结束进程: PID={pid}, exe_path={exe_path}")
		return True
	except psutil.NoSuchProcess:
		logger.warning(f"[psutil进程kill] 未找到进程: PID={pid}")
	except Exception as e:
		logger.exception(f"[psutil进程kill] 结束进程失败: PID={pid}: {e}")
	return False



# 新版 kill_process_by_name：参数更直观，日志更详细，纯 psutil 实现
def kill_process_by_name(
	exe_name: str,
	parent_dir: str = None,
	logger=None,
	lock_path: str = None,
	wait_loops: int = 9,
	wait_time: int = 1
) -> bool:
	"""
	优先用 lock 文件 pid 杀进程，kill 后多次等待，进程名和路径都一致才 kill。

	Args:
		exe_name (str): 进程名（如 helloworld_lock.exe）。
		parent_dir (str, optional): 目标目录（用于路径比对，可为 None）。
		logger (logging.Logger, optional): 日志对象。
		lock_path (str, optional): lock 文件路径，优先用其 pid 杀进程。
		wait_loops (int, optional): kill 后等待循环次数。
		wait_time (int, optional): 每次等待秒数。
	Returns:
		bool: 是否成功 kill。
	"""
	logger = logger or logging.getLogger()
	current_pid = os.getpid()
	exe_name_only = os.path.basename(exe_name).lower()
	target_path = os.path.normcase(os.path.abspath(os.path.join(parent_dir, exe_name_only))) if parent_dir else None
	logger.info(f"[进程查杀] 目标进程名: {exe_name_only}, 目标目录: {parent_dir}, 目标绝对路径: {target_path}, lock_path={lock_path}, 当前进程PID={current_pid}")
	# 1. 优先用 lock 文件 pid 杀进程
	if lock_path:
		logger.info(f"优先尝试用 lock 文件 pid 杀进程: {lock_path}")
		if kill_by_pid_from_lock(lock_path, logger):
			logger.info("lock pid kill 成功，直接返回")
			for _ in range(wait_loops):
				time.sleep(wait_time)
			return True
		else:
			logger.info("lock pid kill 失败，尝试进程名+路径查杀")
	# 2. 进程名+路径查杀（名称和路径都一致才 kill）
	killed = 0
	debug_hit_list = []
	for proc in psutil.process_iter(['pid', 'name', 'exe']):
		try:
			name = (proc.info['name'] or '').lower()
			exe_path = proc.info['exe'] or ''
			exe_path_norm = os.path.normcase(os.path.abspath(exe_path)) if exe_path else ''
			pid = proc.info['pid']
			logger.debug(f"[进程遍历] name={name}, pid={pid}, exe_path={exe_path}, exe_path_norm={exe_path_norm}")
			# 只在进程名和路径都一致时 kill
			if name == exe_name_only and target_path and exe_path_norm == target_path:
				if pid == current_pid:
					logger.warning(f"[自杀保护] 命中自身进程，跳过 kill: PID={pid}, exe_path={exe_path_norm}")
					print(f"[自杀保护] 命中自身进程，跳过 kill: PID={pid}, exe_path={exe_path_norm}")
					continue
				hit_info = f"命中进程: PID={pid}, exe_path={exe_path_norm}"
				debug_hit_list.append(hit_info)
				try:
					proc.kill()
					logger.info(f"已结束进程: {exe_path_norm} (PID={pid})")
					print(f"已结束进程: {exe_path_norm} (PID={pid})")
					killed += 1
				except Exception as e:
					logger.warning(f"结束进程失败: {exe_path_norm} (PID={pid}): {e}")
					print(f"[WARN] 结束进程失败: {exe_path_norm} (PID={pid}): {e}")
		except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess) as e:
			logger.warning(f"进程遍历异常: PID={getattr(proc, 'pid', '?')}, 错误: {e}")
	# 输出所有命中进程信息，便于调试
	if debug_hit_list:
		logger.info("[调试] 命中进程列表：\n" + "\n".join(debug_hit_list))
	# 3. kill 后多次等待
	for _ in range(wait_loops):
		time.sleep(wait_time)
	if killed == 0:
		logger.info(f"未找到待结束进程: {exe_name_only}")
		print(f"未找到待结束进程: {exe_name_only}")
	if killed > 0:
		logger.info(f"已尝试 kill {killed} 个进程，等待 {wait_loops * wait_time} 秒以确保文件释放...")
	return killed > 0
