"""
Ed25519 公钥签名/验签模块，依赖 cryptography。
"""

# 标准库
import os

from cryptography.exceptions import InvalidSignature
from cryptography.hazmat.primitives import serialization
# 第三方依赖
from cryptography.hazmat.primitives.asymmetric.ed25519 import Ed25519PrivateKey

KEY_DIR = os.path.join(os.path.dirname(__file__), '..', 'keys')
PRIVATE_KEY_FILE = os.path.join(KEY_DIR, 'private_key.pem')
PUBLIC_KEY_FILE = os.path.join(KEY_DIR, 'public_key.pem')

os.makedirs(KEY_DIR, exist_ok=True)

def generate_keypair() -> tuple[str, str]:
    """
    生成 Ed25519 密钥对，并保存到文件。

    Returns:
        tuple[str, str]: 私钥文件路径、公钥文件路径。
    """
    private_key = Ed25519PrivateKey.generate()
    public_key = private_key.public_key()
    priv_dir = os.path.dirname(PRIVATE_KEY_FILE)
    pub_dir = os.path.dirname(PUBLIC_KEY_FILE)
    if priv_dir and not os.path.exists(priv_dir):
        os.makedirs(priv_dir, exist_ok=True)
    if pub_dir and not os.path.exists(pub_dir):
        os.makedirs(pub_dir, exist_ok=True)
    with open(PRIVATE_KEY_FILE, 'wb') as f:
        f.write(private_key.private_bytes(
            encoding=serialization.Encoding.PEM,
            format=serialization.PrivateFormat.PKCS8,
            encryption_algorithm=serialization.NoEncryption()
        ))
    with open(PUBLIC_KEY_FILE, 'wb') as f:
        f.write(public_key.public_bytes(
            encoding=serialization.Encoding.PEM,
            format=serialization.PublicFormat.SubjectPublicKeyInfo
        ))
    return PRIVATE_KEY_FILE, PUBLIC_KEY_FILE

def sign_file(file_path: str, private_key_path: str = PRIVATE_KEY_FILE) -> str:
    """
    使用 Ed25519 私钥对文件内容进行签名，签名结果保存为 .sig 文件。

    Args:
        file_path (str): 待签名文件路径。
        private_key_path (str, optional): 私钥文件路径。
    Returns:
        str: 签名文件路径。
    """
    with open(private_key_path, 'rb') as f:
        private_key = serialization.load_pem_private_key(f.read(), password=None)
    with open(file_path, 'rb') as f:
        data = f.read()
    signature = private_key.sign(data)
    sig_path = file_path + '.sig'
    sig_dir = os.path.dirname(sig_path)
    if sig_dir and not os.path.exists(sig_dir):
        os.makedirs(sig_dir, exist_ok=True)
    with open(sig_path, 'wb') as f:
        f.write(signature)
    return sig_path

def verify_file(file_path: str, sig_path: str = None, public_key_path: str = PUBLIC_KEY_FILE) -> bool:
    """
    使用 Ed25519 公钥验证文件签名。

    Args:
        file_path (str): 待验证文件路径。
        sig_path (str, optional): 签名文件路径。
        public_key_path (str, optional): 公钥文件路径。
    Returns:
        bool: 验证是否通过。
    """
    if sig_path is None:
        sig_path = file_path + '.sig'
    with open(public_key_path, 'rb') as f:
        public_key = serialization.load_pem_public_key(f.read())
    with open(file_path, 'rb') as f:
        data = f.read()
    if not os.path.isfile(sig_path):
        return False
    with open(sig_path, 'rb') as f:
        signature = f.read()
    try:
        public_key.verify(signature, data)
        return True
    except InvalidSignature:
        return False

if __name__ == '__main__':
    import argparse
    parser = argparse.ArgumentParser(description='Ed25519升级包签名/验签工具')
    parser.add_argument('--gen', action='store_true', help='生成密钥对')
    parser.add_argument('--sign', type=str, help='对文件签名')
    parser.add_argument('--verify', nargs=2, metavar=('file', 'sig'), help='验证文件签名')
    args = parser.parse_args()
    if args.gen:
        pk, pubk = generate_keypair()
        print(f'密钥已生成: {pk}, {pubk}')
    elif args.sign:
        sig = sign_file(args.sign)
        print(f'签名已生成: {sig}')
    elif args.verify:
        ok = verify_file(args.verify[0], args.verify[1])
        print('验签结果:', '通过' if ok else '失败')
    else:
        parser.print_help()
