# 标准库
import logging
import os
import shutil

from .backup import backup_dir, restore_backup
# 本地依赖
from .config import load_config
from .lock_utils import set_forced_closed_marker, was_file_forced_closed
from .log_utils import ProgressBar, setup_logger
from .package import overwrite_files, verify_and_extract
from .platform_utils import is_admin, normalize_path, set_executable_permission
from .process_utils import kill_process_by_name
from .version_utils import compare_version


def is_new_version(current_version: str, latest_tag: str) -> bool:
    """
    判断是否有新版本，兼容多种格式。

    Args:
        current_version (str): 当前版本号。
        latest_tag (str): 最新版本号。
    Returns:
        bool: 是否有新版本。
    """
    try:
        result = compare_version(latest_tag, current_version)
    except Exception as e:
        logging.warning(f'版本号解析失败: {e}，回退为字符串比较: current={current_version}, latest={latest_tag}')
        result = str(latest_tag) > str(current_version)
    logging.info(f"当前版本: {current_version}，最新版本: {latest_tag}，需要升级: {result}")
    return result


class UpdateManager:
    def __init__(self, config_path: str, workdir: str = None, logger: logging.Logger = None):
        """
        升级管理器，负责自动化升级流程。

        Args:
            config_path (str): 配置文件路径。
            workdir (str, optional): 工作目录。
            logger (logging.Logger, optional): 日志记录器。
        """
        self.workdir = normalize_path(workdir or os.getcwd())
        self.logger = logger or setup_logger(logdir=self.workdir)
        self.config = load_config(normalize_path(config_path), self.logger)
        self.backup_root = normalize_path(self.config.get('backup_dir', os.path.join(self.workdir, 'backup')))
        self.enable_git = self.config.get('enable_git', True)
        self.enable_tag = self.config.get('enable_tag', True)
        self.enable_backup = self.config.get('enable_backup', True)
        self.enable_sig = self.config.get('enable_sig', True)
        self.enable_hash = self.config.get('enable_hash', True)
        self.enable_multithread = self.config.get('enable_multithread', False)
        self.target_exe = normalize_path(self.config.get('target_exe', None)) if self.config.get('target_exe', None) else None
        self.update_zip = normalize_path(self.config.get('update_zip', None)) if self.config.get('update_zip', None) else None
        self.update_hash = self.config.get('update_hash', None)
        self.update_tag = self.config.get('update_tag', None)
        self.extract_dir = normalize_path(self.config.get('extract_dir', os.path.join(self.workdir, 'update_tmp')))

    def run(self) -> bool:
        """
        执行升级流程。

        Returns:
            bool: 升级是否成功。
        """
        self.logger.info('升级流程开始')
        if not is_admin():
            self.logger.error('请以管理员权限运行升级程序！')
            return False
        if self.enable_backup:
            backup_path = backup_dir(self.workdir, self.backup_root, self.logger)
        else:
            backup_path = None

        # 检查主程序文件是否被强制关闭
        if self.target_exe and was_file_forced_closed(self.target_exe):
            self.logger.warning(f'主程序 {self.target_exe} 之前被强制关闭，建议检查数据完整性。')

        if self.target_exe:
            kill_process_by_name(self.target_exe, self.workdir, self.logger)
            set_forced_closed_marker(self.target_exe)

        # 下载与校验升级包（支持多线程扩展）
        if self.enable_multithread:
            ok = self.download_and_verify_multithread()
        else:
            self.logger.info('校验并解压升级包...')
            with ProgressBar(total=100, desc="解压校验", width=40) as bar:
                ok = verify_and_extract(
                    self.update_zip, self.extract_dir,
                    expected_hash=self.update_hash if self.enable_hash else None,
                    sig_check=self.enable_sig, logger=self.logger, progress_cb=lambda n: bar.update(n))
        if not ok:
            self.logger.error('升级包校验或解压失败，尝试回滚...')
            if backup_path:
                restore_backup(backup_path, self.workdir, self.logger)
            return False

        # 覆盖主程序
        self.logger.info('覆盖主程序文件...')
        with ProgressBar(total=100, desc="文件覆盖", width=40) as bar:
            overwrite_files(self.extract_dir, self.workdir, self.logger, progress_cb=lambda n: bar.update(n))

        self.logger.info('升级流程完成')
        return True

    def download_and_verify_multithread(self) -> bool:
        """
        多线程下载与校验扩展点，当前为占位实现。

        Returns:
            bool: 下载与校验是否成功。
        """
        self.logger.info('多线程下载与校验功能尚未实现，暂用单线程流程')
        with ProgressBar(total=100, desc="多线程解压校验", width=40) as bar:
            return verify_and_extract(
                self.update_zip, self.extract_dir,
                expected_hash=self.update_hash if self.enable_hash else None,
                sig_check=self.enable_sig, logger=self.logger, progress_cb=lambda n: bar.update(n))

    # 预留多线程下载、可选加密包、web/GUI等扩展点

logger = logging.getLogger("SoIdea-update-python.manager")

def atomic_update(src_dir: str, dst_dir: str) -> str:
    """
    原子升级：先写入临时目录，成功后整体替换。

    Args:
        src_dir (str): 源目录。
        dst_dir (str): 目标目录。
    Returns:
        str: 备份目录路径。
    """
    tmp_dir = normalize_path(dst_dir + "_tmp")
    if os.path.exists(tmp_dir):
        shutil.rmtree(tmp_dir)
    shutil.copytree(normalize_path(src_dir), tmp_dir)
    # 权限处理
    for root, dirs, files in os.walk(tmp_dir):
        for f in files:
            set_executable_permission(os.path.join(root, f))
    # 备份原目录
    backup_dir = normalize_path(dst_dir + "_bak")
    if os.path.exists(backup_dir):
        shutil.rmtree(backup_dir)
    if os.path.exists(dst_dir):
        shutil.move(dst_dir, backup_dir)
    with ProgressBar(total=100, desc="原子替换", width=40) as bar:
        shutil.move(tmp_dir, dst_dir)
        bar.update(100)
    logger.info(f"原子升级完成: {dst_dir}")
    # 回滚点：如需回滚可用 backup_dir
    return backup_dir
