"""
版本号工具：兼容 packaging 和 semver，统一解析、比较、递增
"""

# 第三方依赖
from packaging.version import InvalidVersion, Version

try:
    import semver
    HAS_SEMVER = True
except ImportError:
    HAS_SEMVER = False

# 公共版本号输出函数，根据入口文件名返回对应版本号。
VERSION_BUILD_UPDATE_PACKAGE = '1.0.44'
VERSION_UPDATER = '1.1.44'
VERSION_WEB_SERVICE = '1.2.44'
VERSION_HELLOWORLD_LOCK = '1.3.44'

def get_version_info(entry: str = None) -> str:
    """
    公共版本号输出函数，根据入口文件名返回对应版本号。

    Args:
        entry (str, optional): 文件名（如 'build_update_package.py'），不区分大小写。
    Returns:
        str: 版本号字符串。
    """
    entry = (entry or '').lower()
    if 'build_update_package' in entry:
        return VERSION_BUILD_UPDATE_PACKAGE
    elif 'updater' in entry:
        return VERSION_UPDATER
    elif 'web_service' in entry:
        return VERSION_WEB_SERVICE
    elif 'helloworld_lock' in entry:
        return VERSION_HELLOWORLD_LOCK
    else:
        return 'unknown'


def parse_version(ver_str: str):
    """
    解析版本号，优先用 packaging，失败则 semver，最后回退字符串。

    Args:
        ver_str (str): 版本号字符串。
    Returns:
        Version | semver.VersionInfo | str: 解析后的版本对象或原始字符串。
    """
    try:
        return Version(ver_str)
    except InvalidVersion:
        try:
            return semver.VersionInfo.parse(ver_str)
        except ValueError:
            return ver_str

def compare_version(v1: str, v2: str) -> bool:
    """
    比较版本号，判断 v1 是否大于 v2。

    Args:
        v1 (str): 版本号1。
        v2 (str): 版本号2。
    Returns:
        bool: 如果 v1 > v2 返回 True，否则返回 False。
    """
    try:
        return Version(v1) > Version(v2)
    except InvalidVersion:
        try:
            return semver.compare(v1, v2) > 0
        except ValueError:
            return str(v1) > str(v2)

def bump_version(ver_str: str, part: str = 'patch') -> str:
    """
    递增版本号，优先用 semver，失败则原样返回。

    Args:
        ver_str (str): 版本号字符串。
        part (str, optional): 递增部分（'major', 'minor', 'patch'）。默认 'patch'。
    Returns:
        str: 递增后的版本号字符串。
    """
    try:
        v = semver.VersionInfo.parse(ver_str)
        if part == 'major':
            return str(v.bump_major())
        elif part == 'minor':
            return str(v.bump_minor())
        else:
            return str(v.bump_patch())
    except ValueError:
        return ver_str
