import datetime
import logging
# 标准库
import os
import shutil
import subprocess
import sys
import time

from .process import restart_main_program
from .process_utils import kill_process_by_name, safe_copy2


def git_backup(repo_dir, logger=None):
    """
    自动 git add/commit 当前目录所有变更，commit 信息带时间戳。
    """
    ts = datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    try:
        subprocess.run(['git', 'add', '.'], cwd=repo_dir, check=True)
        subprocess.run(['git', 'commit', '-m', f'auto-backup before upgrade {ts}'], cwd=repo_dir, check=True)
        msg = f"[GIT] 已自动备份所有变更 (commit at {ts})"
        if logger:
            logger.info(msg)
        print(msg)
    except Exception as e:
        msg = f"[GIT] 自动备份失败: {e}"
        if logger:
            logger.warning(msg)
        print(msg)

def check_admin_permission(logger=None):
    """
    检查当前进程是否有管理员/root 权限，跨平台。
    """
    is_admin = False
    if os.name == 'nt':
        try:
            import ctypes
            is_admin = ctypes.windll.shell32.IsUserAnAdmin() != 0
        except Exception:
            is_admin = False
    else:
        is_admin = (os.geteuid() == 0)
    if not is_admin:
        msg = '警告：当前进程没有管理员/root 权限，部分操作可能失败。'
        if logger:
            logger.warning(msg)
        print(msg)
    return is_admin

def backup_exe(target_exe, logger=None):
    """
    备份目标 EXE 为 .bak 文件。
    """
    bak_exe = target_exe + ".bak"
    if os.path.exists(target_exe):
        shutil.copy2(target_exe, bak_exe)
        msg = f"已备份旧 EXE: {bak_exe}"
        if logger:
            logger.info(msg)
        print(msg)
    return bak_exe

def restore_exe(target_exe, logger=None):
    """
    回滚 .bak 文件为原名。
    """
    bak_exe = target_exe + ".bak"
    if os.path.exists(bak_exe):
        shutil.move(bak_exe, target_exe)
        msg = f"复制新 EXE 失败，已回滚旧 EXE: {bak_exe}"
        if logger:
            logger.warning(msg)
        print(msg)
        return True
    return False
"""
自升级工具库：支持跨平台的自升级流程，便于主程序/升级器等复用。
- 检测是否在 update 目录
- 结束父目录同名进程
- 复制自身到父目录
- 重启父目录 exe 并退出自身
"""


def is_in_update_dir(update_dir_name='update'):
    """判断当前程序是否运行在 update 目录下（可自定义目录名）"""
    cur_dir = os.path.abspath(os.path.dirname(sys.argv[0]))
    return os.path.basename(cur_dir).lower() == update_dir_name.lower()

def get_parent_dir(path):
    return os.path.dirname(os.path.abspath(path))


def self_upgrade(exe_name, update_dir=None, logger=None, wait_loops=3, wait_time=3):
    """
    通用自升级流程：
    - 结束父目录同名进程
    - 复制自身到父目录（多次重试）
    - 重启父目录 exe 并退出自身
    参数：
        exe_name: 目标 exe 文件名（如 helloworld_lock.exe）
        update_dir: 当前 update 目录（默认自动检测）
        logger: 可选日志对象
    """
    cur_dir = update_dir or os.path.abspath(os.path.dirname(sys.argv[0]))
    parent_dir = get_parent_dir(cur_dir)
    target_exe = os.path.join(parent_dir, exe_name)

    # 1. 权限检测
    check_admin_permission(logger)

    # 2. git 备份
    git_backup(parent_dir, logger)

    # 3. 备份旧 EXE
    backup_exe(target_exe, logger)
    # 1. 输出kill参数，仅用进程名杀进程（不再默认用 lock pid）

    msg = f"即将结束进程，参数: exe_name={exe_name}, parent_dir={parent_dir}"
    if logger:
        logger.info(msg)
    else:
        logging.getLogger().info(msg)
    print(msg)
    # 只用进程名杀进程
    if logger:
        logger.info(f"仅用进程名结束父目录进程: {exe_name} in {parent_dir}")
    kill_process_by_name(exe_name, parent_dir, logger=logger, wait_loops=wait_loops, wait_time=wait_time)
    time.sleep(1)  # 结束进程后多等1秒，确保释放
    # 2. 复制自身到父目录（多次重试）
    src_exe = sys.executable if getattr(sys, 'frozen', False) else os.path.abspath(sys.argv[0])
    if logger:
        logger.info(f"复制自身到: {target_exe}")
    ok = safe_copy2(src_exe, target_exe, max_retry=5, wait=1, logger=logger)
    if not ok:
        errmsg = f"自升级失败: 文件被占用，无法复制 {src_exe} 到 {target_exe}"
        if logger:
            logger.error(errmsg)
        else:
            logging.getLogger().error(errmsg)
        print(errmsg)
        # 自动回滚
        restored = restore_exe(target_exe, logger)
        if restored:
            # 可选：重启旧 EXE
            try:
                restart_main_program(target_exe)
            except Exception as e:
                print(f"[回滚] 重启旧 EXE 失败: {e}")
        sys.exit(1)
    # 3. 启动父目录 exe，增加重试机制
    max_start_retry = 6
    start_wait = 2
    started = False
    for i in range(max_start_retry):
        try:
            if logger:
                logger.info(f"第{i+1}次尝试重启父目录 exe: {target_exe}")
            restart_main_program(target_exe)
            started = True
            break
        except Exception as e:
            errmsg = f"第{i+1}次重启父目录 exe 失败: {e}"
            if logger:
                logger.error(errmsg)
            else:
                logging.getLogger().error(errmsg)
            print(errmsg)
            time.sleep(start_wait)
    if not started:
        errmsg = f"多次尝试后仍无法启动父目录 exe: {target_exe}，请检查文件和权限。"
        if logger:
            logger.error(errmsg)
        else:
            logging.getLogger().error(errmsg)
        print(errmsg)
        sys.exit(2)
    # 4. 退出自身
    if logger:
        logger.info("自升级完成，退出自身")
    sys.exit(0)
    # 4. 退出自身
    if logger:
        logger.info("自升级完成，退出自身")
    sys.exit(0)
    sys.exit(0)
