

import os
import subprocess
import time

import pyminizip

from .backup import backup_dir
from .config import load_config
from .package import overwrite_files, verify_and_extract
from .process_utils import kill_process_by_pid, safe_copy2


# 解压 zip 文件到指定目录，仅用 pyminizip，支持密码。
def unzip_with_password(zip_path: str, extract_to: str, password: str = None, logger=None) -> bool:
    """
    仅用 pyminizip 解压 zip 文件到指定目录，支持密码。
    Args:
        zip_path (str): zip 文件路径。
        extract_to (str): 解压目标目录。
        password (str, optional): 解压密码。
        logger (logging.Logger, optional): 日志记录器。
    Returns:
        bool: 解压是否成功。
    """
    try:
        pyminizip.uncompress(zip_path, password, extract_to, 0)
        if logger:
            logger.info(f"升级包已解压到: {extract_to} (密码:{'***' if password else '无'})")
        return True
    except Exception as e:
        if logger:
            logger.error(f"解压升级包失败: {e}")
        return False







def launch_updater(updater_path: str, old_exe_path: str, old_pid: int, logger=None) -> None:
    """
    启动 updater.exe 并传递旧exe路径和pid参数。

    Args:
        updater_path (str): updater 可执行文件路径。
        old_exe_path (str): 旧主程序路径。
        old_pid (int): 旧主程序进程号。
        logger (logging.Logger, optional): 日志记录器。
    """
    cmd = [updater_path, "--old-exe", old_exe_path, "--old-pid", str(old_pid)]
    if logger:
        logger.info(f"启动 updater: {' '.join(cmd)}")
    os.spawnv(os.P_NOWAIT, updater_path, cmd)


def wait_and_kill_process(pid: int, logger=None, timeout: int = 30) -> bool:
    """
    等待并结束指定pid进程，复用 kill_process_by_pid。

    Args:
        pid (int): 进程号。
        logger (logging.Logger, optional): 日志记录器。
        timeout (int, optional): 等待超时时间（秒）。默认 30。
    Returns:
        bool: 是否成功结束进程。
    """
    for _ in range(timeout):
        if not kill_process_by_pid(pid, logger):
            time.sleep(1)
        else:
            return True
    if logger:
        logger.error(f"进程 {pid} 未能正常结束")
    return False


def replace_exe(new_exe: str, target_path: str, logger=None) -> bool:
    """
    用新exe替换旧exe，复用 safe_copy2。

    Args:
        new_exe (str): 新主程序路径。
        target_path (str): 目标路径。
        logger (logging.Logger, optional): 日志记录器。
    Returns:
        bool: 是否替换成功。
    """
    try:
        safe_copy2(new_exe, target_path, logger=logger)
        if logger:
            logger.info(f"已用新文件替换: {target_path}")
        return True
    except Exception as e:
        if logger:
            logger.error(f"替换主程序失败: {e}")
        return False


def upgrade_by_upload(zip_path: str, logger=None, config_path: str = None) -> tuple:
    """
    升级包上传后自动升级主程序，供 web_service.py 调用。

    Args:
        zip_path (str): 升级包路径。
        logger (logging.Logger, optional): 日志记录器。
        config_path (str, optional): 配置文件路径。
    Returns:
        tuple: (success: bool, msg: str)
    """
    # 加载配置
    config = load_config(config_path, logger) if config_path else {}
    workdir = config.get('workdir', os.getcwd())
    backup_root = config.get('backup_dir', os.path.join(workdir, 'backup'))
    target_exe = config.get('target_exe', None)
    extract_dir = config.get('extract_dir', os.path.join(workdir, 'update_tmp'))
    # 校验并解压升级包（如需密码请在 config 中传递）
    ok = verify_and_extract(zip_path, extract_dir, logger=logger)
    if not ok:
        return False, '升级包校验或解压失败'
    # 备份主程序
    if target_exe and os.path.exists(target_exe):
        backup_dir(workdir, backup_root, logger)
    # 覆盖主程序文件
    overwrite_files(extract_dir, workdir, logger)
    msg = f'升级完成，主程序已覆盖: {workdir}'
    if logger:
        logger.info(msg)
    return True, msg

def run_and_log_version(exe_path: str, logger=None) -> None:
    """
    运行新 web_service.exe 并记录版本号。

    Args:
        exe_path (str): 可执行文件路径。
        logger (logging.Logger, optional): 日志记录器。
    """
    try:
        result = subprocess.run([exe_path, "--version"], capture_output=True, text=True, timeout=10)
        if logger:
            logger.info(f"新web_service版本号: {result.stdout.strip()}")
        return result.stdout.strip()
    except Exception as e:
        if logger:
            logger.error(f"运行新web_service获取版本号失败: {e}")
        return None
