# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['solid2',
 'solid2.core',
 'solid2.core.builtins',
 'solid2.core.object_base',
 'solid2.examples',
 'solid2.extensions',
 'solid2.extensions.bosl2',
 'solid2.extensions.greedy_scad_interface',
 'solid2.libs',
 'solid2.libs.BOSL2.scripts',
 'solid2.libs.py_scadparser']

package_data = \
{'': ['*'],
 'solid2.examples': ['11-font/*'],
 'solid2.libs': ['BOSL2/*',
                 'BOSL2/images/*',
                 'BOSL2/tests/*',
                 'BOSL2/tutorials/*']}

install_requires = \
['ply>=3.11,<4.0', 'setuptools>=65.6.3']

setup_kwargs = {
    'name': 'solidpython2',
    'version': '2.0.0b4',
    'description': 'Python interface to the OpenSCAD declarative geometry language',
    'long_description': '\n.. image:: https://readthedocs.org/projects/solidpython2/badge/?version=latest\n    :target: http://solidpython2.readthedocs.io/en/latest/?badge=latest\n    :alt: Documentation Status\n\n**If you switch from the regular SolidPython:master branch to this branch, have a\nlook at** `Version 2.x.x`_.\n\nSolidPython\n===========\n\n.. contents::\n   \nOpenSCAD for Python\n-------------------\n\nSolidPython is a generalization of Phillip Tiefenbacher\'s openscad\nmodule, found on `Thingiverse <http://www.thingiverse.com/thing:1481>`__. It\ngenerates valid OpenSCAD code from Python code with minimal overhead. Here\'s a\nsimple example:\n\nThis Python code:\n\n.. code:: python\n\n    from solid2 import *\n    d = difference()(\n        cube(10),\n        sphere(15)\n    )\n    d.as_scad()\n\nGenerates this OpenSCAD code:\n\n.. code:: python\n\n    difference(){\n        cube(10);\n        sphere(15);\n    }\n\nThat doesn\'t seem like such a savings, but the following SolidPython code is a\nlot shorter (and I think clearer) than the SCAD code it compiles to:\n\n.. code:: python\n\n    from solid2 import *\n    d = cube(5) + right(5)(sphere(5)) - cylinder(r=2, h=6)\n\nGenerates this OpenSCAD code:\n\n.. code::\n\n    difference(){\n        union(){\n            cube(5);\n            translate( [5, 0,0]){\n                sphere(5);\n            }\n        }\n        cylinder(r=2, h=6);\n    }\n\nAdvantages\n----------\n\nIn contrast to OpenSCAD -- which is a constrained domain specific language --\nPython is a full blown modern programming language and as such supports\npretty much all modern programming features. Furthermore a huge number of\nlibraries is available.\n\nSolidPython lets you use all these fancy python features to generate your\nconstructive solid geometry models.\n\nOn the one hand it makes the generation of your models a lot easier, because\nyou don\'t need to learn another domain specific language and you can use all\nthe programming technique you\'re already familiar with. On the other hand it\ngives you a lot more power, because you can use all the comprehensive python\nlibraries to generate your models.\n\nI would almost say this enables you to do what ever you want with ease.\nAs (maybe little uncommon) example, you could write a program that:\n\n  - looks up the mail adress of your actuall president (based on your ip address)\n  - writes a mail to him or her and asks for a portrait\n  - waits for a reply\n  - generates a heightmap from the picture you received and maps it onto a vase\n\nThis should be pretty straight forward with SolidPython but is impossible with\npure OpenSCAD.\n\nFurhtermore SolidPython 2.x.x is designed to be extendible. As such you can extend SolidPython itself using python. Actually parts of SolidPython itself are implemented as extensions (everything but the core one-to-one mapping of OpenScad to Python), these include operators, access style syntax, convenience functions, scad_interface and bosl2 support. Furthermore some of the SolidPython 1.x.x solid.utils features are also implemented as extensions (bill of material & part-hole).\n\nInstalling SolidPython\n----------------------\n\n-  Install latest release via\n   `PyPI <https://pypi.python.org/pypi/solidpython2>`__:\n\n   .. code:: bash\n\n       pip install solidpython2\n\n   (You may need to use ``sudo pip install solidpython2``, depending on\n   your environment. This is commonly discouraged though. You\'ll be happiest \n   working in a `virtual environment <https://docs.python.org/3/tutorial/venv.html>`__ \n   where you can easily control dependencies for a given project)\n\n- Install current master straight from Github:\n\n  .. code:: bash\n\n      pip install git+https://github.com/jeff-dh/SolidPython\n\nUsing SolidPython\n-----------------\n\n-  Include SolidPython at the top of your Python file:\n\n   .. code:: python\n\n       from solid2 import *\n\n   (See `this issue <https://github.com/SolidCode/SolidPython/issues/114>`__ for \n   a discussion of other import styles)\n\n-  OpenSCAD uses curly-brace blocks ({}) to create its tree. SolidPython\n   uses parentheses with comma-delimited lists. \n   \n   **OpenSCAD:**\n\n   .. code::\n\n       difference(){\n           cube(10);\n           sphere(15);\n       }\n\n   **SolidPython:**\n\n   .. code::\n\n       d = difference()(\n           cube(10),  # Note the comma between each element!\n           sphere(15)\n       )\n\n-  Call ``py_scad_obj.as_scad()`` to generate SCAD code. This returns\n   a string of valid OpenSCAD code.\n-  *or*: call ``py_scad_obj.save_as_scad("filepath.scad")`` to store\n   that code in a file.\n-  If ``filepath.scad`` is open in the OpenSCAD IDE and Design => \'Automatic\n   Reload and Compile\' is checked in the OpenSCAD IDE, running\n   ``py_scad_obj.save_as_scad()`` from Python will load the object in the\n   IDE.\n-  Alternately, you could call OpenSCAD\'s command line and render\n   straight to STL.\n\nImporting OpenSCAD code\n-----------------------\n\n- Use ``solid2.import_scad(path)`` to import OpenSCAD code. Relative paths will check the current location designated in `OpenSCAD library directories <https://en.wikibooks.org/wiki/OpenSCAD_User_Manual/Libraries>`__.\n\n**Ex:** \n\n``scadfile.scad``\n\n.. code::\n\n    module box(w,h,d){\n        cube([w,h,d]);\n    }\n\n``your_file.py``\n\n.. code:: python\n\n    from solid2 import *\n\n    scadfile = import_scad(\'/path/to/scadfile.scad\') \n    b = scadfile.box(2,4,6)\n    b.save_as_scad(\'out_file.scad\')\n\n- Recursively import OpenSCAD code by calling ``import_scad()`` with a directory argument.\n\n.. code:: python\n\n    from solid2 import *\n\n    # MCAD is OpenSCAD\'s most common utility library: https://github.com/openscad/MCAD\n    # If it\'s installed for OpenSCAD (on MacOS, at: ``$HOME/Documents/OpenSCAD/libraries``)\n    mcad = import_scad(\'MCAD\')\n\n    # MCAD contains about 15 separate packages, each included as its own namespace\n    print(dir(mcad)) # => [\'bearing\', \'bitmap\', \'boxes\', etc...]\n    mount = mcad.motors.stepper_motor_mount(nema_standard=17)\n    mount.save_as_scad(\'motor_mount_file.scad\')\n\n- OpenSCAD has the ``use()`` and ``include()`` statements for importing SCAD code, and SolidPython has them, too. They pollute the global namespace, though, and you may have better luck with ``import_scad()``,\n\n**Ex:**\n\n``scadfile.scad``\n\n.. code::\n\n    module box(w,h,d){\n        cube([w,h,d]);\n    }\n\n``your_file.py``\n\n.. code:: python\n\n    from solid2 import *\n\n    # use() puts the module `box()` into the global namespace\n    use(\'/path/to/scadfile.scad\') \n    b = box(2,4,6)\n    scad_render_to_file(b, \'out_file.scad\')\n\n\nExample Code\n------------\n\nThe best way to learn how SolidPython works is to look at the included\nexample code. If you\'ve installed SolidPython, the following line of\nPython will print (the location of) the examples directory:\n\n.. code:: python\n\n    import os, solid2; print(os.path.dirname(solid2.__file__) + \'/examples\')\n        \n\nOr browse the example code on Github\n`here <https://github.com/jeff-dh/SolidPython/tree/exp_solid/solid2/examples>`__\n\nExtra syntactic sugar\n=====================\n\nBasic operators\n---------------\n\nSolidPython overrides the basic operators + and | (union), - (difference), \\*\nand & (intersection) and ~ (debug). So\n\n.. code:: python\n\n    c = cylinder(r=10, h=5) + cylinder(r=2, h=30)\n\nis the same as:\n\n.. code:: python\n\n    c = union()(\n        cylinder(r=10, h=5),\n        cylinder(r=2, h=30)\n    )\n\nLikewise:\n\n.. code:: python\n\n    c = cylinder(r=10, h=5)\n    c -= cylinder(r=2, h=30)\n\nis the same as:\n\n.. code:: python\n\n    c = difference()(\n        cylinder(r=10, h=5),\n        cylinder(r=2, h=30)\n    )\n\nAccess Style Syntax\n-------------------\n\nSince at least some people (including me) don\'t like the OpenSCAD Syntax, SolidPython 2.x.x introduces the support for the so called "Access-Style-Syntax". This enables you to call some of the SolidPython / OpenSCAD functions as member functions of any OpenSCADObject instead of wrapping it in an instance of it.\n\nIn other words, e.g. code:\n\n.. code:: python\n\n  up(10)(cube(1))\n  #is equal to\n  cube(1).up(10)\n\nThe available member functions are the following:\n\n.. code:: python\n\n  union, difference, intersection, translate, scale, rotate, mirror, resize,\n  color, offset, hull, render, projection, surface, linear_extrude,\n  rotate_extrude, debug, background, root and disable\n\nAlso the convenience functions are available:\n\n.. code:: python\n\n  up, down, left, right, forward, fwd, back, translateX, translateY, translateZ,\n  rotateX, rotateY, rotateZ, mirrorX, mirrorY, mirrorZ, scaleX, scaleY, scaleZ,\n  resizeX, resizeY, resizeZ\n\nFurthermore you can chain these functions, because they all return the transformed OpenSCADObject, e.g.:\n\n.. code:: python\n\n  cube(1).up(10).back(20).rotate(10, 0, 5).mirror(1, 0, 0).color("green").root()\n\nConvenience functions\n---------------------\n\nSolidPython includes a number of convenience functions. Currently these\ninclude:\n\nDirections for arranging things:\n\n.. code:: python\n\n  up, down, left, right, forward, fwd, back\n\nTransformations per dimension:\n\n.. code:: python\n\n  translateX, translateY, translateZ, rotateX, rotateY, rotateZ, mirrorX,\n  mirrorY, mirrorZ, resizeX, resizeY, resizeZ, scaleX, scaleY, scaleZ\n\nFurthermore the operations `translate, scale, resize, mirror, rotate, cube and square` are overwritten in a way that they accept single integer or float values as first parameter. (`translate(1, 2, 3)` equals `translate([1, 2, 3])`)\n\n.. code:: python\n\n    cylinder().rotateY(90).up(10)\n\nseems a lot clearer to me than:\n\n.. code:: python\n\n    translate([0,0,10])(\n        rotate([0, 90, 0])(\n          cylinder()\n    ))\n\nFeatures\n========\n\nBOSL2\n-----\n\nSolidPython supports -- at least -- quite a lot of the **bosl2** library. You can use it by importing the ``solid2.extensions.bosl2``. Take a look at `bosl2 example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/07-libs-bosl2.py>`_ and `mazebox example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/16-mazebox-bosl2.py>`_ to get an idea how to use it and what\'s possible.\n\nI would suggest to use it as kind of a standard library for SolidPython.\nTake a look at their `Wiki <https://github.com/revarbat/BOSL2/wiki>`_ to get an idea about it\'s features.\n\n\nAnimation, Customizer, custom Fonts, ImplicitCad, Extensions\n------------------------------------------------------------\n\nSolidPython supports the following features\n\n* native **OpenSCAD customizer** support `customizer example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/10-customizer.py>`_ `greedy scad interface example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/17-greedy-scad-interface.py>`_\n* native **OpenSCAD animation** support `animation example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/12-animation.py>`_ and `animation example 2 <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/13-animated-bouncing-ball.py>`_\n* **custom fonts** `fonts example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/11-fonts.py>`_\n* supports **ImplicitCAD** `implicitCAD example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/14-implicitCAD.py>`_ `implicitCAD example 2 <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/15-implicitCAD2.py>`_\n* SolidPython is extendible `extensions example 1 <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/08-extensions.py>`_  `extension example 2 <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/09-code-attach-extension.py>`_\n\nJupyter Renderer\n================\n\nSolidPython can be rendered inside a Jupyter Notebook using ViewScad. Unfortunately the pypi version of ``viewscad`` seems to be not compatible with ``solid2``. @jreiberkyle created `this viewscad fork <https://github.com/jreiberkyle/ViewSCAD>`__ and made it work with `solid2` (`#7 <https://github.com/jeff-dh/SolidPython/issues/7>`__)\n\nVersion 2.x.x\n=============\n\nSolidPython 2.x.x is a refactored version of SolidPython 1.x.x.\nThe refactoring process was based on the following proposal:\nhttps://github.com/SolidCode/SolidPython/issues/169\n\nThe goal was to\n\n* extract the "core" from SolidPython\n* make a solid package that only contains the fundamentals (+ a few convenience features) \n* make it extendible\n* try to get complex libraries working properly (mcad, bosl, bosl2)\n* **KISS**: ``from solid2 import *`` -> imports only ~1000 lines of source code and has (almost?) all the feautres SolidPython 1.x.x has\n* be a drop in replacement for SolidPython 1.x.x -- as far as possible, see Backwards Compatibility Section\n* get all kinds of nice features working (see Features section)\n\nThe result is a refactored and in some parts rewritten version of SolidPython we would like to release as SolidPython 2.x.x. The major improvement is a code base that should be better maintainable and extendible.\n\nBesides these benefits SolidPython 2.x.x implemented quite a few nice new features (cf. Features section).\n\nFeatures\n--------\n\nSolidPython 2.x.x has support for the following new features:\n\n* **bosl2** - SolidPython is now able to handle bosl2 pretty well (don\'t know whether everything works, but quite a lot). `bosl2 example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/07-libs-bosl2.py>`_ and `mazebox example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/16-mazebox-bosl2.py>`_\n* native **OpenSCAD customizer** support `customizer example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/10-customizer.py>`_ and `greedy scad interface example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/17-greedy-scad-interface.py>`_\n* native **OpenSCAD animation** support `animation example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/12-animation.py>`_ and `animation example 2 <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/13-animated-bouncing-ball.py>`_\n* **custom fonts** `fonts example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/11-fonts.py>`_\n* supports **ImplicitCAD** `implicitCAD example <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/14-implicitCAD.py>`_ and `implicitCAD example 2 <https://github.com/jeff-dh/SolidPython/blob/exp_solid/solid2/examples/15-implicitCAD2.py>`_\n\nFurthermore it has several minor improvements, like these which are based on ideas from *posts* from the SolidPython universe:\n\n* use invert operator (~) as # in OpenSCAD `#167 <https://github.com/SolidCode/SolidPython/pull/167>`_\n* convenience function including to pass sizes as integer parameters (``translate(10, 20, 30)``) `#63 <https://github.com/SolidCode/SolidPython/pull/63#issuecomment-688171416>`_\n* *access-style* syntax: ``cube(1).up(5).rotate(45, 0, 0)`` `#66 <https://github.com/SolidCode/SolidPython/pull/66>`_ This is additional! The OpenSCAD / SolidPython style syntax is still fully supported.\n\nAnother nice little feature especially to play around and debug it is that the ``__repr__`` operator of each "OpenSCADObject" now calls ``scad_render``. With this the python shell becomes pretty good in debuging and playing around with solid code and the library itself:\n\n.. code:: python\n\n  >>> from solid2 import *\n  >>> c = cube(5)\n  >>> c.up(5)\n  translate(v = [0, 0, 5]) {\n          cube(size = 5);\n  };\n  >>> c.up(5).save_as_scad()\n  \'/home/xxx/xxx/xxx/SolidPython/expsolid_out.scad\'\n  >>>\n\nBackwards compatibility\n-----------------------\n\nSolidPython 2.x.x should be a complete and mostly backwards compatible drop in\nreplacement for SolidPython 1.x.x.\nThe backwards compatibility is not 100% as depicted by the version number.\nSomethings (and even interfaces) changed. We tried to stay as backward\ncompatible as possible.  The package should behave 98% the same as SolidPython\nunless you do some "deep access" -- that\'s by 99% chance not backwards\ncompatible (like modifying OpenSCADObjects or import internal modules).\n\nAs long as you stick to:\n\n.. code:: python\n\n  from solid2 import *\n\nyou shoul be fine.\n\n**solid.utils**\n\n``solid.utils`` consisted of convenience functions and "modelling extensions" (kind of a small third party library like `mcad, bosl, bosl2`).\nThe convenience functions are now -- or the missing ones are supposed to be -- part of `solid2.extensions.convenience` and are automatically importet with the main package.\n\nConcerning the "modelling extensions" I would actually like to get rid of them as part of the SolidPython 2.x.x package. The resons are the following:\n\n* these modelling extensions (like `extrude_along_path, splines, screw_threads, part_hole,...`) don\'t align with the (core) purpose of SolidPython as I understand it (I think SolidPython is supposed to be a python "wrapper" / interface for OpenSCAD)\n* these modelling extensions are "yet another implementation" of common modelling task that need to be maintained. I would prefere a SolidPython design where these features are outsourced into a third party library\n* SolidPython 2.x.x has a pretty good **bosl2** support and bosl2 has all (?) the features provided by `solid.utils`:\n\n  * extrude_along_path: https://github.com/revarbat/BOSL2/wiki/mutators.scad#module-path_extrude\n  * First-class Negative Space (Holes): https://github.com/revarbat/BOSL2/wiki/attachments.scad#module-diff\n  * Splines / Bezier: https://github.com/revarbat/BOSL2/wiki/beziers.scad\n  * Screw threads: https://github.com/revarbat/BOSL2/wiki/screws.scad https://github.com/revarbat/BOSL2/wiki/metric_screws.scad https://github.com/revarbat/BOSL2/wiki/threading.scad\n  * distributors: https://github.com/revarbat/BOSL2/wiki/distributors.scad\n  * bouding boxes: https://github.com/revarbat/BOSL2/wiki/mutators.scad#module-bounding_box\n  * arcs, pie slices, tubes, ...: https://github.com/revarbat/BOSL2/wiki/shapes3d.scad https://github.com/revarbat/BOSL2/wiki/drawing.scad\n  * cut models in "half" / by a plane: https://github.com/revarbat/BOSL2/wiki/mutators.scad#functionmodule-half_of\n  * attachments: https://github.com/revarbat/BOSL2/wiki/attachments.scad\n\nAnd a looooot more.....\n\nI don\'t see why SolidPython should implement and maintain its own set of these features. Furthermore I assume a third party library (like `bosl2`) is probably able to provide more sophisticated implementations than we will ever be able to provide.\n\nPlease take a look at the `bosl2` implementations. I did some very basic tests in ``examples/07-libs-bosl2.py`` and -- at least -- was able to create basic examples for the core `solid.utils` features using bosl2.\n\nI would also be fine with a python third party library that implements these features, but I would like to seperate it from SolidPython itself. The reason is to achieve a SolidPython module which is independent from it (development, bugs, maintainance) with the goal to get an as solid and stable as possible SolidPython (core) package.\n\nBUT, since I assume quite a few people out there are using `solid.utils` up until now and simply getting rid of it might cause some brouhaha, my suggestion for a compromise is the `solid_legay` extension.\n\n**solid2_legacy**\n\nThe `solid2_legacy` extension is basicly everything that used to be `solid.utils`. Furhtermore it tries to "mimic" the SolidPython 1.x.x interface. This is the effort to become as backward compatible as possible. This might for example be useful when trying to get existing SolidPython 1.x.x code running.\n\nThe `solid2_legacy` extension got extracted into a seperate repo (and pip package). You should be able to just import the package if it is installed or somewhere in your import path.\n\nIf you want to use those features import the extension and take a look at it.\n\n.. code:: python\n\n  from solid2_legacy import *\n\nAnyway SolidPython 1.x.x `imports` do not work with SolidPython 2.x.x! (see Interface changes - imoprt paths have changed)\n\nI was able to get the SolidPython 1.x.x examples running just by changing the imports and they all (except for the splines example which seems to have an internal issue) worked "out of the box".\n\n\n**Interface changes**\n\n* OpenSCAD identifier escaping:\n        * all *illegal* python idetifiers are escape with a single prepending underscore\n        * special variables ``$fn -> _fn`` (*note*: ``segments`` still works)\n        * identifier starting with a digit ``module 12ptStar() -> _12ptStar()`` (*note*: ``__12ptStar`` still works)\n        * python keywords ``module import() -> _import()`` (*note*: ``import\\_``  still works)\n\n* import paths have changed (a lot)\n    * as long as you only import the root package it should be fine, otherwise probably not\n    \n    .. code:: python\n    \n            from solid2 import * #fine\n            from solid2 import objects #crash\n            from solid2 import solidpython #crash\n            from solid2 import splines #crash\n            from solid2 import utils #crash\n\n* all extensions have been moved:\n    * solid.utils has been moved to ``solid2_legacy``. If you want to use them import that extension\n    * there are some example implementations of the part / hole feature and\n      bill of materials in ``solid2_legacy``. They seem to work but are\n      not tested extensively. Take a look at ``examples/xx_legacy*``.\n    * please take a look at the bosl2 example. BOSL2 provides many features which\n      might be alternatives.\n\n* OpenSCADObject internally changed a lot\n    If you access it directly\n    (e.g. mycube.set_modifier) this might not work. But if you import\n    ``solid2_legacy`` some dummy methods will be monkey patched onto\n    OpenSCADObject so you might be able to at least run the code, but it\n    might render not correctly.\n\n* maybe some more things I can\'t remember. Some function signatures changed\n  slightly. But as long as as you stick to the regular public interface\n  everything should be fine.\n\n\nContact\n=======\n\nEnjoy!\n\nIf you have any questions or bug reports please report them to the SolidPython\n`GitHub page <https://github.com/jeff-dh/SolidPython>`__!\n\n\n\nCheers!\n\nLicense\n=======\n\nThis library is free software; you can redistribute it and/or modify it\nunder the terms of the GNU Lesser General Public License as published by\nthe Free Software Foundation; either version 2.1 of the License, or (at\nyour option) any later version.\n\nThis library is distributed in the hope that it will be useful, but\nWITHOUT ANY WARRANTY; without even the implied warranty of\nMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser\nGeneral Public License for more details.\n\n`Full text of the\nlicense <http://www.gnu.org/licenses/old-licenses/lgpl-2.1.txt>`__.\n\nSome class docstrings are derived from the `OpenSCAD User Manual\n<https://en.wikibooks.org/wiki/OpenSCAD_User_Manual>`__, so \nare available under the `Creative Commons Attribution-ShareAlike License\n<https://creativecommons.org/licenses/by-sa/3.0/>`__. \n\n',
    'author': 'jeff',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/jeff-dh/SolidPython',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7',
}


setup(**setup_kwargs)
