"""
Spec-Kit bridge adapter for markdown artifact synchronization.

This adapter implements the BridgeAdapter interface to sync Spec-Kit markdown artifacts
(spec.md, plan.md, tasks.md, constitution.md) with SpecFact plan bundles and protocols.
"""

from __future__ import annotations

import hashlib
import re
from pathlib import Path
from typing import Any

from beartype import beartype
from icontract import ensure, require

from specfact_cli.adapters.base import BridgeAdapter
from specfact_cli.importers.speckit_converter import SpecKitConverter
from specfact_cli.importers.speckit_scanner import SpecKitScanner
from specfact_cli.models.bridge import BridgeConfig
from specfact_cli.models.capabilities import ToolCapabilities
from specfact_cli.models.change import ChangeProposal, ChangeTracking


class SpecKitAdapter(BridgeAdapter):
    """
    Spec-Kit bridge adapter implementing BridgeAdapter interface.

    This adapter provides bidirectional sync between Spec-Kit markdown artifacts
    (generated by Spec-Kit slash commands) and SpecFact plan bundles/protocols.
    """

    @beartype
    @ensure(lambda result: result is None, "Must return None")
    def __init__(self) -> None:
        """Initialize Spec-Kit adapter."""
        self.hash_store: dict[str, str] = {}

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @require(lambda repo_path: repo_path.is_dir(), "Repository path must be a directory")
    @ensure(lambda result: isinstance(result, bool), "Must return bool")
    def detect(self, repo_path: Path, bridge_config: BridgeConfig | None = None) -> bool:
        """
        Detect if this is a Spec-Kit repository.

        Args:
            repo_path: Path to repository root
            bridge_config: Optional bridge configuration (for cross-repo detection)

        Returns:
            True if Spec-Kit structure detected, False otherwise
        """
        # Check for cross-repo Spec-Kit
        base_path = repo_path
        if bridge_config and bridge_config.external_base_path:
            base_path = bridge_config.external_base_path

        # Check for Spec-Kit structure
        # Priority: .specify/specs/ (canonical) > docs/specs/ > specs/ (root, legacy)
        specify_dir = base_path / ".specify"
        specify_specs_dir = base_path / ".specify" / "specs"
        specs_dir = base_path / "specs"
        docs_specs_dir = base_path / "docs" / "specs"

        return (
            (specify_specs_dir.exists() and specify_specs_dir.is_dir())
            or (specify_dir.exists() and specify_dir.is_dir())
            or (specs_dir.exists() and specs_dir.is_dir())
            or (docs_specs_dir.exists() and docs_specs_dir.is_dir())
        )

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @require(lambda repo_path: repo_path.is_dir(), "Repository path must be a directory")
    @ensure(lambda result: isinstance(result, ToolCapabilities), "Must return ToolCapabilities")
    def get_capabilities(self, repo_path: Path, bridge_config: BridgeConfig | None = None) -> ToolCapabilities:
        """
        Get Spec-Kit adapter capabilities.

        Args:
            repo_path: Path to repository root
            bridge_config: Optional bridge configuration (for cross-repo detection)

        Returns:
            ToolCapabilities instance for Spec-Kit adapter
        """
        base_path = repo_path
        if bridge_config and bridge_config.external_base_path:
            base_path = bridge_config.external_base_path

        # Determine layout (classic vs modern)
        specify_dir = base_path / ".specify"
        docs_specs_dir = base_path / "docs" / "specs"

        if docs_specs_dir.exists():
            layout = "modern"
            specs_dir_path = "docs/specs"
        elif specify_dir.exists():
            layout = "modern"
            specs_dir_path = "specs"
        else:
            layout = "classic"
            specs_dir_path = "specs"

        # Check for constitution file (set has_custom_hooks flag)
        scanner = SpecKitScanner(base_path)
        has_constitution, _ = scanner.has_constitution()
        has_custom_hooks = has_constitution

        return ToolCapabilities(
            tool="speckit",
            version=None,  # Spec-Kit version not tracked in files
            layout=layout,
            specs_dir=specs_dir_path,
            has_external_config=bridge_config is not None and bridge_config.external_base_path is not None,
            has_custom_hooks=has_custom_hooks,
            supported_sync_modes=["bidirectional", "unidirectional"],  # Spec-Kit supports bidirectional sync
        )

    @beartype
    @require(
        lambda artifact_key: isinstance(artifact_key, str) and len(artifact_key) > 0, "Artifact key must be non-empty"
    )
    @ensure(lambda result: result is None, "Must return None")
    def import_artifact(
        self,
        artifact_key: str,
        artifact_path: Path | dict[str, Any],
        project_bundle: Any,  # ProjectBundle - avoid circular import
        bridge_config: BridgeConfig | None = None,
    ) -> None:
        """
        Import artifact from Spec-Kit format to SpecFact.

        Args:
            artifact_key: Artifact key (e.g., "specification", "plan", "tasks")
            artifact_path: Path to artifact file
            project_bundle: Project bundle to update
            bridge_config: Bridge configuration (may contain external_base_path)
        """
        if not isinstance(artifact_path, Path):
            msg = f"Spec-Kit adapter requires Path, got {type(artifact_path)}"
            raise ValueError(msg)

        # Determine base path for cross-repo support
        base_path = artifact_path.parent.parent.parent if bridge_config and bridge_config.external_base_path else None
        if base_path is None:
            base_path = artifact_path.parent.parent.parent  # Navigate from specs/{feature}/spec.md

        scanner = SpecKitScanner(base_path)
        converter = SpecKitConverter(base_path)

        # Parse based on artifact key
        if artifact_key == "specification":
            self._import_specification(artifact_path, project_bundle, scanner, converter, bridge_config)
        elif artifact_key == "plan":
            self._import_plan(artifact_path, project_bundle, scanner, converter, bridge_config)
        elif artifact_key == "tasks":
            self._import_tasks(artifact_path, project_bundle, scanner, converter, bridge_config)
        else:
            msg = f"Unsupported artifact key: {artifact_key}"
            raise ValueError(msg)

    @beartype
    @require(
        lambda artifact_key: isinstance(artifact_key, str) and len(artifact_key) > 0, "Artifact key must be non-empty"
    )
    @ensure(lambda result: isinstance(result, (Path, dict)), "Must return Path or dict")
    def export_artifact(
        self,
        artifact_key: str,
        artifact_data: Any,  # Feature, ChangeProposal, etc. - avoid circular import
        bridge_config: BridgeConfig | None = None,
    ) -> Path | dict[str, Any]:
        """
        Export artifact from SpecFact to Spec-Kit format.

        Args:
            artifact_key: Artifact key (e.g., "specification", "plan", "tasks")
            artifact_data: Data to export (Feature, etc.)
            bridge_config: Bridge configuration (may contain external_base_path)

        Returns:
            Path to exported file
        """
        # Determine base path for cross-repo support
        base_path = None
        if bridge_config and bridge_config.external_base_path:
            base_path = bridge_config.external_base_path
        else:
            # For export, use current directory as base (will be overridden by resolve_path if needed)
            base_path = Path.cwd()

        converter = SpecKitConverter(base_path)

        # Export based on artifact key
        if artifact_key == "specification":
            return self._export_specification(artifact_data, converter, bridge_config)
        if artifact_key == "plan":
            return self._export_plan(artifact_data, converter, bridge_config)
        if artifact_key == "tasks":
            return self._export_tasks(artifact_data, converter, bridge_config)
        msg = f"Unsupported artifact key: {artifact_key}"
        raise ValueError(msg)

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @require(lambda repo_path: repo_path.is_dir(), "Repository path must be a directory")
    @ensure(lambda result: isinstance(result, BridgeConfig), "Must return BridgeConfig")
    def generate_bridge_config(self, repo_path: Path) -> BridgeConfig:
        """
        Generate bridge configuration for Spec-Kit.

        Auto-detects format with priority:
        1. .specify/specs/ (canonical - recommended by Spec-Kit)
        2. docs/specs/ (modern layout)
        3. specs/ at root (classic layout - backward compatibility)

        Args:
            repo_path: Path to repository root

        Returns:
            BridgeConfig instance for Spec-Kit
        """
        # Auto-detect format based on actual specs location
        # Priority order: .specify/specs/ > docs/specs/ > specs/ (root)
        # According to Spec-Kit documentation, .specify/specs/ is the canonical location
        specify_specs_dir = repo_path / ".specify" / "specs"
        docs_specs_dir = repo_path / "docs" / "specs"
        classic_specs_dir = repo_path / "specs"

        # Check actual specs location (prioritize .specify/specs/)
        if specify_specs_dir.exists() and specify_specs_dir.is_dir():
            # Canonical format: specs in .specify/specs/ (use modern preset with updated paths)
            return BridgeConfig.preset_speckit_specify()
        if docs_specs_dir.exists() and docs_specs_dir.is_dir():
            # Modern format: specs in docs/specs/
            return BridgeConfig.preset_speckit_modern()
        if classic_specs_dir.exists() and classic_specs_dir.is_dir():
            # Classic format: specs at root (backward compatibility)
            return BridgeConfig.preset_speckit_classic()
        # Default to specify format if .specify exists (most common)
        if (repo_path / ".specify").exists():
            return BridgeConfig.preset_speckit_specify()
        # Otherwise default to classic (will fail later if wrong)
        return BridgeConfig.preset_speckit_classic()

    @beartype
    @require(lambda bundle_dir: isinstance(bundle_dir, Path), "Bundle directory must be Path")
    @require(lambda bundle_dir: bundle_dir.exists(), "Bundle directory must exist")
    @ensure(lambda result: result is None or isinstance(result, ChangeTracking), "Must return ChangeTracking or None")
    def load_change_tracking(
        self, bundle_dir: Path, bridge_config: BridgeConfig | None = None
    ) -> ChangeTracking | None:
        """
        Load change tracking (Spec-Kit doesn't have change tracking).

        Args:
            bundle_dir: Path to bundle directory
            bridge_config: Optional bridge configuration

        Returns:
            None (Spec-Kit doesn't support change tracking)
        """
        return None

    @beartype
    @require(lambda bundle_dir: isinstance(bundle_dir, Path), "Bundle directory must be Path")
    @require(lambda bundle_dir: bundle_dir.exists(), "Bundle directory must exist")
    @require(
        lambda change_tracking: isinstance(change_tracking, ChangeTracking), "Change tracking must be ChangeTracking"
    )
    @ensure(lambda result: result is None, "Must return None")
    def save_change_tracking(
        self, bundle_dir: Path, change_tracking: ChangeTracking, bridge_config: BridgeConfig | None = None
    ) -> None:
        """
        Save change tracking (Spec-Kit doesn't support change tracking).

        Args:
            bundle_dir: Path to bundle directory
            change_tracking: ChangeTracking instance to save
            bridge_config: Optional bridge configuration

        Raises:
            NotImplementedError: Spec-Kit doesn't support change tracking
        """
        msg = "Spec-Kit adapter does not support change tracking"
        raise NotImplementedError(msg)

    @beartype
    @require(lambda bundle_dir: isinstance(bundle_dir, Path), "Bundle directory must be Path")
    @require(lambda bundle_dir: bundle_dir.exists(), "Bundle directory must exist")
    @require(lambda change_name: isinstance(change_name, str) and len(change_name) > 0, "Change name must be non-empty")
    @ensure(lambda result: result is None or isinstance(result, ChangeProposal), "Must return ChangeProposal or None")
    def load_change_proposal(
        self, bundle_dir: Path, change_name: str, bridge_config: BridgeConfig | None = None
    ) -> ChangeProposal | None:
        """
        Load change proposal (Spec-Kit doesn't have change proposals).

        Args:
            bundle_dir: Path to bundle directory
            change_name: Change identifier
            bridge_config: Optional bridge configuration

        Returns:
            None (Spec-Kit doesn't support change proposals)
        """
        return None

    @beartype
    @require(lambda bundle_dir: isinstance(bundle_dir, Path), "Bundle directory must be Path")
    @require(lambda bundle_dir: bundle_dir.exists(), "Bundle directory must exist")
    @require(lambda proposal: isinstance(proposal, ChangeProposal), "Proposal must be ChangeProposal")
    @ensure(lambda result: result is None, "Must return None")
    def save_change_proposal(
        self, bundle_dir: Path, proposal: ChangeProposal, bridge_config: BridgeConfig | None = None
    ) -> None:
        """
        Save change proposal (Spec-Kit doesn't support change proposals).

        Args:
            bundle_dir: Path to bundle directory
            proposal: ChangeProposal instance to save
            bridge_config: Optional bridge configuration

        Raises:
            NotImplementedError: Spec-Kit doesn't support change proposals
        """
        msg = "Spec-Kit adapter does not support change proposals"
        raise NotImplementedError(msg)

    # Private helper methods for import/export

    def _import_specification(
        self,
        spec_path: Path,
        project_bundle: Any,  # ProjectBundle
        scanner: SpecKitScanner,
        converter: SpecKitConverter,
        bridge_config: BridgeConfig | None,
    ) -> None:
        """Import specification from Spec-Kit spec.md."""
        from specfact_cli.models.plan import Feature, Story
        from specfact_cli.models.source_tracking import SourceTracking
        from specfact_cli.utils.feature_keys import normalize_feature_key

        # Parse spec.md
        spec_data = scanner.parse_spec_markdown(spec_path)
        if not spec_data:
            return

        # Extract feature information
        feature_key = spec_data.get("feature_key", spec_path.parent.name.upper().replace("-", "_"))
        feature_title = spec_data.get("feature_title")
        # If feature_title not found, try to extract from first H1 header in spec.md
        if not feature_title:
            try:
                content = spec_path.read_text(encoding="utf-8")
                # Try multiple patterns: "Feature Specification: Title", "# Title", etc.
                title_match = (
                    re.search(r"^#\s+Feature Specification:\s*(.+)$", content, re.MULTILINE)
                    or re.search(r"^#\s+(.+?)\s+Feature", content, re.MULTILINE)
                    or re.search(r"^#\s+(.+)$", content, re.MULTILINE)
                )
                if title_match:
                    feature_title = title_match.group(1).strip()
            except Exception:
                pass
        # Ensure feature_title is never None (Pydantic validation requirement)
        if not feature_title or feature_title.strip() == "":
            feature_title = "Unknown Feature"

        # Extract stories
        stories: list[Story] = []
        spec_stories = spec_data.get("stories", [])
        for story_data in spec_stories:
            story_key = story_data.get("key", "UNKNOWN")
            story_title = story_data.get("title", "Unknown Story")
            priority = story_data.get("priority", "P3")
            acceptance = story_data.get("acceptance", [])

            # Calculate story points from priority
            priority_map = {"P1": 8, "P2": 5, "P3": 3, "P4": 1}
            story_points = priority_map.get(priority, 3)

            story = Story(
                key=story_key,
                title=story_title,
                acceptance=acceptance if acceptance else [f"{story_title} is implemented"],
                tags=[priority],
                story_points=story_points,
                value_points=story_points,
                tasks=[],
                confidence=0.8,
                draft=False,
                scenarios=story_data.get("scenarios"),
                contracts=None,
            )
            stories.append(story)

        # Extract outcomes from requirements
        requirements = spec_data.get("requirements", [])
        outcomes: list[str] = []
        for req in requirements:
            if isinstance(req, dict):
                outcomes.append(req.get("text", ""))
            elif isinstance(req, str):
                outcomes.append(req)

        # Extract acceptance criteria from success criteria
        success_criteria = spec_data.get("success_criteria", [])
        acceptance: list[str] = []
        for sc in success_criteria:
            if isinstance(sc, dict):
                acceptance.append(sc.get("text", ""))
            elif isinstance(sc, str):
                acceptance.append(sc)

        # Create or update feature
        if not hasattr(project_bundle, "features") or project_bundle.features is None:
            project_bundle.features = {}

        # Normalize key for matching
        normalized_key = normalize_feature_key(feature_key)
        existing_feature = None
        if isinstance(project_bundle.features, dict):
            # Try to find existing feature by normalized key
            for key, feat in project_bundle.features.items():
                if normalize_feature_key(key) == normalized_key:
                    existing_feature = feat
                    break

        if existing_feature:
            # Update existing feature
            existing_feature.title = feature_title
            existing_feature.outcomes = outcomes if outcomes else existing_feature.outcomes
            existing_feature.acceptance = acceptance if acceptance else existing_feature.acceptance
            existing_feature.stories = stories
            existing_feature.constraints = spec_data.get("edge_cases", [])
        else:
            # Create new feature
            feature = Feature(
                key=feature_key,
                title=feature_title,
                outcomes=outcomes if outcomes else [f"Provides {feature_title} functionality"],
                acceptance=acceptance if acceptance else [f"{feature_title} is functional"],
                constraints=spec_data.get("edge_cases", []),
                stories=stories,
                confidence=0.8,
                draft=False,
                source_tracking=None,
                contract=None,
                protocol=None,
            )

            # Store Spec-Kit path in source_tracking
            base_path = spec_path.parent.parent.parent if bridge_config and bridge_config.external_base_path else None
            if base_path is None:
                base_path = spec_path.parent.parent.parent

            speckit_path = (
                str(spec_path.relative_to(base_path)) if base_path else f"specs/{spec_path.parent.name}/spec.md"
            )
            source_metadata = {
                "path": speckit_path,
                "speckit_path": speckit_path,
                "speckit_type": "specification",
            }
            if bridge_config and bridge_config.external_base_path:
                source_metadata["speckit_base_path"] = str(bridge_config.external_base_path)

            feature.source_tracking = SourceTracking(tool="speckit", source_metadata=source_metadata)

            if isinstance(project_bundle.features, dict):
                project_bundle.features[feature_key] = feature
            else:
                if project_bundle.features is None:
                    project_bundle.features = {}
                if isinstance(project_bundle.features, dict):
                    project_bundle.features[feature_key] = feature

    @beartype
    @require(lambda plan_path: plan_path.exists(), "Plan path must exist")
    @require(lambda plan_path: plan_path.is_file(), "Plan path must be a file")
    @require(lambda project_bundle: project_bundle is not None, "Project bundle must not be None")
    @ensure(lambda result: result is None, "Must return None")
    def _import_plan(
        self,
        plan_path: Path,
        project_bundle: Any,  # ProjectBundle
        scanner: SpecKitScanner,
        converter: SpecKitConverter,
        bridge_config: BridgeConfig | None,
    ) -> None:
        """Import plan from Spec-Kit plan.md."""
        from specfact_cli.models.plan import Feature
        from specfact_cli.models.source_tracking import SourceTracking
        from specfact_cli.utils.feature_keys import normalize_feature_key

        # Parse plan.md
        plan_data = scanner.parse_plan_markdown(plan_path)
        if not plan_data:
            return

        # Extract feature ID from path (specs/{feature_id}/plan.md)
        feature_id = plan_path.parent.name
        normalized_feature_id = normalize_feature_key(feature_id)

        # Find or create feature in bundle
        if not hasattr(project_bundle, "features") or project_bundle.features is None:
            project_bundle.features = {}

        matching_feature = None
        if isinstance(project_bundle.features, dict):
            for key, feat in project_bundle.features.items():
                if normalize_feature_key(key) == normalized_feature_id:
                    matching_feature = feat
                    break

        # If feature doesn't exist, create minimal feature from plan
        if not matching_feature:
            feature_key = feature_id.upper().replace("-", "_")
            # Try to extract title from plan.md first line
            try:
                content = plan_path.read_text(encoding="utf-8")
                title_match = re.search(r"^#\s+(.+)$", content, re.MULTILINE)
                feature_title = title_match.group(1).strip() if title_match else "Unknown Feature"
            except Exception:
                feature_title = "Unknown Feature"

            matching_feature = Feature(
                key=feature_key,
                title=feature_title,
                outcomes=[],
                acceptance=[],
                constraints=[],
                stories=[],
                confidence=0.8,
                draft=False,
                source_tracking=None,
                contract=None,
                protocol=None,
            )

            # Store Spec-Kit path in source_tracking
            base_path = plan_path.parent.parent.parent if bridge_config and bridge_config.external_base_path else None
            if base_path is None:
                base_path = plan_path.parent.parent.parent

            speckit_path = (
                str(plan_path.relative_to(base_path)) if base_path else f"specs/{plan_path.parent.name}/plan.md"
            )
            source_metadata = {
                "path": speckit_path,
                "speckit_path": speckit_path,
                "speckit_type": "plan",
            }
            if bridge_config and bridge_config.external_base_path:
                source_metadata["speckit_base_path"] = str(bridge_config.external_base_path)

            matching_feature.source_tracking = SourceTracking(tool="speckit", source_metadata=source_metadata)

            if isinstance(project_bundle.features, dict):
                project_bundle.features[feature_key] = matching_feature

    @beartype
    @require(lambda tasks_path: tasks_path.exists(), "Tasks path must exist")
    @require(lambda tasks_path: tasks_path.is_file(), "Tasks path must be a file")
    @require(lambda project_bundle: project_bundle is not None, "Project bundle must not be None")
    @ensure(lambda result: result is None, "Must return None")
    def _import_tasks(
        self,
        tasks_path: Path,
        project_bundle: Any,  # ProjectBundle
        scanner: SpecKitScanner,
        converter: SpecKitConverter,
        bridge_config: BridgeConfig | None,
    ) -> None:
        """Import tasks from Spec-Kit tasks.md."""
        from specfact_cli.utils.feature_keys import normalize_feature_key

        # Parse tasks.md
        tasks_data = scanner.parse_tasks_markdown(tasks_path)
        if not tasks_data:
            return

        # Extract feature ID from path (specs/{feature_id}/tasks.md)
        feature_id = tasks_path.parent.name
        normalized_feature_id = normalize_feature_key(feature_id)

        # Find matching feature in bundle
        if hasattr(project_bundle, "features") and project_bundle.features:
            matching_feature = None
            if isinstance(project_bundle.features, dict):
                for key, feat in project_bundle.features.items():
                    if normalize_feature_key(key) == normalized_feature_id:
                        matching_feature = feat
                        break

            if matching_feature and hasattr(matching_feature, "stories"):
                # Map tasks to stories based on story_ref
                tasks = tasks_data.get("tasks", [])
                for task in tasks:
                    story_ref = task.get("story_ref", "")
                    task_desc = task.get("description", "")

                    # Find matching story
                    for story in matching_feature.stories:
                        if story_ref and story_ref in story.key:
                            if not story.tasks:
                                story.tasks = []
                            story.tasks.append(task_desc)
                            break

    @beartype
    @require(lambda feature: feature is not None, "Feature must not be None")
    @ensure(lambda result: isinstance(result, Path), "Must return Path")
    def _export_specification(
        self,
        feature: Any,
        converter: SpecKitConverter,
        bridge_config: BridgeConfig | None,  # Feature
    ) -> Path:
        """Export specification to Spec-Kit spec.md."""
        # Use converter to convert Feature to Spec-Kit spec.md format
        # Implementation details would go here
        msg = "Spec-Kit adapter export_specification not yet fully implemented"
        raise NotImplementedError(msg)

    def _export_plan(
        self,
        plan_data: Any,
        converter: SpecKitConverter,
        bridge_config: BridgeConfig | None,  # PlanBundle or Feature
    ) -> Path:
        """Export plan to Spec-Kit plan.md."""
        from specfact_cli.models.plan import Feature, PlanBundle

        # Determine base path
        base_path = converter.repo_path
        if bridge_config and bridge_config.external_base_path:
            base_path = bridge_config.external_base_path

        # If plan_data is a Feature, we need to get the bundle context
        # For now, we'll generate plan.md for the feature
        if isinstance(plan_data, Feature):
            # Generate plan.md for a single feature (create minimal bundle for context)
            from specfact_cli.models.plan import PlanBundle, Product, Release

            # Create minimal bundle with features as list
            minimal_bundle = PlanBundle(
                version="1.0",
                idea=None,
                business=None,
                product=Product(
                    themes=[], releases=[Release(name="v0.1", objectives=[], scope=[plan_data.key], risks=[])]
                ),
                features=[plan_data],
                metadata=None,
                clarifications=None,
            )
            plan_content = converter._generate_plan_markdown(plan_data, minimal_bundle)
            feature_id = plan_data.key.lower().replace("_", "-")
        elif isinstance(plan_data, PlanBundle):
            # Generate plan.md for first feature (Spec-Kit has one plan.md per feature)
            if plan_data.features:
                feature = (
                    plan_data.features[0]
                    if isinstance(plan_data.features, list)
                    else next(iter(plan_data.features.values()))
                )
                plan_content = converter._generate_plan_markdown(feature, plan_data)
                feature_id = feature.key.lower().replace("_", "-")
            else:
                msg = "Plan bundle has no features to export"
                raise ValueError(msg)
        else:
            msg = f"Unsupported plan_data type: {type(plan_data)}"
            raise ValueError(msg)

        # Determine output path from bridge config or use default
        if bridge_config and "plan" in bridge_config.artifacts:
            artifact_path = bridge_config.resolve_path("plan", {"feature_id": feature_id}, base_path=base_path)
        else:
            # Default path
            artifact_path = base_path / "specs" / feature_id / "plan.md"

        # Ensure directory exists
        artifact_path.parent.mkdir(parents=True, exist_ok=True)

        # Write plan.md
        artifact_path.write_text(plan_content, encoding="utf-8")

        return artifact_path

    @beartype
    @require(lambda feature: feature is not None, "Feature must not be None")
    @ensure(lambda result: isinstance(result, Path), "Must return Path")
    def _export_tasks(
        self,
        feature: Any,
        converter: SpecKitConverter,
        bridge_config: BridgeConfig | None,  # Feature
    ) -> Path:
        """Export tasks to Spec-Kit tasks.md."""
        from specfact_cli.models.plan import Feature

        if not isinstance(feature, Feature):
            msg = f"Expected Feature, got {type(feature)}"
            raise ValueError(msg)

        # Determine base path
        base_path = converter.repo_path
        if bridge_config and bridge_config.external_base_path:
            base_path = bridge_config.external_base_path

        # Generate tasks.md content using converter
        tasks_content = converter._generate_tasks_markdown(feature)

        # Determine output path from bridge config or use default
        feature_id = feature.key.lower().replace("_", "-")
        if bridge_config and "tasks" in bridge_config.artifacts:
            artifact_path = bridge_config.resolve_path("tasks", {"feature_id": feature_id}, base_path=base_path)
        else:
            # Default path
            artifact_path = base_path / "specs" / feature_id / "tasks.md"

        # Ensure directory exists
        artifact_path.parent.mkdir(parents=True, exist_ok=True)

        # Write tasks.md
        artifact_path.write_text(tasks_content, encoding="utf-8")

        return artifact_path

    # Private helper methods for bidirectional sync (from SpecKitSync)

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @require(lambda repo_path: repo_path.is_dir(), "Repository path must be a directory")
    @ensure(lambda result: isinstance(result, dict), "Must return dict")
    def _detect_speckit_changes(self, repo_path: Path) -> dict[str, Any]:
        """
        Detect changes in Spec-Kit artifacts.

        Args:
            repo_path: Path to repository

        Returns:
            Dictionary of detected changes keyed by file path
        """
        changes: dict[str, Any] = {}

        # Check for modern Spec-Kit format (.specify directory)
        specify_dir = repo_path / ".specify"
        if specify_dir.exists():
            # Monitor .specify/memory/ files
            memory_dir = repo_path / ".specify" / "memory"
            if memory_dir.exists():
                for memory_file in memory_dir.glob("*.md"):
                    relative_path = str(memory_file.relative_to(repo_path))
                    current_hash = self._get_file_hash(memory_file)
                    stored_hash = self.hash_store.get(relative_path, "")

                    if current_hash != stored_hash:
                        changes[relative_path] = {
                            "file": memory_file,
                            "hash": current_hash,
                            "type": "modified" if stored_hash else "new",
                        }

        # Monitor specs/ directory for feature specifications
        # Check all possible layouts: .specify/specs/ (canonical) > docs/specs/ > specs/ (root)
        # Priority order matches generate_bridge_config() detection logic
        # Note: Check all layouts regardless of whether .specify exists (some repos may have specs without .specify)
        specify_specs_dir = repo_path / ".specify" / "specs"
        docs_specs_dir = repo_path / "docs" / "specs"
        classic_specs_dir = repo_path / "specs"

        # Check canonical .specify/specs/ first
        if specify_specs_dir.exists() and specify_specs_dir.is_dir():
            for spec_dir in specify_specs_dir.iterdir():
                if spec_dir.is_dir():
                    for spec_file in spec_dir.glob("*.md"):
                        relative_path = str(spec_file.relative_to(repo_path))
                        current_hash = self._get_file_hash(spec_file)
                        stored_hash = self.hash_store.get(relative_path, "")

                        if current_hash != stored_hash:
                            changes[relative_path] = {
                                "file": spec_file,
                                "hash": current_hash,
                                "type": "modified" if stored_hash else "new",
                            }
        # Check modern docs/specs/ layout
        elif docs_specs_dir.exists() and docs_specs_dir.is_dir():
            for spec_dir in docs_specs_dir.iterdir():
                if spec_dir.is_dir():
                    for spec_file in spec_dir.glob("*.md"):
                        relative_path = str(spec_file.relative_to(repo_path))
                        current_hash = self._get_file_hash(spec_file)
                        stored_hash = self.hash_store.get(relative_path, "")

                        if current_hash != stored_hash:
                            changes[relative_path] = {
                                "file": spec_file,
                                "hash": current_hash,
                                "type": "modified" if stored_hash else "new",
                            }
        # Check classic specs/ at root (backward compatibility)
        elif classic_specs_dir.exists() and classic_specs_dir.is_dir():
            for spec_dir in classic_specs_dir.iterdir():
                if spec_dir.is_dir():
                    for spec_file in spec_dir.glob("*.md"):
                        relative_path = str(spec_file.relative_to(repo_path))
                        current_hash = self._get_file_hash(spec_file)
                        stored_hash = self.hash_store.get(relative_path, "")

                        if current_hash != stored_hash:
                            changes[relative_path] = {
                                "file": spec_file,
                                "hash": current_hash,
                                "type": "modified" if stored_hash else "new",
                            }

        return changes

    def _detect_specfact_changes(self, repo_path: Path) -> dict[str, Any]:
        """
        Detect changes in SpecFact artifacts.

        Args:
            repo_path: Path to repository

        Returns:
            Dictionary of detected changes keyed by file path
        """
        changes: dict[str, Any] = {}

        # Monitor .specfact/plans/ files
        plans_dir = repo_path / ".specfact" / "plans"
        if plans_dir.exists():
            for plan_file in plans_dir.glob("*.yaml"):
                relative_path = str(plan_file.relative_to(repo_path))
                current_hash = self._get_file_hash(plan_file)
                stored_hash = self.hash_store.get(relative_path, "")

                if current_hash != stored_hash:
                    changes[relative_path] = {
                        "file": plan_file,
                        "hash": current_hash,
                        "type": "modified" if stored_hash else "new",
                    }

        # Monitor .specfact/protocols/ files
        protocols_dir = repo_path / ".specfact" / "protocols"
        if protocols_dir.exists():
            for protocol_file in protocols_dir.glob("*.yaml"):
                relative_path = str(protocol_file.relative_to(repo_path))
                current_hash = self._get_file_hash(protocol_file)
                stored_hash = self.hash_store.get(relative_path, "")

                if current_hash != stored_hash:
                    changes[relative_path] = {
                        "file": protocol_file,
                        "hash": current_hash,
                        "type": "modified" if stored_hash else "new",
                    }

        return changes

    @beartype
    @require(lambda speckit_changes: isinstance(speckit_changes, dict), "Speckit changes must be dict")
    @require(lambda specfact_changes: isinstance(specfact_changes, dict), "Specfact changes must be dict")
    @ensure(lambda result: isinstance(result, dict), "Must return dict")
    def _merge_changes(self, speckit_changes: dict[str, Any], specfact_changes: dict[str, Any]) -> dict[str, Any]:
        """
        Merge changes from both sources.

        Args:
            speckit_changes: Spec-Kit detected changes
            specfact_changes: SpecFact detected changes

        Returns:
            Merged changes dictionary
        """
        merged: dict[str, Any] = {}

        # Merge Spec-Kit changes
        for key, change in speckit_changes.items():
            merged[key] = {
                "source": "speckit",
                **change,
            }

        # Merge SpecFact changes
        for key, change in specfact_changes.items():
            if key in merged:
                # Conflict detected
                merged[key]["conflict"] = True
                merged[key]["specfact_change"] = change
            else:
                merged[key] = {
                    "source": "specfact",
                    **change,
                }

        return merged

    @beartype
    @require(lambda speckit_changes: isinstance(speckit_changes, dict), "Speckit changes must be dict")
    @require(lambda specfact_changes: isinstance(specfact_changes, dict), "Specfact changes must be dict")
    @ensure(lambda result: isinstance(result, list), "Must return list")
    def _detect_conflicts(
        self, speckit_changes: dict[str, Any], specfact_changes: dict[str, Any]
    ) -> list[dict[str, Any]]:
        """
        Detect conflicts between Spec-Kit and SpecFact changes.

        Args:
            speckit_changes: Spec-Kit detected changes
            specfact_changes: SpecFact detected changes

        Returns:
            List of conflict dictionaries
        """
        conflicts: list[dict[str, Any]] = []

        for key in set(speckit_changes.keys()) & set(specfact_changes.keys()):
            conflicts.append(
                {
                    "key": key,
                    "speckit_change": speckit_changes[key],
                    "specfact_change": specfact_changes[key],
                }
            )

        return conflicts

    def _resolve_conflicts(self, conflicts: list[dict[str, Any]]) -> dict[str, Any]:
        """
        Resolve conflicts with merge strategy.

        Strategy:
        - Priority: SpecFact > Spec-Kit for artifacts (specs/*)
        - Priority: Spec-Kit > SpecFact for memory files (.specify/memory/)

        Args:
            conflicts: List of conflict dictionaries

        Returns:
            Dictionary of resolved conflicts
        """
        resolved: dict[str, Any] = {}

        for conflict in conflicts:
            key = conflict["key"]
            if key.startswith(".specify/memory/"):
                # Memory files: Spec-Kit wins
                resolved[key] = "speckit"
            else:
                # Artifacts: SpecFact wins
                resolved[key] = "specfact"

        return resolved

    @beartype
    @require(lambda file_path: file_path.exists(), "File path must exist")
    @require(lambda file_path: file_path.is_file(), "File path must be a file")
    @ensure(lambda result: isinstance(result, str) and len(result) == 64, "Must return 64-char hex digest")
    def _get_file_hash(self, file_path: Path) -> str:
        """
        Get SHA256 hash of file content.

        Args:
            file_path: Path to file

        Returns:
            Hex digest of file hash
        """
        content = file_path.read_bytes()
        return hashlib.sha256(content).hexdigest()

    # Public helper methods for sync operations (used by sync.py)

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @require(lambda repo_path: repo_path.is_dir(), "Repository path must be a directory")
    def discover_features(self, repo_path: Path, bridge_config: BridgeConfig | None = None) -> list[dict[str, Any]]:
        """
        Discover features from Spec-Kit repository.

        This is a public helper method for sync operations to discover features
        without directly instantiating SpecKitScanner.

        Args:
            repo_path: Path to repository root
            bridge_config: Optional bridge configuration (for cross-repo support)

        Returns:
            List of feature dictionaries
        """
        base_path = repo_path
        if bridge_config and bridge_config.external_base_path:
            base_path = bridge_config.external_base_path

        scanner = SpecKitScanner(base_path)
        return scanner.discover_features()

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @require(lambda repo_path: repo_path.is_dir(), "Repository path must be a directory")
    @require(
        lambda direction: direction in ("speckit", "specfact", "both"),
        "Direction must be 'speckit', 'specfact', or 'both'",
    )
    @ensure(lambda result: isinstance(result, dict), "Must return dict")
    def detect_changes(
        self, repo_path: Path, direction: str = "both", bridge_config: BridgeConfig | None = None
    ) -> dict[str, Any]:
        """
        Detect changes in Spec-Kit or SpecFact artifacts.

        This is a public helper method for sync operations to detect changes
        without directly instantiating SpecKitSync.

        Args:
            repo_path: Path to repository root
            direction: Direction to detect changes ("speckit", "specfact", or "both")
            bridge_config: Optional bridge configuration (for cross-repo support)

        Returns:
            Dictionary with "speckit_changes" and/or "specfact_changes" keys
        """
        base_path = repo_path
        if bridge_config and bridge_config.external_base_path:
            base_path = bridge_config.external_base_path

        result: dict[str, Any] = {}
        if direction in ("speckit", "both"):
            result["speckit_changes"] = self._detect_speckit_changes(base_path)
        if direction in ("specfact", "both"):
            result["specfact_changes"] = self._detect_specfact_changes(base_path)
        return result

    @beartype
    @require(lambda speckit_changes: isinstance(speckit_changes, dict), "Speckit changes must be dict")
    @require(lambda specfact_changes: isinstance(specfact_changes, dict), "Specfact changes must be dict")
    @ensure(lambda result: isinstance(result, list), "Must return list")
    def detect_conflicts(
        self, speckit_changes: dict[str, Any], specfact_changes: dict[str, Any]
    ) -> list[dict[str, Any]]:
        """
        Detect conflicts between Spec-Kit and SpecFact changes.

        This is a public helper method for sync operations to detect conflicts.

        Args:
            speckit_changes: Spec-Kit detected changes
            specfact_changes: SpecFact detected changes

        Returns:
            List of conflict dictionaries
        """
        return self._detect_conflicts(speckit_changes, specfact_changes)

    @beartype
    @require(lambda plan_bundle: plan_bundle is not None, "Plan bundle must not be None")
    def export_bundle(
        self,
        plan_bundle: Any,  # PlanBundle - avoid circular import
        repo_path: Path,
        progress_callback: Any | None = None,  # Callable[[int, int], None] - avoid circular import
        bridge_config: BridgeConfig | None = None,
    ) -> int:
        """
        Export a full plan bundle to Spec-Kit format.

        This is a public helper method for sync operations to export entire bundles.
        It's a convenience method that wraps the bundle-level conversion operation.

        Args:
            plan_bundle: PlanBundle to export
            repo_path: Path to repository root
            progress_callback: Optional callback function(current: int, total: int) for progress updates
            bridge_config: Optional bridge configuration (for cross-repo support)

        Returns:
            Number of features converted
        """
        base_path = repo_path
        if bridge_config and bridge_config.external_base_path:
            base_path = bridge_config.external_base_path

        converter = SpecKitConverter(base_path)
        return converter.convert_to_speckit(plan_bundle, progress_callback)
