"""
Spec-Kit scanner for importing Spec-Kit projects.

This module provides functionality to scan Spec-Kit repositories,
parse their structure, and extract features, stories, and requirements
from markdown artifacts generated by Spec-Kit commands.

Spec-Kit uses slash commands (/speckit.specify, /speckit.plan, etc.) to
generate markdown artifacts in specs/ and .specify/ directories.
"""

from __future__ import annotations

import re
from pathlib import Path
from typing import Any

from beartype import beartype
from icontract import ensure, require


class SpecKitScanner:
    """
    Scanner for Spec-Kit repositories.

    Scans Spec-Kit directory structure, parses markdown files (spec.md, plan.md, tasks.md),
    and extracts features, user stories, requirements, and tasks.
    """

    # Spec-Kit directory structure
    SPECIFY_DIR = ".specify"
    SPECIFY_MEMORY_DIR = ".specify/memory"
    SPECS_DIR = "specs"

    @beartype
    def __init__(self, repo_path: Path) -> None:
        """
        Initialize Spec-Kit scanner.

        Args:
            repo_path: Path to Spec-Kit repository root
        """
        self.repo_path = Path(repo_path)

    @beartype
    @ensure(lambda result: isinstance(result, bool), "Must return boolean")
    def is_speckit_repo(self) -> bool:
        """
        Check if repository is a Spec-Kit project.

        Returns:
            True if Spec-Kit structure detected, False otherwise
        """
        # Check for Spec-Kit format (.specify directory)
        specify_dir = self.repo_path / self.SPECIFY_DIR
        return specify_dir.exists() and specify_dir.is_dir()

    @beartype
    @ensure(lambda result: isinstance(result, dict), "Must return dictionary")
    @ensure(lambda result: "is_speckit" in result, "Must include is_speckit key")
    def scan_structure(self) -> dict[str, Any]:
        """
        Scan Spec-Kit directory structure.

        Returns:
            Dictionary with detected structure information
        """
        structure = {
            "is_speckit": False,
            "specify_dir": None,
            "specify_memory_dir": None,
            "specs_dir": None,
            "spec_files": [],
            "feature_dirs": [],
            "memory_files": [],
        }

        if not self.is_speckit_repo():
            return structure

        structure["is_speckit"] = True

        # Check for .specify directory
        specify_dir = self.repo_path / self.SPECIFY_DIR
        if specify_dir.exists() and specify_dir.is_dir():
            structure["specify_dir"] = str(specify_dir)

            # Check for .specify/memory directory
            specify_memory_dir = self.repo_path / self.SPECIFY_MEMORY_DIR
            if specify_memory_dir.exists():
                structure["specify_memory_dir"] = str(specify_memory_dir)
                structure["memory_files"] = [str(f) for f in specify_memory_dir.glob("*.md")]

        # Check for specs directory
        specs_dir = self.repo_path / self.SPECS_DIR
        if specs_dir.exists():
            structure["specs_dir"] = str(specs_dir)
            # Find all feature directories (specs/*/)
            for spec_dir in specs_dir.iterdir():
                if spec_dir.is_dir():
                    structure["feature_dirs"].append(str(spec_dir))
                    # Find all markdown files in each feature directory
                    for md_file in spec_dir.glob("*.md"):
                        structure["spec_files"].append(str(md_file))
                    # Also check for contracts/*.yaml
                    contracts_dir = spec_dir / "contracts"
                    if contracts_dir.exists():
                        for yaml_file in contracts_dir.glob("*.yaml"):
                            structure["spec_files"].append(str(yaml_file))

        return structure

    @beartype
    @ensure(lambda result: isinstance(result, list), "Must return list")
    @ensure(lambda result: all(isinstance(f, dict) for f in result), "All items must be dictionaries")
    def discover_features(self) -> list[dict[str, Any]]:
        """
        Discover all features from specs directory.

        Returns:
            List of feature dictionaries with parsed data from spec.md, plan.md, tasks.md
        """
        features: list[dict[str, Any]] = []
        structure = self.scan_structure()

        if not structure["is_speckit"] or not structure["feature_dirs"]:
            return features

        for feature_dir_path in structure["feature_dirs"]:
            feature_dir = Path(feature_dir_path)
            spec_file = feature_dir / "spec.md"

            if spec_file.exists():
                spec_data = self.parse_spec_markdown(spec_file)
                if spec_data:
                    # Parse plan.md if it exists
                    plan_file = feature_dir / "plan.md"
                    if plan_file.exists():
                        plan_data = self.parse_plan_markdown(plan_file)
                        spec_data["plan"] = plan_data

                    # Parse tasks.md if it exists
                    tasks_file = feature_dir / "tasks.md"
                    if tasks_file.exists():
                        tasks_data = self.parse_tasks_markdown(tasks_file)
                        spec_data["tasks"] = tasks_data

                    features.append(spec_data)

        return features

    @beartype
    @require(lambda spec_file: spec_file is not None, "Spec file path must not be None")
    @require(lambda spec_file: spec_file.suffix == ".md", "Spec file must be markdown")
    @ensure(
        lambda result, spec_file: result is None or (isinstance(result, dict) and "feature_key" in result),
        "Must return None or dict with feature_key",
    )
    def parse_spec_markdown(self, spec_file: Path) -> dict[str, Any] | None:
        """
        Parse a Spec-Kit spec.md file to extract features, stories, requirements, and success criteria.

        Args:
            spec_file: Path to spec.md file

        Returns:
            Dictionary with extracted feature and story information, or None if file doesn't exist
        """
        if not spec_file.exists():
            return None

        try:
            content = spec_file.read_text(encoding="utf-8")
            spec_data: dict[str, Any] = {
                "feature_key": None,
                "feature_title": None,
                "feature_branch": None,
                "created_date": None,
                "status": None,
                "stories": [],
                "requirements": [],
                "success_criteria": [],
                "edge_cases": [],
            }

            # Extract frontmatter (if present)
            frontmatter_match = re.search(r"^---\n(.*?)\n---", content, re.MULTILINE | re.DOTALL)
            if frontmatter_match:
                frontmatter = frontmatter_match.group(1)
                # Extract Feature Branch
                branch_match = re.search(r"\*\*Feature Branch\*\*:\s*`(.+?)`", frontmatter)
                if branch_match:
                    spec_data["feature_branch"] = branch_match.group(1).strip()
                # Extract Created date
                created_match = re.search(r"\*\*Created\*\*:\s*(\d{4}-\d{2}-\d{2})", frontmatter)
                if created_match:
                    spec_data["created_date"] = created_match.group(1).strip()
                # Extract Status
                status_match = re.search(r"\*\*Status\*\*:\s*(.+?)(?:\n|$)", frontmatter)
                if status_match:
                    spec_data["status"] = status_match.group(1).strip()

            # Extract feature key from directory name (specs/001-feature-name/spec.md)
            spec_dir = spec_file.parent
            if spec_dir.name:
                spec_data["feature_key"] = spec_dir.name.upper().replace("-", "_")
                # If feature_branch not found in frontmatter, use directory name
                if not spec_data["feature_branch"]:
                    spec_data["feature_branch"] = spec_dir.name

            # Extract feature title from spec.md header
            title_match = re.search(r"^#\s+Feature Specification:\s*(.+)$", content, re.MULTILINE)
            if title_match:
                spec_data["feature_title"] = title_match.group(1).strip()

            # Extract user stories with full context
            story_pattern = r"###\s+User Story\s+(\d+)\s*-\s*(.+?)\s*\(Priority:\s*(P\d+)\)"
            stories = re.finditer(story_pattern, content, re.MULTILINE | re.DOTALL)

            story_counter = 1
            for story_match in stories:
                story_number = story_match.group(1)
                story_title = story_match.group(2).strip()
                priority = story_match.group(3)

                # Find story content (between this story and next story or end of section)
                story_start = story_match.end()
                next_story_match = re.search(r"###\s+User Story\s+\d+", content[story_start:], re.MULTILINE)
                story_end = story_start + next_story_match.start() if next_story_match else len(content)
                story_content = content[story_start:story_end]

                # Extract "As a..." description
                as_a_match = re.search(
                    r"As a (.+?), I want (.+?) so that (.+?)(?=\n\n|\*\*Why|\*\*Independent|\*\*Acceptance)",
                    story_content,
                    re.DOTALL,
                )
                as_a_text = ""
                if as_a_match:
                    as_a_text = f"As a {as_a_match.group(1)}, I want {as_a_match.group(2)}, so that {as_a_match.group(3)}".strip()

                # Extract "Why this priority" text
                why_priority_match = re.search(
                    r"\*\*Why this priority\*\*:\s*(.+?)(?=\n\n|\*\*Independent|$)", story_content, re.DOTALL
                )
                why_priority = why_priority_match.group(1).strip() if why_priority_match else ""

                # Extract INVSEST criteria
                invsest_criteria: dict[str, str | None] = {
                    "independent": None,
                    "negotiable": None,
                    "valuable": None,
                    "estimable": None,
                    "small": None,
                    "testable": None,
                }
                for criterion in ["Independent", "Negotiable", "Valuable", "Estimable", "Small", "Testable"]:
                    criterion_match = re.search(rf"\*\*{criterion}\*\*:\s*(YES|NO)", story_content, re.IGNORECASE)
                    if criterion_match:
                        invsest_criteria[criterion.lower()] = criterion_match.group(1).upper()

                # Extract acceptance scenarios
                acceptance_pattern = r"(\d+)\.\s+\*\*Given\*\*\s+(.+?),\s+\*\*When\*\*\s+(.+?),\s+\*\*Then\*\*\s+(.+?)(?=\n\n|\n\d+\.|\n###|$)"
                acceptances = re.finditer(acceptance_pattern, story_content, re.DOTALL)

                acceptance_criteria = []
                for acc_match in acceptances:
                    given = acc_match.group(2).strip()
                    when = acc_match.group(3).strip()
                    then = acc_match.group(4).strip()
                    acceptance_criteria.append(f"Given {given}, When {when}, Then {then}")

                # Extract scenarios (Primary, Alternate, Exception, Recovery)
                scenarios = {
                    "primary": [],
                    "alternate": [],
                    "exception": [],
                    "recovery": [],
                }
                scenarios_section = re.search(r"\*\*Scenarios:\*\*\s*\n(.*?)(?=\n\n|\*\*|$)", story_content, re.DOTALL)
                if scenarios_section:
                    scenarios_text = scenarios_section.group(1)
                    # Extract Primary scenarios
                    primary_matches = re.finditer(
                        r"- \*\*Primary Scenario\*\*:\s*(.+?)(?=\n-|\n|$)", scenarios_text, re.DOTALL
                    )
                    for match in primary_matches:
                        scenarios["primary"].append(match.group(1).strip())
                    # Extract Alternate scenarios
                    alternate_matches = re.finditer(
                        r"- \*\*Alternate Scenario\*\*:\s*(.+?)(?=\n-|\n|$)", scenarios_text, re.DOTALL
                    )
                    for match in alternate_matches:
                        scenarios["alternate"].append(match.group(1).strip())
                    # Extract Exception scenarios
                    exception_matches = re.finditer(
                        r"- \*\*Exception Scenario\*\*:\s*(.+?)(?=\n-|\n|$)", scenarios_text, re.DOTALL
                    )
                    for match in exception_matches:
                        scenarios["exception"].append(match.group(1).strip())
                    # Extract Recovery scenarios
                    recovery_matches = re.finditer(
                        r"- \*\*Recovery Scenario\*\*:\s*(.+?)(?=\n-|\n|$)", scenarios_text, re.DOTALL
                    )
                    for match in recovery_matches:
                        scenarios["recovery"].append(match.group(1).strip())

                story_key = f"STORY-{story_counter:03d}"
                spec_data["stories"].append(
                    {
                        "key": story_key,
                        "number": story_number,
                        "title": story_title,
                        "priority": priority,
                        "as_a": as_a_text,
                        "why_priority": why_priority,
                        "invsest": invsest_criteria,
                        "acceptance": acceptance_criteria,
                        "scenarios": scenarios,
                    }
                )
                story_counter += 1

            # Extract functional requirements (FR-XXX)
            req_pattern = r"-?\s*\*\*FR-(\d+)\*\*:\s*System MUST\s+(.+?)(?=\n-|\n\*|\n\n|\*\*FR-|$)"
            requirements = re.finditer(req_pattern, content, re.MULTILINE | re.DOTALL)

            for req_match in requirements:
                req_id = req_match.group(1)
                req_text = req_match.group(2).strip()
                spec_data["requirements"].append(
                    {
                        "id": f"FR-{req_id}",
                        "text": req_text,
                    }
                )

            # Extract success criteria (SC-XXX)
            sc_pattern = r"-?\s*\*\*SC-(\d+)\*\*:\s*(.+?)(?=\n-|\n\*|\n\n|\*\*SC-|$)"
            success_criteria = re.finditer(sc_pattern, content, re.MULTILINE | re.DOTALL)

            for sc_match in success_criteria:
                sc_id = sc_match.group(1)
                sc_text = sc_match.group(2).strip()
                spec_data["success_criteria"].append(
                    {
                        "id": f"SC-{sc_id}",
                        "text": sc_text,
                    }
                )

            # Extract edge cases section
            edge_case_section = re.search(r"### Edge Cases\n(.*?)(?=\n##|$)", content, re.MULTILINE | re.DOTALL)
            if edge_case_section:
                edge_case_text = edge_case_section.group(1)
                # Extract individual edge cases (lines starting with -)
                edge_case_pattern = r"- (.+?)(?=\n-|\n|$)"
                edge_cases = re.finditer(edge_case_pattern, edge_case_text, re.MULTILINE)
                for ec_match in edge_cases:
                    ec_text = ec_match.group(1).strip()
                    if ec_text:
                        spec_data["edge_cases"].append(ec_text)

            return spec_data

        except Exception as e:
            raise ValueError(f"Failed to parse spec.md: {e}") from e

    @beartype
    @require(lambda plan_file: plan_file is not None, "Plan file path must not be None")
    @require(lambda plan_file: plan_file.suffix == ".md", "Plan file must be markdown")
    @ensure(
        lambda result: result is None or (isinstance(result, dict) and "dependencies" in result),
        "Must return None or dict with dependencies",
    )
    def parse_plan_markdown(self, plan_file: Path) -> dict[str, Any] | None:
        """
        Parse a Spec-Kit plan.md file to extract technical context and architecture.

        Args:
            plan_file: Path to plan.md file

        Returns:
            Dictionary with extracted plan information, or None if file doesn't exist
        """
        if not plan_file.exists():
            return None

        try:
            content = plan_file.read_text(encoding="utf-8")
            plan_data: dict[str, Any] = {
                "summary": None,
                "language_version": None,
                "dependencies": [],
                "technology_stack": [],
                "constraints": [],
                "unknowns": [],
                "constitution_check": {},
                "phases": [],
                "architecture": {},
            }

            # Extract summary
            summary_match = re.search(r"^## Summary\n(.*?)(?=\n##|$)", content, re.MULTILINE | re.DOTALL)
            if summary_match:
                plan_data["summary"] = summary_match.group(1).strip()

            # Extract technical context
            tech_context_match = re.search(r"^## Technical Context\n(.*?)(?=\n##|$)", content, re.MULTILINE | re.DOTALL)
            if tech_context_match:
                tech_context = tech_context_match.group(1)
                # Extract language/version
                lang_match = re.search(r"\*\*Language/Version\*\*:\s*(.+?)(?=\n|$)", tech_context, re.MULTILINE)
                if lang_match:
                    plan_data["language_version"] = lang_match.group(1).strip()

                # Extract dependencies
                deps_match = re.search(
                    r"\*\*Primary Dependencies\*\*:\s*\n(.*?)(?=\n\*\*|$)", tech_context, re.MULTILINE | re.DOTALL
                )
                if deps_match:
                    deps_text = deps_match.group(1)
                    # Extract list items
                    dep_items = re.finditer(r"- `(.+?)`\s*-?\s*(.+?)(?=\n-|\n|$)", deps_text, re.MULTILINE)
                    for dep_match in dep_items:
                        dep_name = dep_match.group(1).strip()
                        dep_desc = dep_match.group(2).strip() if dep_match.group(2) else ""
                        plan_data["dependencies"].append({"name": dep_name, "description": dep_desc})

                # Extract Technology Stack
                stack_match = re.search(
                    r"\*\*Technology Stack\*\*:\s*\n(.*?)(?=\n\*\*|$)", tech_context, re.MULTILINE | re.DOTALL
                )
                if stack_match:
                    stack_text = stack_match.group(1)
                    stack_items = re.finditer(r"- (.+?)(?=\n-|\n|$)", stack_text, re.MULTILINE)
                    for item_match in stack_items:
                        plan_data["technology_stack"].append(item_match.group(1).strip())

                # Extract Constraints
                constraints_match = re.search(
                    r"\*\*Constraints\*\*:\s*\n(.*?)(?=\n\*\*|$)", tech_context, re.MULTILINE | re.DOTALL
                )
                if constraints_match:
                    constraints_text = constraints_match.group(1)
                    constraint_items = re.finditer(r"- (.+?)(?=\n-|\n|$)", constraints_text, re.MULTILINE)
                    for item_match in constraint_items:
                        plan_data["constraints"].append(item_match.group(1).strip())

                # Extract Unknowns
                unknowns_match = re.search(
                    r"\*\*Unknowns\*\*:\s*\n(.*?)(?=\n\*\*|$)", tech_context, re.MULTILINE | re.DOTALL
                )
                if unknowns_match:
                    unknowns_text = unknowns_match.group(1)
                    unknown_items = re.finditer(r"- (.+?)(?=\n-|\n|$)", unknowns_text, re.MULTILINE)
                    for item_match in unknown_items:
                        plan_data["unknowns"].append(item_match.group(1).strip())

            # Extract Constitution Check section (CRITICAL for /speckit.analyze)
            constitution_match = re.search(
                r"^## Constitution Check\n(.*?)(?=\n##|$)", content, re.MULTILINE | re.DOTALL
            )
            if constitution_match:
                constitution_text = constitution_match.group(1)
                plan_data["constitution_check"] = {
                    "article_vii": {},
                    "article_viii": {},
                    "article_ix": {},
                    "status": None,
                }
                # Extract Article VII (Simplicity)
                article_vii_match = re.search(
                    r"\*\*Article VII \(Simplicity\)\*\*:\s*\n(.*?)(?=\n\*\*|$)",
                    constitution_text,
                    re.MULTILINE | re.DOTALL,
                )
                if article_vii_match:
                    article_vii_text = article_vii_match.group(1)
                    plan_data["constitution_check"]["article_vii"] = {
                        "using_3_projects": re.search(r"- \[([ x])\]", article_vii_text) is not None,
                        "no_future_proofing": re.search(r"- \[([ x])\]", article_vii_text) is not None,
                    }
                # Extract Article VIII (Anti-Abstraction)
                article_viii_match = re.search(
                    r"\*\*Article VIII \(Anti-Abstraction\)\*\*:\s*\n(.*?)(?=\n\*\*|$)",
                    constitution_text,
                    re.MULTILINE | re.DOTALL,
                )
                if article_viii_match:
                    article_viii_text = article_viii_match.group(1)
                    plan_data["constitution_check"]["article_viii"] = {
                        "using_framework_directly": re.search(r"- \[([ x])\]", article_viii_text) is not None,
                        "single_model_representation": re.search(r"- \[([ x])\]", article_viii_text) is not None,
                    }
                # Extract Article IX (Integration-First)
                article_ix_match = re.search(
                    r"\*\*Article IX \(Integration-First\)\*\*:\s*\n(.*?)(?=\n\*\*|$)",
                    constitution_text,
                    re.MULTILINE | re.DOTALL,
                )
                if article_ix_match:
                    article_ix_text = article_ix_match.group(1)
                    plan_data["constitution_check"]["article_ix"] = {
                        "contracts_defined": re.search(r"- \[([ x])\]", article_ix_text) is not None,
                        "contract_tests_written": re.search(r"- \[([ x])\]", article_ix_text) is not None,
                    }
                # Extract Status
                status_match = re.search(r"\*\*Status\*\*:\s*(PASS|FAIL)", constitution_text, re.IGNORECASE)
                if status_match:
                    plan_data["constitution_check"]["status"] = status_match.group(1).upper()

            # Extract Phases
            phase_pattern = r"^## Phase (-?\d+):\s*(.+?)\n(.*?)(?=\n## Phase|$)"
            phases = re.finditer(phase_pattern, content, re.MULTILINE | re.DOTALL)
            for phase_match in phases:
                phase_num = phase_match.group(1)
                phase_name = phase_match.group(2).strip()
                phase_content = phase_match.group(3).strip()
                plan_data["phases"].append(
                    {
                        "number": phase_num,
                        "name": phase_name,
                        "content": phase_content,
                    }
                )

            return plan_data

        except Exception as e:
            raise ValueError(f"Failed to parse plan.md: {e}") from e

    @beartype
    @require(lambda tasks_file: tasks_file is not None, "Tasks file path must not be None")
    @require(lambda tasks_file: tasks_file.suffix == ".md", "Tasks file must be markdown")
    @ensure(
        lambda result: result is None or (isinstance(result, dict) and "tasks" in result),
        "Must return None or dict with tasks",
    )
    def parse_tasks_markdown(self, tasks_file: Path) -> dict[str, Any] | None:
        """
        Parse a Spec-Kit tasks.md file to extract tasks with IDs, story mappings, and dependencies.

        Args:
            tasks_file: Path to tasks.md file

        Returns:
            Dictionary with extracted task information, or None if file doesn't exist
        """
        if not tasks_file.exists():
            return None

        try:
            content = tasks_file.read_text(encoding="utf-8")
            tasks_data: dict[str, Any] = {
                "tasks": [],
                "phases": [],
            }

            # Extract tasks (format: - [ ] [TaskID] [P?] [Story?] Description)
            task_pattern = r"- \[([ x])\] \[?([T\d]+)\]?\s*\[?([P])?\]?\s*\[?([US\d]+)?\]?\s*(.+?)(?=\n-|\n##|$)"
            tasks = re.finditer(task_pattern, content, re.MULTILINE | re.DOTALL)

            for task_match in tasks:
                checked = task_match.group(1) == "x"
                task_id = task_match.group(2)
                is_parallel = task_match.group(3) == "P"
                story_ref = task_match.group(4)
                description = task_match.group(5).strip()

                tasks_data["tasks"].append(
                    {
                        "id": task_id,
                        "description": description,
                        "checked": checked,
                        "parallel": is_parallel,
                        "story_ref": story_ref,
                    }
                )

            # Extract phase sections and map tasks to phases
            phase_pattern = r"^## Phase (\d+): (.+?)\n(.*?)(?=\n## Phase|$)"
            phases = re.finditer(phase_pattern, content, re.MULTILINE | re.DOTALL)

            for phase_match in phases:
                phase_num = phase_match.group(1)
                phase_name = phase_match.group(2).strip()
                phase_content = phase_match.group(3)

                # Find tasks in this phase
                phase_tasks = []
                phase_task_pattern = (
                    r"- \[([ x])\] \[?([T\d]+)\]?\s*\[?([P])?\]?\s*\[?([US\d]+)?\]?\s*(.+?)(?=\n-|\n##|$)"
                )
                phase_task_matches = re.finditer(phase_task_pattern, phase_content, re.MULTILINE | re.DOTALL)

                for task_match in phase_task_matches:
                    checked = task_match.group(1) == "x"
                    task_id = task_match.group(2)
                    is_parallel = task_match.group(3) == "P"
                    story_ref = task_match.group(4)
                    description = task_match.group(5).strip()

                    phase_tasks.append(
                        {
                            "id": task_id,
                            "description": description,
                            "checked": checked,
                            "parallel": is_parallel,
                            "story_ref": story_ref,
                            "phase": phase_num,
                            "phase_name": phase_name,
                        }
                    )

                tasks_data["phases"].append(
                    {
                        "number": phase_num,
                        "name": phase_name,
                        "content": phase_content,
                        "tasks": phase_tasks,
                    }
                )

            return tasks_data

        except Exception as e:
            raise ValueError(f"Failed to parse tasks.md: {e}") from e

    def parse_memory_files(self, memory_dir: Path) -> dict[str, Any]:
        """
        Parse Spec-Kit memory files (constitution.md, etc.).

        Args:
            memory_dir: Path to memory directory

        Returns:
            Dictionary with extracted memory information
        """
        memory_data: dict[str, Any] = {
            "constitution": None,
            "principles": [],
            "constraints": [],
            "version": None,
        }

        if not memory_dir.exists():
            return memory_data

        # Parse constitution.md
        constitution_file = memory_dir / "constitution.md"
        if constitution_file.exists():
            try:
                content = constitution_file.read_text(encoding="utf-8")
                memory_data["constitution"] = content

                # Extract version
                version_match = re.search(r"\*\*Version\*\*:\s*(\d+\.\d+\.\d+)", content, re.MULTILINE)
                if version_match:
                    memory_data["version"] = version_match.group(1)

                # Extract principles (from "### I. Principle Name" or "### Principle Name" sections)
                principle_pattern = r"###\s+(?:[IVX]+\.\s*)?(.+?)(?:\s*\(NON-NEGOTIABLE\))?\n\n(.*?)(?=\n###|\n##|$)"
                principles = re.finditer(principle_pattern, content, re.MULTILINE | re.DOTALL)

                for prin_match in principles:
                    principle_name = prin_match.group(1).strip()
                    principle_content = prin_match.group(2).strip() if prin_match.group(2) else ""
                    # Skip placeholder principles
                    if not principle_name.startswith("["):
                        # Extract rationale if present
                        rationale_match = re.search(
                            r"\*\*Rationale\*\*:\s*(.+?)(?=\n\n|\n###|\n##|$)", principle_content, re.DOTALL
                        )
                        rationale = rationale_match.group(1).strip() if rationale_match else ""

                        memory_data["principles"].append(
                            {
                                "name": principle_name,
                                "description": (
                                    principle_content.split("**Rationale**")[0].strip()
                                    if "**Rationale**" in principle_content
                                    else principle_content
                                ),
                                "rationale": rationale,
                            }
                        )

                # Extract constraints from Governance section
                governance_section = re.search(r"## Governance\n(.*?)(?=\n##|$)", content, re.MULTILINE | re.DOTALL)
                if governance_section:
                    # Look for constraint patterns
                    constraint_pattern = r"- (.+?)(?=\n-|\n|$)"
                    constraints = re.finditer(constraint_pattern, governance_section.group(1), re.MULTILINE)
                    for const_match in constraints:
                        const_text = const_match.group(1).strip()
                        if const_text and not const_text.startswith("["):
                            memory_data["constraints"].append(const_text)

            except Exception:
                # Non-fatal error - log but continue
                pass

        return memory_data
