"""
Sync command - Bidirectional synchronization for Spec-Kit and repositories.

This module provides commands for synchronizing changes between Spec-Kit artifacts,
repository changes, and SpecFact plans.
"""

from __future__ import annotations

import shutil
from pathlib import Path
from typing import Any

import typer
from rich.console import Console
from rich.progress import Progress, SpinnerColumn, TextColumn

from specfact_cli.models.plan import PlanBundle
from specfact_cli.sync.speckit_sync import SpecKitSync


app = typer.Typer(help="Synchronize Spec-Kit artifacts and repository changes")
console = Console()


def _sync_speckit_to_specfact(repo: Path, converter: Any, scanner: Any, progress: Any) -> tuple[PlanBundle, int, int]:
    """
    Sync Spec-Kit artifacts to SpecFact format.

    Returns:
        Tuple of (merged_bundle, features_updated, features_added)
    """
    from specfact_cli.generators.plan_generator import PlanGenerator
    from specfact_cli.utils.structure import SpecFactStructure
    from specfact_cli.validators.schema import validate_plan_bundle

    plan_path = repo / SpecFactStructure.DEFAULT_PLAN
    existing_bundle: PlanBundle | None = None

    if plan_path.exists():
        validation_result = validate_plan_bundle(plan_path)
        if isinstance(validation_result, tuple):
            is_valid, _error, bundle = validation_result
            if is_valid and bundle:
                existing_bundle = bundle

    # Convert Spec-Kit to SpecFact
    converted_bundle = converter.convert_plan(None if not existing_bundle else plan_path)

    # Merge with existing plan if it exists
    features_updated = 0
    features_added = 0

    if existing_bundle:
        feature_keys_existing = {f.key for f in existing_bundle.features}

        for feature in converted_bundle.features:
            if feature.key in feature_keys_existing:
                existing_idx = next(i for i, f in enumerate(existing_bundle.features) if f.key == feature.key)
                existing_bundle.features[existing_idx] = feature
                features_updated += 1
            else:
                existing_bundle.features.append(feature)
                features_added += 1

        # Update product themes
        themes_existing = set(existing_bundle.product.themes)
        themes_new = set(converted_bundle.product.themes)
        existing_bundle.product.themes = list(themes_existing | themes_new)

        # Write merged bundle
        generator = PlanGenerator()
        generator.generate(existing_bundle, plan_path)
        return existing_bundle, features_updated, features_added
    # Write new bundle
    generator = PlanGenerator()
    generator.generate(converted_bundle, plan_path)
    return converted_bundle, 0, len(converted_bundle.features)


@app.command("spec-kit")
def sync_spec_kit(
    repo: Path = typer.Option(
        Path("."),
        "--repo",
        help="Path to repository",
        exists=True,
        file_okay=False,
        dir_okay=True,
    ),
    bidirectional: bool = typer.Option(
        False,
        "--bidirectional",
        help="Enable bidirectional sync (Spec-Kit ↔ SpecFact)",
    ),
    plan: Path | None = typer.Option(
        None,
        "--plan",
        help="Path to SpecFact plan bundle for SpecFact → Spec-Kit conversion (default: .specfact/plans/main.bundle.yaml)",
    ),
    overwrite: bool = typer.Option(
        False,
        "--overwrite",
        help="Overwrite existing Spec-Kit artifacts (delete all existing before sync)",
    ),
    watch: bool = typer.Option(
        False,
        "--watch",
        help="Watch mode for continuous sync",
    ),
    interval: int = typer.Option(
        5,
        "--interval",
        help="Watch interval in seconds (default: 5)",
        min=1,
    ),
) -> None:
    """
    Sync changes between Spec-Kit artifacts and SpecFact.

    Synchronizes markdown artifacts generated by Spec-Kit slash commands
    with SpecFact plan bundles and protocols.

    Example:
        specfact sync spec-kit --repo . --bidirectional
    """
    from specfact_cli.importers.speckit_converter import SpecKitConverter
    from specfact_cli.importers.speckit_scanner import SpecKitScanner
    from specfact_cli.utils.structure import SpecFactStructure
    from specfact_cli.validators.schema import validate_plan_bundle

    console.print(f"[bold cyan]Syncing Spec-Kit artifacts from:[/bold cyan] {repo}")

    # Watch mode (not implemented yet)
    if watch:
        console.print("[yellow]→ Watch mode enabled (not implemented yet)[/yellow]")
        console.print(f"[dim]Would watch for changes every {interval} seconds[/dim]")
        raise typer.Exit(0)

    # Step 1: Detect Spec-Kit repository
    scanner = SpecKitScanner(repo)
    if not scanner.is_speckit_repo():
        console.print("[bold red]✗[/bold red] Not a Spec-Kit repository")
        console.print("[dim]Expected Spec-Kit structure (.specify/ directory)[/dim]")
        raise typer.Exit(1)

    console.print("[bold green]✓[/bold green] Detected Spec-Kit repository")

    # Step 2: Detect SpecFact structure
    specfact_exists = (repo / SpecFactStructure.ROOT).exists()

    if not specfact_exists:
        console.print("[yellow]⚠[/yellow] SpecFact structure not found")
        console.print(f"[dim]Initialize with: specfact plan init --scaffold --repo {repo}[/dim]")
        # Create structure automatically
        SpecFactStructure.ensure_structure(repo)
        console.print("[bold green]✓[/bold green] Created SpecFact structure")

    if specfact_exists:
        console.print("[bold green]✓[/bold green] Detected SpecFact structure")

    sync = SpecKitSync(repo)
    converter = SpecKitConverter(repo)

    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        console=console,
    ) as progress:
        # Step 3: Scan Spec-Kit artifacts
        task = progress.add_task("[cyan]📦[/cyan] Scanning Spec-Kit artifacts...", total=None)
        features = scanner.discover_features()
        progress.update(task, description=f"[green]✓[/green] Found {len(features)} features in specs/")

        # Step 4: Sync based on mode
        specfact_changes: dict[str, Any] = {}
        conflicts: list[dict[str, Any]] = []
        features_converted_speckit = 0

        if bidirectional:
            # Bidirectional sync: Spec-Kit → SpecFact and SpecFact → Spec-Kit
            # Step 5.1: Spec-Kit → SpecFact (unidirectional sync)
            task = progress.add_task("[cyan]📝[/cyan] Converting Spec-Kit → SpecFact...", total=None)
            merged_bundle, features_updated, features_added = _sync_speckit_to_specfact(
                repo, converter, scanner, progress
            )

            if features_updated > 0 or features_added > 0:
                progress.update(
                    task,
                    description=f"[green]✓[/green] Updated {features_updated}, Added {features_added} features",
                )
                console.print(f"[dim]  - Updated {features_updated} features[/dim]")
                console.print(f"[dim]  - Added {features_added} new features[/dim]")
            else:
                progress.update(
                    task,
                    description=f"[green]✓[/green] Created plan with {len(merged_bundle.features)} features",
                )

            # Step 5.2: SpecFact → Spec-Kit (reverse conversion)
            task = progress.add_task("[cyan]🔄[/cyan] Converting SpecFact → Spec-Kit...", total=None)

            # Detect SpecFact changes
            specfact_changes = sync.detect_specfact_changes(repo)

            if specfact_changes:
                # Load plan bundle and convert to Spec-Kit
                # Use provided plan path, or default to main plan
                if plan:
                    plan_path = plan if plan.is_absolute() else repo / plan
                else:
                    plan_path = repo / SpecFactStructure.DEFAULT_PLAN

                if plan_path.exists():
                    validation_result = validate_plan_bundle(plan_path)
                    if isinstance(validation_result, tuple):
                        is_valid, _error, plan_bundle = validation_result
                        if is_valid and plan_bundle:
                            # Handle overwrite mode
                            if overwrite:
                                # Delete existing Spec-Kit artifacts before conversion
                                specs_dir = repo / "specs"
                                if specs_dir.exists():
                                    console.print(
                                        "[yellow]⚠[/yellow] Overwrite mode: Removing existing Spec-Kit artifacts..."
                                    )
                                    shutil.rmtree(specs_dir)
                                    specs_dir.mkdir(parents=True, exist_ok=True)
                                    console.print("[green]✓[/green] Existing artifacts removed")

                            # Convert SpecFact plan bundle to Spec-Kit markdown
                            features_converted_speckit = converter.convert_to_speckit(plan_bundle)
                            progress.update(
                                task,
                                description=f"[green]✓[/green] Converted {features_converted_speckit} features to Spec-Kit",
                            )
                            mode_text = "overwritten" if overwrite else "generated"
                            console.print(
                                f"[dim]  - {mode_text.capitalize()} spec.md, plan.md, tasks.md for {features_converted_speckit} features[/dim]"
                            )
                        else:
                            progress.update(task, description="[yellow]⚠[/yellow] Plan bundle validation failed")
                            console.print("[yellow]⚠[/yellow] Could not load plan bundle for conversion")
                    else:
                        progress.update(task, description="[yellow]⚠[/yellow] Plan bundle not found")
                else:
                    progress.update(task, description="[green]✓[/green] No SpecFact plan to sync")
            else:
                progress.update(task, description="[green]✓[/green] No SpecFact changes to sync")

            # Detect conflicts between both directions
            speckit_changes = sync.detect_speckit_changes(repo)
            conflicts = sync.detect_conflicts(speckit_changes, specfact_changes)

            if conflicts:
                console.print(f"[yellow]⚠[/yellow] Found {len(conflicts)} conflicts")
                console.print("[dim]Conflicts resolved using priority rules (SpecFact > Spec-Kit for artifacts)[/dim]")
            else:
                console.print("[bold green]✓[/bold green] No conflicts detected")
        else:
            # Unidirectional sync: Spec-Kit → SpecFact
            task = progress.add_task("[cyan]📝[/cyan] Converting to SpecFact format...", total=None)

            merged_bundle, features_updated, features_added = _sync_speckit_to_specfact(
                repo, converter, scanner, progress
            )

            if features_updated > 0 or features_added > 0:
                task = progress.add_task("[cyan]🔀[/cyan] Merging with existing plan...", total=None)
                progress.update(
                    task,
                    description=f"[green]✓[/green] Updated {features_updated} features, Added {features_added} features",
                )
                console.print(f"[dim]  - Updated {features_updated} features[/dim]")
                console.print(f"[dim]  - Added {features_added} new features[/dim]")
            else:
                progress.update(
                    task, description=f"[green]✓[/green] Created plan with {len(merged_bundle.features)} features"
                )
                console.print(f"[dim]Created plan with {len(merged_bundle.features)} features[/dim]")

            # Report features synced
            console.print()
            if features:
                console.print("[bold cyan]Features synced:[/bold cyan]")
                for feature in features:
                    feature_key = feature.get("feature_key", "UNKNOWN")
                    feature_title = feature.get("title", "Unknown Feature")
                    console.print(f"  - [cyan]{feature_key}[/cyan]: {feature_title}")

        # Step 8: Output Results
        console.print()
        if bidirectional:
            console.print("[bold cyan]Sync Summary (Bidirectional):[/bold cyan]")
            console.print(f"  - Spec-Kit → SpecFact: Updated {features_updated}, Added {features_added} features")
            if specfact_changes:
                console.print(
                    f"  - SpecFact → Spec-Kit: {features_converted_speckit} features converted to Spec-Kit markdown"
                )
            else:
                console.print("  - SpecFact → Spec-Kit: No changes detected")
            if conflicts:
                console.print(f"  - Conflicts: {len(conflicts)} detected and resolved")
            else:
                console.print("  - Conflicts: None detected")
        else:
            console.print("[bold cyan]Sync Summary (Unidirectional):[/bold cyan]")
            if features:
                console.print(f"  - Features synced: {len(features)}")
            if features_updated > 0 or features_added > 0:
                console.print(f"  - Updated: {features_updated} features")
                console.print(f"  - Added: {features_added} new features")
            console.print("  - Direction: Spec-Kit → SpecFact")

    console.print()
    console.print("[bold green]✓[/bold green] Sync complete!")


@app.command("repository")
def sync_repository(
    repo: Path = typer.Option(
        Path("."),
        "--repo",
        help="Path to repository",
        exists=True,
        file_okay=False,
        dir_okay=True,
    ),
    target: Path | None = typer.Option(
        None,
        "--target",
        help="Target directory for artifacts (default: .specfact)",
    ),
    watch: bool = typer.Option(
        False,
        "--watch",
        help="Watch mode for continuous sync",
    ),
    interval: int = typer.Option(
        5,
        "--interval",
        help="Watch interval in seconds (default: 5)",
        min=1,
    ),
    confidence: float = typer.Option(
        0.5,
        "--confidence",
        help="Minimum confidence threshold for feature detection (default: 0.5)",
        min=0.0,
        max=1.0,
    ),
) -> None:
    """
    Sync code changes to SpecFact artifacts.

    Monitors repository code changes, updates plan artifacts based on detected
    features/stories, and tracks deviations from manual plans.

    Example:
        specfact sync repository --repo . --confidence 0.5
    """
    from specfact_cli.sync.repository_sync import RepositorySync

    console.print(f"[bold cyan]Syncing repository changes from:[/bold cyan] {repo}")

    if target is None:
        target = repo / ".specfact"

    sync = RepositorySync(repo, target, confidence_threshold=confidence)

    if watch:
        console.print("[yellow]→ Watch mode enabled (not implemented yet)[/yellow]")
        console.print(f"[dim]Would watch for changes every {interval} seconds[/dim]")
        raise typer.Exit(0)

    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        console=console,
    ) as progress:
        # Step 1: Detect code changes
        task = progress.add_task("Detecting code changes...", total=None)
        result = sync.sync_repository_changes(repo)
        progress.update(task, description=f"✓ Detected {len(result.code_changes)} code changes")

        # Step 2: Show plan updates
        if result.plan_updates:
            task = progress.add_task("Updating plan artifacts...", total=None)
            total_features = sum(update.get("features", 0) for update in result.plan_updates)
            progress.update(task, description=f"✓ Updated plan artifacts ({total_features} features)")

        # Step 3: Show deviations
        if result.deviations:
            task = progress.add_task("Tracking deviations...", total=None)
            progress.update(task, description=f"✓ Found {len(result.deviations)} deviations")

    # Report results
    console.print(f"[bold cyan]Code Changes:[/bold cyan] {len(result.code_changes)}")
    if result.plan_updates:
        console.print(f"[bold cyan]Plan Updates:[/bold cyan] {len(result.plan_updates)}")
    if result.deviations:
        console.print(f"[yellow]⚠[/yellow] Found {len(result.deviations)} deviations from manual plan")
        console.print("[dim]Run 'specfact plan compare' for detailed deviation report[/dim]")
    else:
        console.print("[bold green]✓[/bold green] No deviations detected")

    console.print("[bold green]✓[/bold green] Repository sync complete!")
