from typing import Callable, Union, Optional, List, Any
import os

import numpy as np
import numpy.typing as npt
import soundfile as sf

from ..data.config import Config
from ..processors.abstract_data_processor import AbstractDataProcessor
from .multi_spectrogram import MultiSpectrogram
from ..data.data import AudioPadding, ListOrdering
from ..misc.utils import get_multi_stft
from ..exceptions.lib_exceptions import WrongConfigurationException, BadTypeException


class SpectrogramFactory:
    def __init__(self, 
                config : Config,
                data_processor : Optional[AbstractDataProcessor] = None,
                audio_padder : Optional[Callable[[np.ndarray, int], np.ndarray]] = AudioPadding.NONE,
                ordering : ListOrdering = ListOrdering.ALTERNATE
                ) -> None:
        # Set the config
        self.__config = config

        # Set the data processor
        self.__processor = data_processor

        # Set the padding functino
        self.__audio_padder = audio_padder

        # set ordering
        self.__ordering = ordering

        # Check if config is correct
        if self.__audio_padder is not None and self.__config.audio_length is None:
            raise WrongConfigurationException("Audio Padder can't be userd without a audio_length configured in the config object.")
        
    @property
    def num_channel(self) -> int:
        return self.__config.num_channel

    def get_spectrogram_from_audio(self, audio_array : np.ndarray) -> MultiSpectrogram:
        """
        Converts an audio array to a MultiSpectrogram instance.

        Args:
            audio_array (np.ndarray): 
                A numpy array representing the audio data. The shape of the array should be (num_channels, samples).

        Raises:
            WrongConfigurationException: 
                If the number of channels in the audio array does not match the expected number of channels in the configuration.
            WrongConfigurationException: 
                If the audio length or audio padding function is not provided in the configuration.

        Returns:
            MultiSpectrogram: 
                A MultiSpectrogram instance created from the given audio array.
        """
        # Get the channel number
        num_channels = audio_array.shape[0]

        # padding
        if self.__audio_padder is not None:
            audio_array = self.__audio_padder(audio_array, self.__config.audio_length)

        # If audio is mono, we can copy 
        if num_channels == 1 and self.__config.num_channel > 1:
            audio_array = np.stack([audio_array]*self.__config.num_channel, axis = 0)

        # If number of channels and desired channel number fit, we can do stfts
        if num_channels == self.__config.num_channel:
            stfts = get_multi_stft(audio_array, **self.__config.get_istft_kwargs())
            return MultiSpectrogram.from_stfts(self.__config, self.__processor, self.__ordering, *stfts)
        
        else:
            raise WrongConfigurationException(f"Cannot handle data with {num_channels} channels. Configuration is set for {self.__config.num_channel} channels.")
        
    def get_spectrogram_from_path(self, file_path : str) -> MultiSpectrogram:
        """
        Converts an audio file from the given file path to a MultiSpectrogram instance.

        Args:
            file_path (str): 
                The path to the audio file.

        Returns:
            MultiSpectrogram: 
                A MultiSpectrogram instance created from the audio file.

        Raises:
            AssertionError: 
                If the file does not exist or if the sample rate of the file does not match the configured sample rate.
        """
        # assert file exist
        assert os.path.isfile(file_path), f"The file {file_path} doesn't exist"

        # Load the audio
        audio, sample_rate = sf.read(file_path, always_2d=True)

        # assert sample rate is correct
        assert sample_rate == self.__config.sample_rate, "Sample rate of the file is different from the one configured in the Config onbject"

        # return the spectro
        return self.get_spectrogram_from_audio(audio.transpose())
    
    def get_spectrograms_from_files(self, audio_or_file_list : List[Union[str, np.ndarray]]) -> List[MultiSpectrogram]:
        """
        Converts a list of audio file paths or numpy arrays to a list of MultiSpectrogram instances.

        Args:
            audio_or_file_list (List[Union[str, np.ndarray]]): 
                A list containing either file paths to audio files (as strings) or numpy arrays representing audio data.

        Raises:
            BadTypeException: 
                If an element in the list is neither a string nor a numpy array.

        Returns:
            List[MultiSpectrogram]: 
                A list of MultiSpectrogram instances created from the audio files or arrays.
        """
        spectros : list[MultiSpectrogram] = []
        for audio_or_file in audio_or_file_list:
            if isinstance(audio_or_file, str):
                spectros.append(
                    self.get_spectrogram_from_path(audio_or_file))
            elif isinstance(audio_or_file, np.ndarray):
                spectros.append(
                    self.get_spectrogram_from_audio(audio_or_file))
            else:
                raise BadTypeException(f"Couldn't handle type : {type(audio_or_file)}. Can only get file_path as str and audio_file as NDArray")
        return spectros
    
    def get_spectrogram_from_model_output(self, model_output : npt.NDArray[np.float64]) -> List[MultiSpectrogram]:
        """From model output, recreate a spectrogram, rearrange the amplitude phase if needed.
        /!\\ the model output should be shaped like (batch, channel, h, w)

        Args:
            model_output (npt.NDArray[np.float64]): [description]

        Returns:
            list[MultiSpectrogram]: Spectrograms from the model output
        """
        # Check if need to be reordered
        if self.__ordering == ListOrdering.AMPLITUDE_PHASE:
            new_order = np.arange(model_output.shape[1]).reshape(2, -1).flatten("F")
            model_output = model_output[:, new_order, :, :] # Rearange the data in correct order

        # Make spectrograms once it's on the correct order
        spectros : list[MultiSpectrogram] = []
        for x in model_output:
            # Backward process
            backward_processed = self.__processor._backward(x)

            # Get the spectrogram
            spectros.append(
                MultiSpectrogram(self.__config, self.__processor, self.__ordering, backward_processed))
            
        return spectros
    
    def get_numpy_dataset(self, 
                          audio_or_file_list : Union[List[Union[str, np.ndarray]], List[MultiSpectrogram]], 
                          use_processor : bool
                          ) -> npt.NDArray[np.float64]:
        """
        Converts a list of audio files or MultiSpectrogram instances to a numpy dataset.

        Args:
            audio_or_file_list (Union[List[Union[str, np.ndarray]], List[MultiSpectrogram]]): 
                A list containing either file paths to audio files, numpy arrays representing audio data,
                or a list of MultiSpectrogram instances.
            use_processor (bool): 
                A boolean flag indicating whether to process the data before converting it to a numpy dataset.

        Raises:
            WrongConfigurationException: 
                If the audio length is not provided in the configuration.
            WrongConfigurationException: 
                If the audio padding function is not provided in the configuration.

        Returns:
            npt.NDArray[np.float64]: 
                A numpy array containing the processed audio data.
        """
        if self.__config.audio_length is None:
            raise WrongConfigurationException("Cannot create a numpy dataset with no audio length provided. \n" + \
                            "Set the audio_length field in the configuration.")
        if self.__audio_padder is None:
            raise WrongConfigurationException("Cannot create a numpy dataset with no audio padding function provided. \n" + \
                            "Set audio_padder argument in the factory constructor.")

        spectros = self.get_spectrograms_from_files(audio_or_file_list)
        X_data = np.zeros((len(spectros), *spectros[0].shape))
        for i, spec in enumerate(spectros):
            X_data[i] = spec.to_rearanged_data(use_processor)
        return X_data
    
    def _get_stft_shape(self):
        """#### Return the shape that a stft would have 

        #### Returns:
            - tuple[int, int]: stft shape
            >>> 
        """
        stft_config = self.__config.get_istft_kwargs()
        audio_length = stft_config.get("audio_length")
        window_length = stft_config.get("window_length")
        hop_length = stft_config.get("hop_length")
        n_fft = stft_config.get("n_fft")
        num_frames = (audio_length - window_length) // hop_length + 1
        num_frequency_bins = n_fft // 2 + 1
        return num_frequency_bins, num_frames


    