import re
import bs4
import sphinx
from pathlib import Path
from bs4 import BeautifulSoup
from typing import Dict, Any, List
from collections import defaultdict
from sphinx.application import Sphinx


__version__ = "1.0.1"
__author__ = 'Adam Korn <hello@dailykitten.net>'


def setup(app: Sphinx) -> Dict[str, Any]:
    app.connect("builder-inited", add_static_path)
    app.connect('build-finished', add_source_code)

    app.setup_extension('sphinx.ext.viewcode')
    app.add_css_file("sphinx-inlinecode.css")

    return {'version': sphinx.__display_version__, 'parallel_read_safe': True}


def add_static_path(app) -> None:
    """Add the path for the ``_static`` folder"""
    app.config.html_static_path.append(
        str(Path(__file__).parent.joinpath("_static").absolute())
    )


def add_source_code(app, exception):
    """Insert source code blocks into documentation entries."""
    root_dir = Path(app.outdir).joinpath("_modules").absolute()
    if not root_dir.exists():
        return

    code_blocks = get_code_blocks(root_dir)
    for doc_file, block_data in code_blocks.items():
        result = insert_source_code(doc_file, block_data)
        doc_file.write_text(str(result), encoding='utf-8')


def get_code_blocks(root_dir: Path) -> Dict[Path, List[Dict]]:
    """Retrieve code blocks from HTML files generated by :mod:`sphinx.ext.viewcode`

    :param root_dir: path of the ``build/html/_modules`` directory
    :return: mapping of documentation file paths to lists of code block data.
    """
    files = get_source_code_files(root_dir)
    mapping = defaultdict(list)

    for file in files:
        soup = BeautifulSoup(file.read_text(encoding='utf-8'), 'html.parser')
        code_blocks = soup.findAll('div', 'viewcode-block')

        for block in code_blocks:
            backlink = block.find("a", "viewcode-back")
            target, qualname = backlink.get('href').split("#")

            target_file = Path(file.parent / Path(target)).resolve()
            mapping[target_file].append({
                "ref_id": qualname,
                "code": block
            })

            # Remove internal link to documentation entry
            backlink.replace_with()

    return mapping


def get_source_code_files(root: Path) -> List[Path]:
    """Retrieves HTML files created by :mod:`sphinx.ext.viewcode`

    :param root: path of the ``build/html/_modules`` directory
    :return: list of all files in the directory and its subdirectories
    """
    files = []

    for entry in root.iterdir():
        if entry.is_file():
            files.append(entry)
        elif entry.is_dir():
            files.extend(get_source_code_files(entry))

    return files


def insert_source_code(file: Path, block_data: List[Dict]) -> BeautifulSoup:
    """Inserts source code blocks into the specified documentation HTML file.

    :param file: path to the HTML file.
    :param block_data: list of dictionaries containing code block data.
    :return: HTML content of the file, with all code blocks inserted
    """
    soup = BeautifulSoup(file.read_text(encoding='utf-8'), 'html.parser')

    for block in block_data:
        # Find the corresponding documentation entry
        doc_entry = soup.find("dt", id=block["ref_id"])

        # Insert formatted source code block after object signature.
        code_block = wrap_code_block(block['code'])
        doc_entry.append(code_block)

        # Remove viewcode link
        viewcode_link = doc_entry.find("span", "viewcode-link")
        viewcode_link.parent.replace_with()

    return soup


def wrap_code_block(code_block: bs4.Tag) -> BeautifulSoup:
    """Wraps the given code block inside a <details> HTML element.

    :param code_block: HTML of the code block to wrap.
    :return: the wrapped code block.
    """
    formatted_block = adjust_indentation(code_block)
    html = """
    <details class="sphinx-inlinecode">
        <summary>
            <span class="pre">View Source Code</span>
        </summary>
        <div class="highlight">
            <pre>{adjusted_code_block}</pre>
        </div>
    </details>
    """.format(adjusted_code_block=formatted_block)
    return BeautifulSoup(html, 'html.parser')


def adjust_indentation(code_block: bs4.Tag) -> bs4.element.Tag:
    """Adjusts indentation of the code block by removing common leading whitespace.

    :param code_block: HTML code block whose indentation needs adjustment.
    :return: code block with adjusted indentation.
    """
    contents = code_block.contents

    if not isinstance(contents[0], bs4.element.NavigableString):
        return code_block  # Block has no indentation

    initial_indent = len(contents[0])
    pattern = fr"[ ]{{{initial_indent}}}(.*)"

    for child in contents:
        if isinstance(child, bs4.NavigableString):
            replacement = re.sub(pattern, r"\1", child)
            child.replace_with(replacement)
        else:
            replacement = re.sub(pattern, r"\1", child.string)
            child.string = replacement

    return code_block
