import os
import json
import torch
import torch.nn as nn
import matplotlib.pyplot as plt
import seaborn as sns
from tqdm.notebook import tqdm
from spotPython.torch.traintest import (
    get_removed_attributes_and_base_net,
    create_train_val_data_loaders,
)
from spotPython.hyperparameters.optimizer import optimizer_handler
from spotPython.utils.device import getDevice
import numpy as np


def _get_config_file(model_path, model_name):
    return os.path.join(model_path, model_name + ".config")


def _get_model_file(model_path, model_name):
    return os.path.join(model_path, model_name + ".tar")


def _get_result_file(model_path, model_name):
    return os.path.join(model_path, model_name + "_results.json")


# def load_model(model_path, model_name, net=None):
#     config_file = _get_config_file(model_path, model_name)
#     model_file = _get_model_file(model_path, model_name)
#     assert os.path.isfile(
#         config_file
#     ), f'Could not find config file "{config_file}". Is the path correct and you have your model config stored here?'
#     assert os.path.isfile(
#         model_file
#     ), f'Could not find model file "{model_file}". Is the path correct and you have your model stored here?'
#     with open(config_file) as f:
#         config_dict = json.load(f)
#     if net is None:
#         act_fn_name = config_dict["act_fn"].pop("name").lower()
#         assert (
#             act_fn_name in act_fn_by_name
#         ), f'Unknown activation function "{act_fn_name}". Please add it to the "act_fn_by_name" dict.'
#         act_fn = act_fn_by_name[act_fn_name]()
#         net = BaseNetwork(act_fn=act_fn, **config_dict)
#     net.load_state_dict(torch.load(model_file))
#     return net


def save_model(model, model_path, model_name):
    config_dict = model.config
    os.makedirs(model_path, exist_ok=True)
    config_file = _get_config_file(model_path, model_name)
    model_file = _get_model_file(model_path, model_name)
    with open(config_file, "w") as f:
        json.dump(config_dict, f)
    torch.save(model.state_dict(), model_file)


def train_model(net, model_name, dataset, shuffle, device, overwrite=False):
    """Train a model on the training set of FashionMNIST.

    Args:
        net: Object of BaseNetwork
        model_name: (str) Name of the model, used for creating the checkpoint names
        max_epochs: Number of epochs we want to (maximally) train for
        patience: If the performance on the validation set has not improved for #patience epochs, we stop training early
        batch_size: Size of batches used in training
        overwrite: Determines how to handle the case when there already exists a checkpoint.
        If True, it will be overwritten. Otherwise, we skip training.
    """
    # CHECKPOINT_PATH = "./checkpoints/"
    # file_exists = os.path.isfile(_get_model_file(CHECKPOINT_PATH, model_name))
    # if file_exists and not overwrite:
    #     print(f'Model file of "{model_name}" already exists. Skipping training...')
    #     with open(_get_result_file(CHECKPOINT_PATH, model_name)) as f:
    #         results = json.load(f)
    # else:
    #     if file_exists:
    #         print("Model file exists, but will be overwritten...")
    device = getDevice(device=device)
    epochs_instance = net.epochs
    optimizer_instance = net.optimizer
    batch_size_instance = net.batch_size
    removed_attributes, net = get_removed_attributes_and_base_net(net)
    print(f"Removed attributes: {removed_attributes}")
    # Defining optimizer, loss and data loader
    # optimizer = optimizer_instance(net.parameters())
    optimizer = optimizer_handler(
        optimizer_name=optimizer_instance,
        params=net.parameters(),
        lr_mult=1.0,  # lr_mult_instance,
        sgd_momentum=0.9,  # sgd_momentum_instance,
    )
    print(f"Optimizer initialized: {optimizer}")
    loss_module = nn.CrossEntropyLoss()
    # train_loader_local = data.DataLoader(
    #     dataset, batch_size=batch_size, shuffle=True, drop_last=True, pin_memory=True
    # )
    train_loader_local, val_loader = create_train_val_data_loaders(
        dataset=dataset, batch_size=batch_size_instance, shuffle=shuffle
    )
    results = None
    val_scores = []
    train_losses, train_scores = [], []
    # best_val_epoch = -1
    for epoch in range(epochs_instance):
        train_acc, val_acc, epoch_losses = epoch_iteration(
            net, loss_module, optimizer, train_loader_local, val_loader, epoch, device
        )
        train_scores.append(train_acc)
        val_scores.append(val_acc)
        train_losses += epoch_losses

        # if len(val_scores) == 1 or val_acc > val_scores[best_val_epoch]:
        #     print("\t   (New best performance, saving model...)")
        #     save_model(net, CHECKPOINT_PATH, model_name)
        #     best_val_epoch = epoch

    # TODO: Move this part to the test method
    # if results is None:
    #     load_model(CHECKPOINT_PATH, model_name, net=net)
    #     test_acc = test_model(net, test_loader)
    #     results = {
    #         "test_acc": test_acc,
    #         "val_scores": val_scores,
    #         "train_losses": train_losses,
    #         "train_scores": train_scores,
    #     }
    #     with open(_get_result_file(CHECKPOINT_PATH, model_name), "w") as f:
    #         json.dump(results, f)

    results = {
        # "test_acc": test_acc,
        "val_scores": val_scores,
        "train_losses": train_losses,
        "train_scores": train_scores,
    }

    # Plot a curve of the validation accuracy
    sns.set()
    plt.plot([i for i in range(1, len(results["train_scores"]) + 1)], results["train_scores"], label="Train")
    plt.plot([i for i in range(1, len(results["val_scores"]) + 1)], results["val_scores"], label="Val")
    plt.xlabel("Epochs")
    plt.ylabel("Validation accuracy")
    plt.ylim(min(results["val_scores"]), max(results["train_scores"]) * 1.01)
    plt.title(f"Validation performance of {model_name}")
    plt.legend()
    plt.show()
    plt.close()

    # print((f" Test accuracy: {results['test_acc']*100.0:4.2f}% ").center(50, "=") + "\n")
    # return results

    # TODO: Return Validation Loss (and not accuracy)
    print(f"Validation Acc: {val_acc}")
    return val_acc, np.nan


def epoch_iteration(net, loss_module, optimizer, train_loader_local, val_loader, epoch, device, TQDM=False):
    # Training
    net.train()
    true_preds, count = 0.0, 0
    epoch_losses = []
    t = tqdm(train_loader_local, leave=False)
    for imgs, labels in t:
        imgs, labels = imgs.to(device), labels.to(device)
        optimizer.zero_grad()
        preds = net(imgs)
        loss = loss_module(preds, labels)
        loss.backward()
        optimizer.step()
        # Record statistics during training
        true_preds += (preds.argmax(dim=-1) == labels).sum().item()
        count += labels.shape[0]
        t.set_description(f"Epoch {epoch+1}: loss={loss.item():4.2f}")
        epoch_losses.append(loss.item())
    train_acc = true_preds / count

    # Validation
    val_acc = test_model(net, val_loader, device)
    print(f"[Epoch {epoch+1}] Training accuracy: {train_acc*100.0:05.2f}%, Validation accuracy: {val_acc*100.0:05.2f}%")
    return train_acc, val_acc, epoch_losses


def test_model(net, data_loader, device):
    """Test a model on a specified dataset.

    Args:
        net: Trained model of type BaseNetwork
        data_loader: DataLoader object of the dataset to test on (validation or test)
    """
    net.eval()
    true_preds, count = 0.0, 0
    for imgs, labels in data_loader:
        imgs, labels = imgs.to(device), labels.to(device)
        with torch.no_grad():
            preds = net(imgs).argmax(dim=-1)
            true_preds += (preds == labels).sum().item()
            count += labels.shape[0]
    test_acc = true_preds / count
    return test_acc
