import sqlite3
import os
import importlib



dbMap = {
    "dbiu.bahn":1,
    "dbiu.bayern":2,
    "dbiu.bundestag": 3,
    "dbiu.bundestag_einfach": 4,
    "dbiu.film_fernsehen": 5,
    "dbiu.haushaltsausstattung": 6,
    "dbiu.straftaten": 7,
    "dbiu.straftaten_einfach": 8,
    "dbiu.kunstsammlung": 9,
    "dbiu.ladepunkte": 10,
    "dbiu.laenderspiele": 11,
    "dbiu.lebensmittel": 12,
    "dbiu.schulstatistik": 13,
    "dbiu.studierende": 14,
    "dbiu.unfallstatistik": 15,
    "dbiu.videospiele_einfach": 16,
    "dbiu.videospiele": 17,
    "dbiu.wetterdaten": 18,
    "dbiu.terra": 19
}

__db=""
dbiu = 'dbiu.'

def setDBName(database):
    global __db

    if isinstance(database, str) and database.startswith(dbiu):
        db_version = dbMap.get(database)
    elif isinstance(database, int):
        db_version = database
        database = dbiu+str(database)
    elif isinstance(database, str):
        db_version = None
    else:
        raise Exception("Invalid database selected")
        
    if(db_version is not None):
        os.system('pip install dbiu_databases=='+str(db_version))
        try:
            import dbiu_databases
        except ImportError:
            raise Exception("Invalid database selected")
    print("Setting DB to: " + database)
    __db = database

    
def run(sql: str):
    if __db.startswith(dbiu) and not __db.endswith(".db"):
        with importlib.resources.path('dbiu_databases', "base.db") as db_path:
            return __run(sql, db_path)
    else:
        return __run(sql, __db)

def __run(sql: str, db: str):
    with sqlite3.connect(db) as con:
        cur = con.cursor()
        cur.execute(sql)
        con.commit()
        return cur

def runFromFile(path: str, limit: int = None):
    sql = getSQLFromFile(path)
    if not sql:
        raise Exception("\nSQL-Datei ist leer. Aufgabe wurde noch nicht bearbeitet.")


    if sql.lower().find("drop") != -1:
        raise Exception("Guter Versuch, aber die Datenbank wird bei jedem Upload zurückgesetzt ;)")

    try:
        res = run(sql)
    except Exception as e:
        raise Exception(f"\n\nSyntax-Fehler in der SQL-Abfrage:\n{str(e)}")

    headers = [h[0] for h in res.description] if res.description else []
    if limit:
        rows = res.fetchmany(limit)
    else:
        rows = res.fetchall() if res else []

    return headers, rows


def runAndGetStringTable_fromFile(path: str, count: int = 5, max_line_length: int = 85):
    try:
        limit = 1000
        headers, rows = runFromFile(path, limit)
        result_count = len(rows)
        rows = rows[:count]
        s = ""

        matrix = [] #[[]*(len(rows)+1)]*len(headers)

        for col in range(len(headers)):
            matrix.append([])
            matrix[-1].append(headers[col])

        if rows is not None:
            for col in range(len(matrix)):
                for row in range(0, len(rows)):
                    if len(rows[row]) > col:
                        matrix[col].append(rows[row][col])

            for col in range(len(matrix)):
                max_length = max([len(str(x)) for x in matrix[col]])

                for row in range(0, len(matrix[col])):
                    matrix[col][row] = str(matrix[col][row]).ljust(max_length)

            normalized_rows = []
            spacing = "  "
            spacing_length = len(spacing) * (len(matrix[0]) - 1)

            line_length = sum([len(x[0]) for x in matrix]) + spacing_length
            
            

            if line_length > max_line_length:
                max_col_length = int(max_line_length/len(matrix))

                for col in range(len(matrix)):
                    for row in range(0, len(matrix[col])):
                        cut_content = len(matrix[col][row].strip()) > max_col_length
                        if cut_content:
                            matrix[col][row] = matrix[col][row][:max_col_length-2] + ".."
                        else:
                            matrix[col][row] = matrix[col][row][:max_col_length]


            for row in range(min(count+1, len(matrix[0]))):
                normalized_rows.append(spacing.join([str(matrix[col][row]) for col in range(len(matrix))]))
            normalized_rows.insert(1, len(normalized_rows[0])*"-")

            s = "\n".join(normalized_rows[:count+2])
            s += "\n" + normalized_rows[1]

            if (result_count > count):
                if result_count == limit:
                    s += "\n... mehr als " + str(limit) + " Zeilen"
                else:
                    s += "\n... " + str(result_count - count) + " weitere Zeilen"
            else:
                s += "\n... keine weiteren Zeilen"
        s = "\n\nDiese Meldung sagt nichts über die Korrektheit der Abgabe aus!\nDie ersten " + str(
            count) + " Zeilen des Ergebnisses der SQL-Abfrage:\n\n" + s

        return s.replace("None", "NULL")
    except Exception as ex:
        print(ex)
        raise ex

def getSQLFromFile(path: str):
    try:
        with open(path, "r") as f:
            s = f.read()
            if not s:
                raise Exception("\nSQL-Datei ist leer. Aufgabe wurde noch nicht bearbeitet.")
            return s
    except FileNotFoundError:
        raise Exception(f"\nSQL-Datei nicht gefunden! Überprüfe, dass der Name korrekt ist ({path.split('/')[-1]}) und die Datei nicht gelöscht oder in einen Unterordner verschoben wurde.")



def getWorkingDir():
    return os.getcwd()


def getWorkingDirFiles():
    return os.listdir()

def mapDatabaseTypes(t):
    # Map SQLite types to Python types
    type_mapping = {
        "INTEGER": "int",
        "TEXT": "str",
        "VARCHAR": "str",
        "REAL": "float",
        "BLOB": "bytes",
        "NUMERIC": "float",
        "DOUBLE": "float"
    }
    t = t.upper()
    t = type_mapping.get(t, t)

    if "INT" in t:
        t = "int"
    elif "VARCHAR" in t:
        t = "str"
    elif "CHAR" in t:
        t = "str"

    return t




def getTableDict():
    res = run("SELECT name FROM sqlite_master WHERE type='table';")
    tables = res.fetchall()

    table_dict = {}
    for table in tables:
        res = run(f"PRAGMA table_info({table[0]})")
        r = res.fetchall()
        if table[0] == "sqlite_sequence":
            continue
        table_dict[table[0]] = [[c[1], mapDatabaseTypes(c[2])] for c in r]

    return table_dict