import re
from pathlib import Path
from typing import Any


def clean_models(raw_path: Path, out_path: Path):
    content = raw_path.read_text()

    # Drop the raw Base declaration block
    content = re.sub(
        r"^class Base\(DeclarativeBase\):.*?^(?=\w)",
        "",
        content,
        flags=re.DOTALL | re.MULTILINE,
    )

    # Drop all autogenerated timestamp columns
    content = re.sub(
        r"^\s+(?:created_at|updated_at|deleted_at):\s*Mapped\[.*?\]\s*=\s*mapped_column\([\s\S]*?\)\s*\n",
        "",
        content,
        flags=re.MULTILINE,
    )

    out_path.write_text(content)


def clean_schema_file(
    path: Path,
    field_map: dict[str, str] | None = None,
    custom_imports: dict[str, str] | None = None,
):
    """
    Cleans a single Pydantic schema file by replacing generic JSON/dict types
    with specific Pydantic models and adding necessary imports.
    Also performs default cleaning operations.
    """
    field_map = field_map or {}
    custom_imports = custom_imports or {}
    text = path.read_text()
    changed = False

    # Default cleaning for common patterns
    text, n = re.subn(r"Union\[(\w+), None\]", r"\1 | None", text)
    if n > 0:
        changed = True

    for field, model in field_map.items():
        # Replace the generic type for the field with the specific model
        pattern = rf"({field}:\s*)list\[dict\[str, Any\]\] \| dict\[str, Any\] \| None"
        repl = rf"\1{model} | None"
        new_text, n = re.subn(pattern, repl, text)

        if n > 0:
            text = new_text
            changed = True

            # Add the corresponding import if it's not already in the file
            if model in custom_imports:
                import_line = custom_imports[model]
                if import_line not in text:
                    # Find the last import statement to insert after
                    import_matches = list(
                        re.finditer(r"^(?:from|import) .+", text, re.MULTILINE)
                    )
                    if import_matches:
                        last_import = import_matches[-1]
                        insert_pos = last_import.end()
                        text = f"{text[:insert_pos]}\n{import_line}{text[insert_pos:]}"
                    else:
                        # No imports found, add at the top of the file
                        text = f"{import_line}\n\n{text}"

    if changed:
        path.write_text(text)
        print(f"Updated schema: {path}")


def clean_schemas(
    schemas_dir: Path,
    custom_jsonb_models: dict[str, Any] | None = None,
    custom_imports: dict[str, str] | None = None,
):
    """
    Iterates over schema files in the specified directory, applying
    default cleaning and custom model mappings if provided.
    """
    custom_jsonb_models = custom_jsonb_models or {}
    custom_imports = custom_imports or {}

    for schema_file in schemas_dir.glob("*.py"):
        if schema_file.name == "__init__.py":
            continue

        table_name = schema_file.stem
        field_map = custom_jsonb_models.get(table_name)

        # Always run clean_schema_file for default cleaning.
        # Pass field_map and custom_imports if they exist for the table.
        clean_schema_file(schema_file, field_map, custom_imports)
