"""Integration tests for OracleDB migration workflow."""

import tempfile
from pathlib import Path

import pytest
from pytest_databases.docker.oracle import OracleService

from sqlspec.adapters.oracledb.config import OracleAsyncConfig, OracleSyncConfig
from sqlspec.migrations.commands import AsyncMigrationCommands, create_migration_commands

pytestmark = pytest.mark.xdist_group("oracle")


def test_oracledb_sync_migration_full_workflow(oracle_23ai_service: OracleService) -> None:
    """Test full OracleDB sync migration workflow: init -> create -> upgrade -> downgrade."""

    test_id = "oracledb_sync_full"
    migration_table = f"sqlspec_migrations_{test_id}"
    users_table = f"users_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleSyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = create_migration_commands(config)

        commands.init(str(migration_dir), package=True)

        assert migration_dir.exists()
        assert (migration_dir / "__init__.py").exists()

        migration_content = f'''"""Initial schema migration."""


def up():
    """Create users table."""
    return ["""
        CREATE TABLE {users_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            name VARCHAR2(255) NOT NULL,
            email VARCHAR2(255) UNIQUE NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """]


def down():
    """Drop users table."""
    return ["DROP TABLE {users_table}"]
'''

        migration_file = migration_dir / "0001_create_users.py"
        migration_file.write_text(migration_content)

        try:
            commands.upgrade()

            with config.provide_session() as driver:
                result = driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{users_table.upper()}'"
                )
                assert len(result.data) == 1

                driver.execute(
                    f"INSERT INTO {users_table} (name, email) VALUES (:1, :2)", ("John Doe", "john@example.com")
                )

                users_result = driver.execute(f"SELECT * FROM {users_table}")
                assert len(users_result.data) == 1
                assert users_result.data[0]["NAME"] == "John Doe"
                assert users_result.data[0]["EMAIL"] == "john@example.com"

            commands.downgrade("base")

            with config.provide_session() as driver:
                result = driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{users_table.upper()}'"
                )
                assert len(result.data) == 0
        finally:
            if config.pool_instance:
                config.close_pool()


async def test_oracledb_async_migration_full_workflow(oracle_23ai_service: OracleService) -> None:
    """Test full OracleDB async migration workflow: init -> create -> upgrade -> downgrade."""

    test_id = "oracledb_async_full"
    migration_table = f"sqlspec_migrations_{test_id}"
    users_table = f"users_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleAsyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
                "min": 1,
                "max": 5,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = AsyncMigrationCommands(config)

        await commands.init(str(migration_dir), package=True)

        assert migration_dir.exists()
        assert (migration_dir / "__init__.py").exists()

        migration_content = f'''"""Initial schema migration."""


def up():
    """Create users table."""
    return ["""
        CREATE TABLE {users_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            name VARCHAR2(255) NOT NULL,
            email VARCHAR2(255) UNIQUE NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """]


def down():
    """Drop users table."""
    return ["DROP TABLE {users_table}"]
'''

        migration_file = migration_dir / "0001_create_users.py"
        migration_file.write_text(migration_content)

        try:
            await commands.upgrade()

            async with config.provide_session() as driver:
                result = await driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{users_table.upper()}'"
                )
                assert len(result.data) == 1

                await driver.execute(
                    f"INSERT INTO {users_table} (name, email) VALUES (:1, :2)", ("John Doe", "john@example.com")
                )

                users_result = await driver.execute(f"SELECT * FROM {users_table}")
                assert len(users_result.data) == 1
                assert users_result.data[0]["NAME"] == "John Doe"
                assert users_result.data[0]["EMAIL"] == "john@example.com"

            await commands.downgrade("base")

            async with config.provide_session() as driver:
                result = await driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{users_table.upper()}'"
                )
                assert len(result.data) == 0
        finally:
            if config.pool_instance:
                await config.close_pool()


def test_oracledb_sync_multiple_migrations_workflow(oracle_23ai_service: OracleService) -> None:
    """Test OracleDB sync workflow with multiple migrations: create -> apply both -> downgrade one -> downgrade all."""

    test_id = "oracledb_sync_multiple"
    migration_table = f"sqlspec_migrations_{test_id}"
    users_table = f"users_{test_id}"
    posts_table = f"posts_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleSyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = create_migration_commands(config)

        commands.init(str(migration_dir), package=True)

        migration1_content = f'''"""Create users table."""


def up():
    """Create users table."""
    return ["""
        CREATE TABLE {users_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            name VARCHAR2(255) NOT NULL,
            email VARCHAR2(255) UNIQUE NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """]


def down():
    """Drop users table."""
    return ["DROP TABLE {users_table}"]
'''
        (migration_dir / "0001_create_users.py").write_text(migration1_content)

        migration2_content = f'''"""Create posts table."""


def up():
    """Create posts table."""
    return ["""
        CREATE TABLE {posts_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            title VARCHAR2(255) NOT NULL,
            content CLOB,
            user_id NUMBER,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            CONSTRAINT fk_{posts_table}_user FOREIGN KEY (user_id) REFERENCES {users_table}(id)
        )
    """]


def down():
    """Drop posts table."""
    return ["DROP TABLE {posts_table}"]
'''
        (migration_dir / "0002_create_posts.py").write_text(migration2_content)

        try:
            commands.upgrade()

            with config.provide_session() as driver:
                users_result = driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{users_table.upper()}'"
                )
                posts_result = driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{posts_table.upper()}'"
                )
                assert len(users_result.data) == 1
                assert len(posts_result.data) == 1

                driver.execute(
                    f"INSERT INTO {users_table} (name, email) VALUES (:1, :2)", ("John Doe", "john@example.com")
                )
                driver.execute(
                    f"INSERT INTO {posts_table} (title, content, user_id) VALUES (:1, :2, :3)",
                    ("Test Post", "This is a test post", 1),
                )

            commands.downgrade("0001")

            with config.provide_session() as driver:
                users_result = driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{users_table.upper()}'"
                )
                posts_result = driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{posts_table.upper()}'"
                )
                assert len(users_result.data) == 1
                assert len(posts_result.data) == 0

            commands.downgrade("base")

            with config.provide_session() as driver:
                users_result = driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name IN ('{users_table.upper()}', '{posts_table.upper()}')"
                )
                assert len(users_result.data) == 0
        finally:
            if config.pool_instance:
                config.close_pool()


async def test_oracledb_async_multiple_migrations_workflow(oracle_23ai_service: OracleService) -> None:
    """Test OracleDB async workflow with multiple migrations: create -> apply both -> downgrade one -> downgrade all."""

    test_id = "oracledb_async_multiple"
    migration_table = f"sqlspec_migrations_{test_id}"
    users_table = f"users_{test_id}"
    posts_table = f"posts_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleAsyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
                "min": 1,
                "max": 5,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = AsyncMigrationCommands(config)

        await commands.init(str(migration_dir), package=True)

        migration1_content = f'''"""Create users table."""


def up():
    """Create users table."""
    return ["""
        CREATE TABLE {users_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            name VARCHAR2(255) NOT NULL,
            email VARCHAR2(255) UNIQUE NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """]


def down():
    """Drop users table."""
    return ["DROP TABLE {users_table}"]
'''
        (migration_dir / "0001_create_users.py").write_text(migration1_content)

        migration2_content = f'''"""Create posts table."""


def up():
    """Create posts table."""
    return ["""
        CREATE TABLE {posts_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            title VARCHAR2(255) NOT NULL,
            content CLOB,
            user_id NUMBER,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            CONSTRAINT fk_posts_user FOREIGN KEY (user_id) REFERENCES {users_table}(id)
        )
    """]


def down():
    """Drop posts table."""
    return ["DROP TABLE {posts_table}"]
'''
        (migration_dir / "0002_create_posts.py").write_text(migration2_content)

        try:
            await commands.upgrade()

            async with config.provide_session() as driver:
                users_result = await driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{users_table.upper()}'"
                )
                posts_result = await driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{posts_table.upper()}'"
                )
                assert len(users_result.data) == 1
                assert len(posts_result.data) == 1

                await driver.execute(
                    f"INSERT INTO {users_table} (name, email) VALUES (:1, :2)", ("John Doe", "john@example.com")
                )
                await driver.execute(
                    f"INSERT INTO {posts_table} (title, content, user_id) VALUES (:1, :2, :3)",
                    ("Test Post", "This is a test post", 1),
                )

            await commands.downgrade("0001")

            async with config.provide_session() as driver:
                users_result = await driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{users_table.upper()}'"
                )
                posts_result = await driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name = '{posts_table.upper()}'"
                )
                assert len(users_result.data) == 1
                assert len(posts_result.data) == 0

            await commands.downgrade("base")

            async with config.provide_session() as driver:
                users_result = await driver.execute(
                    f"SELECT table_name FROM user_tables WHERE table_name IN ('{users_table.upper()}', '{posts_table.upper()}')"
                )
                assert len(users_result.data) == 0
        finally:
            if config.pool_instance:
                await config.close_pool()


def test_oracledb_sync_migration_current_command(oracle_23ai_service: OracleService) -> None:
    """Test the current migration command shows correct version for OracleDB sync."""

    test_id = "oracledb_sync_current"
    migration_table = f"sqlspec_migrations_{test_id}"
    users_table = f"users_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleSyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = create_migration_commands(config)

        try:
            commands.init(str(migration_dir), package=True)

            current_version = commands.current()
            assert current_version is None or current_version == "base"

            migration_content = f'''"""Initial schema migration."""


def up():
    """Create users table."""
    return ["""
        CREATE TABLE {users_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            name VARCHAR2(255) NOT NULL
        )
    """]


def down():
    """Drop users table."""
    return ["DROP TABLE {users_table}"]
'''
            (migration_dir / "0001_create_users.py").write_text(migration_content)

            commands.upgrade()

            current_version = commands.current()
            assert current_version == "0001"

            commands.downgrade("base")

            current_version = commands.current()
            assert current_version is None or current_version == "base"
        finally:
            if config.pool_instance:
                config.close_pool()


async def test_oracledb_async_migration_current_command(oracle_23ai_service: OracleService) -> None:
    """Test the current migration command shows correct version for OracleDB async."""

    test_id = "oracledb_async_current"
    migration_table = f"sqlspec_migrations_{test_id}"
    users_table = f"users_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleAsyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
                "min": 1,
                "max": 5,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = AsyncMigrationCommands(config)

        try:
            await commands.init(str(migration_dir), package=True)

            current_version = await commands.current()
            assert current_version is None or current_version == "base"

            migration_content = f'''"""Initial schema migration."""


def up():
    """Create users table."""
    return ["""
        CREATE TABLE {users_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            name VARCHAR2(255) NOT NULL
        )
    """]


def down():
    """Drop users table."""
    return ["DROP TABLE {users_table}"]
'''
            (migration_dir / "0001_create_users.py").write_text(migration_content)

            await commands.upgrade()

            current_version = await commands.current()
            assert current_version == "0001"

            await commands.downgrade("base")

            current_version = await commands.current()
            assert current_version is None or current_version == "base"
        finally:
            if config.pool_instance:
                await config.close_pool()


def test_oracledb_sync_migration_error_handling(oracle_23ai_service: OracleService) -> None:
    """Test OracleDB sync migration error handling."""

    test_id = "oracledb_sync_error"
    migration_table = f"sqlspec_migrations_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleSyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = create_migration_commands(config)

        try:
            commands.init(str(migration_dir), package=True)

            migration_content = '''"""Migration with invalid SQL."""


def up():
    """Create table with invalid SQL."""
    return ["CREATE INVALID SQL STATEMENT"]


def down():
    """Drop table."""
    return ["DROP TABLE invalid_table"]
'''
            (migration_dir / "0001_invalid.py").write_text(migration_content)

            with pytest.raises(Exception):
                commands.upgrade()

            with config.provide_session() as driver:
                try:
                    result = driver.execute(f"SELECT COUNT(*) as count FROM {migration_table}")
                    assert result.data[0]["COUNT"] == 0
                except Exception:
                    pass
        finally:
            if config.pool_instance:
                config.close_pool()


async def test_oracledb_async_migration_error_handling(oracle_23ai_service: OracleService) -> None:
    """Test OracleDB async migration error handling."""

    test_id = "oracledb_async_error"
    migration_table = f"sqlspec_migrations_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleAsyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
                "min": 1,
                "max": 5,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = AsyncMigrationCommands(config)

        try:
            await commands.init(str(migration_dir), package=True)

            migration_content = '''"""Migration with invalid SQL."""


def up():
    """Create table with invalid SQL."""
    return ["CREATE INVALID SQL STATEMENT"]


def down():
    """Drop table."""
    return ["DROP TABLE invalid_table"]
'''
            (migration_dir / "0001_invalid.py").write_text(migration_content)

            with pytest.raises(Exception):
                await commands.upgrade()

            async with config.provide_session() as driver:
                try:
                    result = await driver.execute(f"SELECT COUNT(*) as count FROM {migration_table}")
                    assert result.data[0]["COUNT"] == 0
                except Exception:
                    pass
        finally:
            if config.pool_instance:
                await config.close_pool()


def test_oracledb_sync_migration_with_transactions(oracle_23ai_service: OracleService) -> None:
    """Test OracleDB sync migrations work properly with transactions."""

    test_id = "oracledb_sync_trans"
    migration_table = f"sqlspec_migrations_{test_id}"
    users_table = f"users_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleSyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = create_migration_commands(config)

        try:
            commands.init(str(migration_dir), package=True)

            migration_content = f'''"""Initial schema migration."""


def up():
    """Create users table."""
    return ["""
        CREATE TABLE {users_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            name VARCHAR2(255) NOT NULL,
            email VARCHAR2(255) UNIQUE NOT NULL
        )
    """]


def down():
    """Drop users table."""
    return ["DROP TABLE {users_table}"]
'''
            (migration_dir / "0001_create_users.py").write_text(migration_content)

            commands.upgrade()

            with config.provide_session() as driver:
                driver.begin()
                try:
                    driver.execute(
                        f"INSERT INTO {users_table} (name, email) VALUES (:1, :2)",
                        ("Transaction User", "trans@example.com"),
                    )

                    result = driver.execute(f"SELECT * FROM {users_table} WHERE name = 'Transaction User'")
                    assert len(result.data) == 1
                    driver.commit()
                except Exception:
                    driver.rollback()
                    raise

                result = driver.execute(f"SELECT * FROM {users_table} WHERE name = 'Transaction User'")
                assert len(result.data) == 1

            with config.provide_session() as driver:
                driver.begin()
                try:
                    driver.execute(
                        f"INSERT INTO {users_table} (name, email) VALUES (:1, :2)",
                        ("Rollback User", "rollback@example.com"),
                    )

                    raise Exception("Intentional rollback")
                except Exception:
                    driver.rollback()

                result = driver.execute(f"SELECT * FROM {users_table} WHERE name = 'Rollback User'")
                assert len(result.data) == 0
        finally:
            if config.pool_instance:
                config.close_pool()


async def test_oracledb_async_migration_with_transactions(oracle_23ai_service: OracleService) -> None:
    """Test OracleDB async migrations work properly with transactions."""

    test_id = "oracledb_async_trans"
    migration_table = f"sqlspec_migrations_{test_id}"
    users_table = f"users_{test_id}"

    with tempfile.TemporaryDirectory() as temp_dir:
        migration_dir = Path(temp_dir) / "migrations"

        config = OracleAsyncConfig(
            pool_config={
                "host": oracle_23ai_service.host,
                "port": oracle_23ai_service.port,
                "service_name": oracle_23ai_service.service_name,
                "user": oracle_23ai_service.user,
                "password": oracle_23ai_service.password,
                "min": 1,
                "max": 5,
            },
            migration_config={"script_location": str(migration_dir), "version_table_name": migration_table},
        )
        commands = AsyncMigrationCommands(config)

        try:
            await commands.init(str(migration_dir), package=True)

            migration_content = f'''"""Initial schema migration."""


def up():
    """Create users table."""
    return ["""
        CREATE TABLE {users_table} (
            id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
            name VARCHAR2(255) NOT NULL,
            email VARCHAR2(255) UNIQUE NOT NULL
        )
    """]


def down():
    """Drop users table."""
    return ["DROP TABLE {users_table}"]
'''
            (migration_dir / "0001_create_users.py").write_text(migration_content)

            await commands.upgrade()

            async with config.provide_session() as driver:
                await driver.begin()
                try:
                    await driver.execute(
                        f"INSERT INTO {users_table} (name, email) VALUES (:1, :2)",
                        ("Transaction User", "trans@example.com"),
                    )

                    result = await driver.execute(f"SELECT * FROM {users_table} WHERE name = 'Transaction User'")
                    assert len(result.data) == 1
                    await driver.commit()
                except Exception:
                    await driver.rollback()
                    raise

                result = await driver.execute(f"SELECT * FROM {users_table} WHERE name = 'Transaction User'")
                assert len(result.data) == 1

            async with config.provide_session() as driver:
                await driver.begin()
                try:
                    await driver.execute(
                        f"INSERT INTO {users_table} (name, email) VALUES (:1, :2)",
                        ("Rollback User", "rollback@example.com"),
                    )

                    raise Exception("Intentional rollback")
                except Exception:
                    await driver.rollback()

                result = await driver.execute(f"SELECT * FROM {users_table} WHERE name = 'Rollback User'")
                assert len(result.data) == 0
        finally:
            if config.pool_instance:
                await config.close_pool()
