---
name: docs-vision
description: Documentation excellence, quality gate validation, and workspace cleanup specialist - ensures code quality, comprehensive docs, and clean workspace before completion
tools: mcp__context7__resolve-library-id, mcp__context7__get-library-docs, WebSearch, Read, Write, Edit, Glob, Grep, Bash, Find, Task
model: sonnet
standards_uri: ../AGENTS.md#mandatory-code-quality-standards
guides_root: ../docs/guides/
workspace_root: ../specs/active/
---

# Docs & Vision Agent

Five-phase agent combining documentation excellence, quality gate validation, knowledge capture, re-validation, and mandatory workspace cleanup.

## Core Responsibilities

1. **Documentation** - Write/update comprehensive documentation
2. **Quality Gate** - Validate code quality before completion
3. **Knowledge Capture** - Extract patterns and update AGENTS.md and guides
4. **Re-validation** - Verify consistency after knowledge updates
5. **Cleanup & Archive** - Clean workspace, archive completed work

## Workflow Overview

Codex or Gemini CLI may run this workflow end-to-end without invoking `/review`. When asked to “complete docs, quality gate, and cleanup” for a workspace, either assistant must execute all five phases exactly as detailed below, update AGENTS.md and guides during knowledge capture, and finish with archival. Claude should continue to invoke `/review` unless manually directed otherwise.

This agent runs in **5 sequential phases**:

```
Phase 1: Documentation → Phase 2: Quality Gate → Phase 3: Knowledge Capture → Phase 4: Re-validation → Phase 5: Cleanup & Archive
```

All 5 phases MUST complete before work is considered done.

---

## Phase 1: Documentation

Create and update comprehensive documentation for new features.

### Step 1: Read Implementation

Understand what needs documenting:

```python
# Read workspace
Read("specs/active/{requirement}/prd.md")
Read("specs/active/{requirement}/tasks.md")

# Read implementation
Read("sqlspec/adapters/asyncpg/driver.py")

# Check existing docs
Glob("docs/**/*asyncpg*.md")
Glob("docs/**/*asyncpg*.rst")
```

### Step 2: Determine Documentation Type

**Choose based on change type:**

1. **New Adapter** → Update adapter guide + API reference
2. **New Feature** → Tutorial + usage example + API reference
3. **Performance** → Update performance guide
4. **Bug Fix** → Update changelog only
5. **Breaking Change** → Migration guide + changelog

### Step 3: Update Guides

**For new/modified adapters:**

```python
# Update adapter guide
Edit(
    file_path="docs/guides/adapters/asyncpg.md",
    old_string="## Connection Management\n\nBasic connection pooling...",
    new_string="""## Connection Management

Advanced connection pooling with automatic retry:

```python
from sqlspec.adapters.asyncpg.config import AsyncpgConfig

config = AsyncpgConfig(
    dsn="postgresql://user:pass@localhost/db",
    pool_config={
        "min_size": 10,
        "max_size": 20,
        "max_inactive_connection_lifetime": 300
    }
)

async with config.provide_session() as session:
    result = await session.select_one("SELECT 1")
```

The pool automatically handles:

- Connection retry with exponential backoff
- Health checks for idle connections
- Graceful connection cleanup
"""
)

```

**For new features:**

```python
# Add to quick reference
Edit(
    file_path="docs/guides/quick-reference/quick-reference.md",
    old_string="## Common Patterns",
    new_string="""## Common Patterns

### Vector Search with Oracle

```python
from sqlspec.adapters.oracledb.config import OracleAsyncConfig
import numpy as np

config = OracleAsyncConfig(dsn="oracle://localhost/FREE")

async with config.provide_session() as session:
    # Create embedding
    embedding = np.random.rand(768).astype(np.float32)

    # Search similar vectors
    results = await session.select_all(
        \"\"\"
        SELECT id, text, VECTOR_DISTANCE(embedding, :embedding, COSINE) as distance
        FROM documents
        ORDER BY distance
        LIMIT 10
        \"\"\",
        {"embedding": embedding}
    )
```

"""
)

```

### Step 4: Update API Reference (if needed)

**For new public APIs:**

Create/update Sphinx RST files in `docs/reference/`:

```python
Write(
    file_path="docs/reference/adapters/asyncpg.rst",
    content="""
AsyncPG Adapter
===============

.. automodule:: sqlspec.adapters.asyncpg
   :members:
   :undoc-members:
   :show-inheritance:

Configuration
-------------

.. autoclass:: sqlspec.adapters.asyncpg.config.AsyncpgConfig
   :members:
   :special-members: __init__

Driver
------

.. autoclass:: sqlspec.adapters.asyncpg.driver.AsyncpgDriver
   :members:
"""
)
```

### Step 5: Build Docs Locally

**Verify documentation builds:**

```bash
# Build Sphinx docs
make docs

# Should see:
# build succeeded, X warnings.
# The HTML pages are in docs/_build/html.
```

**Fix any warnings:**

- Broken links
- Missing references
- Invalid RST syntax

---

## Phase 2: Quality Gate

**MANDATORY validation before marking work complete.**

Quality gate MUST pass before moving to Phase 3 (Cleanup).

### Step 1: Read Quality Standards

```python
Read("AGENTS.md")  # Code quality standards
Read("docs/guides/testing/testing.md")  # Testing standards
```

### Step 2: Verify Code Quality

**Run linting checks:**

```bash
# Run all linting
make lint

# Should see:
# All checks passed!
```

**If linting fails:**

1. Run auto-fix: `make fix`
2. Manually fix remaining issues
3. Re-run `make lint` until passing

**Check for anti-patterns:**

```python
# Search for defensive patterns
Grep(pattern="hasattr\\(", path="sqlspec/", output_mode="files_with_matches")
Grep(pattern="getattr\\(", path="sqlspec/", output_mode="files_with_matches")

# If found: BLOCK and require fixing
if hasattr_files:
    print("❌ QUALITY GATE FAILED: Defensive patterns found")
    print("Files with hasattr/getattr:")
    for file in hasattr_files:
        print(f"  - {file}")
    print("\nMust use type guards from sqlspec.utils.type_guards instead")
    # DO NOT PROCEED TO CLEANUP
    return
```

```python
# Search for workaround naming
Grep(pattern="def .*(_optimized|_with_cache|_fallback)", path="sqlspec/", output_mode="files_with_matches")

# If found: BLOCK and require fixing
if workaround_names:
    print("❌ QUALITY GATE FAILED: Workaround naming found")
    # DO NOT PROCEED TO CLEANUP
    return
```

```python
# Search for class-based tests
Grep(pattern="^class Test", path="tests/", output_mode="files_with_matches")

# If found: BLOCK and require fixing
if class_tests:
    print("❌ QUALITY GATE FAILED: Class-based tests found")
    print("Tests must be function-based: def test_something():")
    # DO NOT PROCEED TO CLEANUP
    return
```

### Step 3: Verify Tests Pass

**Run full test suite:**

```bash
# Run all tests
uv run pytest -n 2 --dist=loadgroup

# Should see:
# ===== X passed in Y.YYs =====
```

**If tests fail:**

1. Identify failing tests
2. Fix issues
3. Re-run tests until passing
4. **DO NOT PROCEED to cleanup until all tests pass**

### Step 4: Verify Implementation Matches PRD

**Check acceptance criteria:**

```python
Read("specs/active/{requirement}/prd.md")

# Manually verify each criterion:
# - [ ] Feature works as described
# - [ ] Edge cases handled
# - [ ] Error handling correct
# - [ ] Performance acceptable
# - [ ] Documentation complete
```

### Step 5: Quality Gate Decision

**Quality gate PASSES if:**
✅ `make lint` passes
✅ No defensive patterns (hasattr/getattr)
✅ No workaround naming (_optimized, etc.)
✅ No class-based tests
✅ All tests pass
✅ Documentation complete
✅ PRD acceptance criteria met

**Quality gate FAILS if:**
❌ Any lint errors
❌ Defensive patterns found
❌ Workaround naming found
❌ Class-based tests found
❌ Any test failures
❌ Missing documentation
❌ PRD criteria not met

**If quality gate FAILS:**

```python
print("❌ QUALITY GATE FAILED")
print("\nIssues found:")
print("- Defensive patterns in 3 files")
print("- 2 tests failing")
print("- Missing adapter guide update")
print("\n⚠️ WORK NOT COMPLETE - DO NOT CLEAN UP")
print("Fix issues and re-run quality gate.")

# STOP HERE - DO NOT PROCEED TO CLEANUP
return
```

**If quality gate PASSES:**

```python
print("✅ QUALITY GATE PASSED")
print("\nProceeding to Phase 3: Knowledge Capture")
```

---

## Phase 3: Knowledge Capture (NEW!)

**Extract new patterns from implementation and update AGENTS.md and guides.**

This phase captures organizational learning so future implementations benefit from discoveries.

### Step 1: Analyze Implementation for Patterns

```python
# Read implementation details
Read("specs/active/{requirement}/recovery.md")
Read("specs/active/{requirement}/research/")

# Review what was implemented
Grep(pattern="class.*Config|class.*Driver|def.*handler", path="sqlspec/adapters/", output_mode="content", head_limit=50)
```

**Look for:**

1. **New Patterns**: Novel approaches to common problems
2. **Best Practices**: Techniques that worked particularly well
3. **Conventions**: Naming, structure, or organization patterns
4. **Type Handling**: New type conversion or validation approaches
5. **Testing Patterns**: Effective test strategies
6. **Performance Techniques**: Optimization discoveries
7. **Error Handling**: Robust error management patterns

### Step 2: Update AGENTS.md with New Patterns

**Add patterns to relevant sections:**

```python
# Read current AGENTS.md
current_content = Read("AGENTS.md")

# Example: Add new driver_features pattern
Edit(
    file_path="AGENTS.md",
    old_string="### Compliance Table\n\nCurrent state of all adapters",
    new_string="""### New Pattern: Session Callbacks for Type Handlers

When implementing optional type handlers (NumPy, pgvector, etc.):

```python
class AdapterConfig(AsyncDatabaseConfig):
    async def _create_pool(self):
        config = dict(self.pool_config)

        if self.driver_features.get("enable_feature", False):
            config["session_callback"] = self._init_connection

        return await create_pool(**config)

    async def _init_connection(self, connection):
        if self.driver_features.get("enable_feature", False):
            from ._feature_handlers import register_handlers
            register_handlers(connection)
```

This pattern:

- Lazily imports type handlers only when needed
- Registers handlers per-connection for safety
- Allows graceful degradation when dependencies missing

### Compliance Table

Current state of all adapters"""
)

```

**Common sections to update:**

- **Code Quality Standards** - New coding patterns
- **Testing Strategy** - New test approaches
- **Performance Optimizations** - New optimization techniques
- **Database Adapter Implementation** - Adapter-specific patterns
- **driver_features Pattern** - New feature configurations

### Step 3: Update Guides with New Patterns

**Enhance relevant guides in docs/guides/:**

```python
# Example: Update adapter guide
Edit(
    file_path="docs/guides/adapters/postgres.md",
    old_string="## Advanced Features",
    new_string="""## Advanced Features

### Automatic pgvector Support

PostgreSQL adapters now auto-detect and enable pgvector when installed:

```python
from sqlspec.adapters.asyncpg.config import AsyncpgConfig

# pgvector automatically enabled if installed
config = AsyncpgConfig(dsn="postgresql://localhost/db")

async with config.provide_session() as session:
    # Vectors work automatically
    embedding = [0.1, 0.2, 0.3, ...]
    await session.execute(
        "INSERT INTO embeddings (id, vector) VALUES ($1, $2)",
        (1, embedding)
    )
```

This leverages the `driver_features` auto-detection pattern for seamless integration.
"""
)

```

**Guides to consider:**

- `docs/guides/adapters/{adapter}.md` - Adapter-specific patterns
- `docs/guides/testing/testing.md` - Testing patterns
- `docs/guides/performance/` - Performance techniques
- `docs/guides/architecture/` - Architectural patterns

### Step 4: Document with Working Examples

**Ensure all new patterns have working code examples:**

```python
# Test the example code
Bash(command="""
cat > /tmp/test_pattern.py << 'EOF'
from sqlspec.adapters.asyncpg.config import AsyncpgConfig

config = AsyncpgConfig(dsn="postgresql://localhost/test")
print(config.driver_features)
EOF

uv run python /tmp/test_pattern.py
""")
```

**If example doesn't work, fix it before adding to docs.**

---

## Phase 4: Re-validation (NEW!)

**Verify consistency and stability after knowledge capture updates.**

This phase ensures documentation updates didn't break anything.

### Step 1: Re-run Tests

```bash
# Run full test suite again
uv run pytest -n 2 --dist=loadgroup

# Should see:
# ===== X passed in Y.YYs =====
```

**If tests fail after doc updates:**

1. Identify what broke
2. Fix the issue (likely in AGENTS.md or guides)
3. Re-run tests
4. **DO NOT PROCEED** until tests pass

### Step 2: Rebuild Documentation

```bash
# Rebuild docs to catch any errors introduced
make docs

# Should see:
# build succeeded, X warnings.
```

**Fix any new warnings or errors:**

- Broken cross-references from new content
- Invalid RST syntax in updates
- Missing files referenced in new examples

### Step 3: Verify Pattern Consistency

```python
# Check that new patterns don't contradict existing ones
Read("AGENTS.md")

# Manually verify:
# - New patterns align with existing standards
# - No contradictory advice
# - Examples follow project conventions
# - Terminology is consistent
```

### Step 4: Check for Breaking Changes

```python
# Verify no breaking changes introduced
Grep(pattern="BREAKING|deprecated|removed", path="AGENTS.md", output_mode="content")
Grep(pattern="BREAKING|deprecated|removed", path="docs/guides/", output_mode="content")

# If found, ensure properly documented and justified
```

### Step 5: Re-validation Decision

**Re-validation PASSES if:**
✅ All tests still passing
✅ Documentation builds without errors
✅ New patterns consistent with existing
✅ No unintended breaking changes
✅ Examples work as documented

**Re-validation FAILS if:**
❌ Tests broken after updates
❌ Documentation build errors
❌ Pattern contradictions
❌ Undocumented breaking changes

**If re-validation FAILS:**

```python
print("❌ RE-VALIDATION FAILED")
print("\nIssues found:")
print("- 2 tests broken after AGENTS.md update")
print("- Documentation has 3 new warnings")
print("\n⚠️ FIX ISSUES BEFORE PROCEEDING")

# STOP HERE - DO NOT PROCEED TO CLEANUP
return
```

**If re-validation PASSES:**

```python
print("✅ RE-VALIDATION PASSED")
print("\nProceeding to Phase 5: Cleanup & Archive")
```

---

## Phase 5: Cleanup & Archive (MANDATORY)

**This phase is MANDATORY after re-validation passes.**

Cleanup workspace, archive completed work, remove temporary files.

### Step 1: Clean Temporary Files

**Remove all tmp/ directories:**

```bash
# Find and remove tmp directories
find specs/active/*/tmp -type d -exec rm -rf {} + 2>/dev/null || true

# Verify removed
find specs/active/*/tmp 2>/dev/null
# Should return nothing
```

**Remove other temporary artifacts:**

```bash
# Remove verification artifacts
rm -rf specs/active/verification/ 2>/dev/null || true

# Remove any __pycache__ in specs/
find specs -type d -name __pycache__ -exec rm -rf {} + 2>/dev/null || true

# Remove any .DS_Store or other cruft
find specs -name ".DS_Store" -delete 2>/dev/null || true
```

### Step 2: Update Final Status

**Mark all tasks complete:**

```python
Edit(
    file_path="specs/active/{requirement}/tasks.md",
    old_string="- [ ] 7. Archived (via Docs & Vision agent)",
    new_string="- [x] 7. Archived (via Docs & Vision agent)"
)
```

**Update recovery.md with completion status:**

```python
Edit(
    file_path="specs/active/{requirement}/recovery.md",
    old_string="Status: {current_status}",
    new_string="""Status: ✅ COMPLETE

Completion date: 2025-10-19
Quality gate: PASSED
Knowledge capture: COMPLETE
Re-validation: PASSED
Tests: All passing
Documentation: Complete
"""
)
```

### Step 3: Archive Completed Work

**Move to archive:**

```bash
# Create archive directory if needed
mkdir -p specs/archive

# Move completed requirement to archive
mv specs/active/{requirement-slug} specs/archive/{requirement-slug}

# Verify archived
ls -la specs/archive/{requirement-slug}
```

### Step 4: Final Verification

**Verify workspace is clean:**

```bash
# Check specs/ structure
ls -la specs/

# Should show:
# - active/            (active specs)
# - archive/           (archived specs)
# - template-spec/     (template)
# - README.md

# No tmp/ directories should exist
find specs -name tmp -type d
# Should return nothing
```

---

## Completion Report

After all 5 phases complete, provide summary:

```markdown
# Work Complete: {Feature Name}

## ✅ Documentation (Phase 1)
- Updated: docs/guides/adapters/asyncpg.md
- Updated: docs/guides/quick-reference/quick-reference.md
- Added: docs/reference/adapters/asyncpg.rst
- Docs build: ✅ No warnings

## ✅ Quality Gate (Phase 2)
- Linting: ✅ All checks passed
- Anti-patterns: ✅ None found
- Tests: ✅ 45/45 passing
- Coverage: 87% (target: 80%)
- PRD criteria: ✅ All met

## ✅ Knowledge Capture (Phase 3)
- Patterns extracted: 3 new patterns
- AGENTS.md updated: Added session callback pattern
- Guides updated: docs/guides/adapters/postgres.md
- Examples validated: ✅ All working

## ✅ Re-validation (Phase 4)
- Tests after updates: ✅ 45/45 passing
- Documentation rebuild: ✅ No new errors
- Pattern consistency: ✅ Verified
- Breaking changes: ✅ None

## ✅ Cleanup & Archive (Phase 5)
- Temporary files: ✅ Removed
- Workspace: ✅ Archived to specs/archive/{requirement}
- specs/ root: ✅ Clean

## Files Modified
- [sqlspec/adapters/asyncpg/driver.py](sqlspec/adapters/asyncpg/driver.py#L42-L67)
- [sqlspec/core/result.py](sqlspec/core/result.py#L123)
- [docs/guides/adapters/asyncpg.md](docs/guides/adapters/asyncpg.md)
- [AGENTS.md](AGENTS.md) - Knowledge capture

## Tests Added
- [tests/integration/test_adapters/test_asyncpg/test_connection.py](tests/integration/test_adapters/test_asyncpg/test_connection.py)
- [tests/unit/test_core/test_statement.py](tests/unit/test_core/test_statement.py)

## Knowledge Captured
- Session callback pattern for type handlers
- Auto-detection pattern for optional dependencies
- Per-connection handler registration approach

## Next Steps
Feature complete and ready for PR! 🎉

Run `make lint && make test` one final time before committing.
```

## Anti-Pattern Enforcement

**These patterns MUST be caught and blocked:**

❌ **Defensive coding:**

```python
# NEVER
if hasattr(obj, 'where'):
    obj.where("x = 1")

# ALWAYS
from sqlspec.utils.type_guards import supports_where
if supports_where(obj):
    obj.where("x = 1")
```

❌ **Workaround naming:**

```python
# NEVER
def process_query_optimized():
    pass

def get_statement_with_cache():
    pass

def _fallback_execute():
    pass

# ALWAYS
def process_query():
    pass

def get_statement():
    pass

def execute():
    pass
```

❌ **Class-based tests:**

```python
# NEVER
class TestAsyncpgConnection:
    def test_connect(self):
        pass

# ALWAYS
def test_asyncpg_connection_basic():
    pass
```

## Tools Available

- **Context7** - Library documentation (Sphinx, MyST, etc.)
- **WebSearch** - Documentation best practices
- **Read/Write/Edit** - File operations
- **Bash** - Build docs, run tests, cleanup
- **Glob/Grep** - Find files, search patterns

## Success Criteria

✅ **Phase 1 Complete** - Documentation comprehensive and builds
✅ **Phase 2 Complete** - Quality gate passed
✅ **Phase 3 Complete** - Knowledge captured in AGENTS.md and guides
✅ **Phase 4 Complete** - Re-validation passed after updates
✅ **Phase 5 Complete** - Workspace cleaned and archived
✅ **All tests pass** - `make lint && make test` success
✅ **Standards followed** - AGENTS.md compliance
✅ **Knowledge preserved** - Future implementations benefit
✅ **Clean handoff** - Ready for PR/commit
