"""Integration tests for SQLite Arrow query support."""

import pytest

from sqlspec.adapters.sqlite import SqliteConfig


@pytest.fixture
def sqlite_arrow_config() -> SqliteConfig:
    """Create SQLite config for Arrow testing."""
    return SqliteConfig(pool_config={"database": ":memory:"})


def test_select_to_arrow_basic(sqlite_arrow_config: SqliteConfig) -> None:
    """Test basic select_to_arrow functionality."""
    import pyarrow as pa

    try:
        with sqlite_arrow_config.provide_session() as session:
            # Create test table with unique name
            session.execute("DROP TABLE IF EXISTS arrow_users")
            session.execute("CREATE TABLE arrow_users (id INTEGER, name TEXT, age INTEGER)")
            session.execute("INSERT INTO arrow_users VALUES (1, 'Alice', 30), (2, 'Bob', 25)")

            # Test Arrow query
            result = session.select_to_arrow("SELECT * FROM arrow_users ORDER BY id")

            assert result is not None
            assert isinstance(result.data, (pa.Table, pa.RecordBatch))
            assert result.rows_affected == 2

            # Convert to pandas and verify
            df = result.to_pandas()
            assert len(df) == 2
            assert list(df["name"]) == ["Alice", "Bob"]
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_table_format(sqlite_arrow_config: SqliteConfig) -> None:
    """Test select_to_arrow with table return format (default)."""
    import pyarrow as pa

    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_table_test")
            session.execute("CREATE TABLE arrow_table_test (id INTEGER, value TEXT)")
            session.execute("INSERT INTO arrow_table_test VALUES (1, 'a'), (2, 'b'), (3, 'c')")

            result = session.select_to_arrow("SELECT * FROM arrow_table_test ORDER BY id", return_format="table")

            assert isinstance(result.data, pa.Table)
            assert result.rows_affected == 3
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_batch_format(sqlite_arrow_config: SqliteConfig) -> None:
    """Test select_to_arrow with batch return format."""
    import pyarrow as pa

    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_batch_test")
            session.execute("CREATE TABLE arrow_batch_test (id INTEGER, value TEXT)")
            session.execute("INSERT INTO arrow_batch_test VALUES (1, 'a'), (2, 'b')")

            result = session.select_to_arrow("SELECT * FROM arrow_batch_test ORDER BY id", return_format="batches")

            assert isinstance(result.data, pa.RecordBatch)
            assert result.rows_affected == 2
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_with_parameters(sqlite_arrow_config: SqliteConfig) -> None:
    """Test select_to_arrow with query parameters."""
    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_params_test")
            session.execute("CREATE TABLE arrow_params_test (id INTEGER, value INTEGER)")
            session.execute("INSERT INTO arrow_params_test VALUES (1, 100), (2, 200), (3, 300)")

            # Test with parameterized query - SQLite uses ? style
            result = session.select_to_arrow("SELECT * FROM arrow_params_test WHERE value > ? ORDER BY id", (150,))

            assert result.rows_affected == 2
            df = result.to_pandas()
            assert list(df["value"]) == [200, 300]
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_empty_result(sqlite_arrow_config: SqliteConfig) -> None:
    """Test select_to_arrow with empty result set."""
    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_empty_test")
            session.execute("CREATE TABLE arrow_empty_test (id INTEGER)")

            result = session.select_to_arrow("SELECT * FROM arrow_empty_test")

            assert result.rows_affected == 0
            assert len(result.to_pandas()) == 0
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_null_handling(sqlite_arrow_config: SqliteConfig) -> None:
    """Test select_to_arrow with NULL values."""
    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_null_test")
            session.execute("CREATE TABLE arrow_null_test (id INTEGER, value TEXT)")
            session.execute("INSERT INTO arrow_null_test VALUES (1, 'a'), (2, NULL), (3, 'c')")

            result = session.select_to_arrow("SELECT * FROM arrow_null_test ORDER BY id")

            df = result.to_pandas()
            assert len(df) == 3
            assert df.iloc[1]["value"] is None or df.isna().iloc[1]["value"]
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_to_polars(sqlite_arrow_config: SqliteConfig) -> None:
    """Test select_to_arrow conversion to Polars DataFrame."""
    pytest.importorskip("polars")

    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_polars_test")
            session.execute("CREATE TABLE arrow_polars_test (id INTEGER, value TEXT)")
            session.execute("INSERT INTO arrow_polars_test VALUES (1, 'a'), (2, 'b')")

            result = session.select_to_arrow("SELECT * FROM arrow_polars_test ORDER BY id")
            df = result.to_polars()

            assert len(df) == 2
            assert df["value"].to_list() == ["a", "b"]
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_large_dataset(sqlite_arrow_config: SqliteConfig) -> None:
    """Test select_to_arrow with larger dataset."""
    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_large_test")
            session.execute("CREATE TABLE arrow_large_test (id INTEGER, value INTEGER)")

            # Insert 1000 rows
            for i in range(1, 1001):
                session.execute("INSERT INTO arrow_large_test VALUES (?, ?)", (i, i * 10))

            result = session.select_to_arrow("SELECT * FROM arrow_large_test ORDER BY id")

            assert result.rows_affected == 1000
            df = result.to_pandas()
            assert len(df) == 1000
            assert df["value"].sum() == sum(i * 10 for i in range(1, 1001))
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_type_preservation(sqlite_arrow_config: SqliteConfig) -> None:
    """Test that SQLite types are properly converted to Arrow types."""
    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_types_test")
            session.execute(
                """
                CREATE TABLE arrow_types_test (
                    id INTEGER,
                    name TEXT,
                    price REAL,
                    created_at TEXT,
                    is_active INTEGER
                )
                """
            )
            session.execute(
                """
                INSERT INTO arrow_types_test VALUES
                (1, 'Item 1', 19.99, '2025-01-01 10:00:00', 1),
                (2, 'Item 2', 29.99, '2025-01-02 15:30:00', 0)
                """
            )

            result = session.select_to_arrow("SELECT * FROM arrow_types_test ORDER BY id")

            df = result.to_pandas()
            assert len(df) == 2
            assert df["name"].dtype == object
            # SQLite INTEGER (for booleans) comes through as int64
            assert df["is_active"].dtype in (int, "int64", "Int64")
    finally:
        sqlite_arrow_config.close_pool()


def test_select_to_arrow_json_handling(sqlite_arrow_config: SqliteConfig) -> None:
    """Test SQLite JSON type handling in Arrow results."""
    try:
        with sqlite_arrow_config.provide_session() as session:
            session.execute("DROP TABLE IF EXISTS arrow_json_test")
            session.execute("CREATE TABLE arrow_json_test (id INTEGER, data TEXT)")
            session.execute(
                """
                INSERT INTO arrow_json_test VALUES
                (1, '{"name": "Alice", "age": 30}'),
                (2, '{"name": "Bob", "age": 25}')
                """
            )

            result = session.select_to_arrow("SELECT * FROM arrow_json_test ORDER BY id")

            # SQLite JSON is stored as TEXT, Arrow converts to string
            df = result.to_pandas()
            assert len(df) == 2
            assert isinstance(df["data"].iloc[0], str)
            assert "Alice" in df["data"].iloc[0]
    finally:
        sqlite_arrow_config.close_pool()
