"""Integration tests for EXPLAIN plan support with oracledb adapter."""

from collections.abc import Generator

import pytest

from sqlspec import SQLResult
from sqlspec.adapters.oracledb import OracleSyncConfig, OracleSyncDriver
from sqlspec.builder import Explain, sql
from sqlspec.core import SQL

pytestmark = pytest.mark.xdist_group("oracle")


@pytest.fixture
def oracle_explain_session(oracle_sync_config: OracleSyncConfig) -> Generator[OracleSyncDriver, None, None]:
    """Create an oracledb session with test table."""
    oracle_sync_config.close_pool()
    with oracle_sync_config.provide_session() as session:
        try:
            session.execute_script("DROP TABLE explain_test")
        except Exception:
            pass

        session.execute_script(
            """
            CREATE TABLE explain_test (
                id NUMBER GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
                name VARCHAR2(255) NOT NULL,
                value NUMBER DEFAULT 0
            )
            """
        )
        session.commit()
        yield session

        try:
            session.execute_script("DROP TABLE explain_test")
        except Exception:
            pass


def test_explain_plan_for_select(oracle_explain_session: OracleSyncDriver) -> None:
    """Test EXPLAIN PLAN FOR on SELECT statement.

    Oracle uses a two-step process:
    1. EXPLAIN PLAN FOR <statement>
    2. SELECT * FROM TABLE(DBMS_XPLAN.DISPLAY())

    This test only verifies step 1 executes without error.
    """
    explain_stmt = Explain("SELECT * FROM explain_test", dialect="oracle")
    result = oracle_explain_session.execute(explain_stmt.build())

    assert isinstance(result, SQLResult)


def test_explain_plan_with_where(oracle_explain_session: OracleSyncDriver) -> None:
    """Test EXPLAIN PLAN FOR with WHERE clause."""
    explain_stmt = Explain("SELECT * FROM explain_test WHERE id = 1", dialect="oracle")
    result = oracle_explain_session.execute(explain_stmt.build())

    assert isinstance(result, SQLResult)


def test_explain_from_query_builder(oracle_explain_session: OracleSyncDriver) -> None:
    """Test EXPLAIN from QueryBuilder via mixin.

    Note: Uses raw SQL since query builder without dialect produces PostgreSQL-style SQL.
    """
    explain_stmt = Explain("SELECT * FROM explain_test WHERE id > 0", dialect="oracle")
    result = oracle_explain_session.execute(explain_stmt.build())

    assert isinstance(result, SQLResult)


def test_explain_from_sql_factory(oracle_explain_session: OracleSyncDriver) -> None:
    """Test sql.explain() factory method."""
    explain_stmt = sql.explain("SELECT * FROM explain_test", dialect="oracle")
    result = oracle_explain_session.execute(explain_stmt.build())

    assert isinstance(result, SQLResult)


def test_explain_from_sql_object(oracle_explain_session: OracleSyncDriver) -> None:
    """Test SQL.explain() method."""
    stmt = SQL("SELECT * FROM explain_test")
    # Use Explain directly with dialect since SQL uses default dialect
    explain_stmt = Explain(stmt.sql, dialect="oracle")
    result = oracle_explain_session.execute(explain_stmt.build())

    assert isinstance(result, SQLResult)


def test_explain_insert(oracle_explain_session: OracleSyncDriver) -> None:
    """Test EXPLAIN PLAN FOR on INSERT statement."""
    explain_stmt = Explain("INSERT INTO explain_test (name, value) VALUES ('test', 1)", dialect="oracle")
    result = oracle_explain_session.execute(explain_stmt.build())

    assert isinstance(result, SQLResult)


def test_explain_update(oracle_explain_session: OracleSyncDriver) -> None:
    """Test EXPLAIN PLAN FOR on UPDATE statement."""
    explain_stmt = Explain("UPDATE explain_test SET value = 100 WHERE id = 1", dialect="oracle")
    result = oracle_explain_session.execute(explain_stmt.build())

    assert isinstance(result, SQLResult)


def test_explain_delete(oracle_explain_session: OracleSyncDriver) -> None:
    """Test EXPLAIN PLAN FOR on DELETE statement."""
    explain_stmt = Explain("DELETE FROM explain_test WHERE id = 1", dialect="oracle")
    result = oracle_explain_session.execute(explain_stmt.build())

    assert isinstance(result, SQLResult)


def test_display_execution_plan(oracle_explain_session: OracleSyncDriver) -> None:
    """Test retrieving execution plan with DBMS_XPLAN.DISPLAY.

    This test demonstrates the full Oracle EXPLAIN workflow:
    1. Execute EXPLAIN PLAN FOR
    2. Query DBMS_XPLAN.DISPLAY() to get the plan
    """
    explain_stmt = Explain("SELECT * FROM explain_test WHERE id = 1", dialect="oracle")
    oracle_explain_session.execute(explain_stmt.build())

    plan_result = oracle_explain_session.execute("SELECT * FROM TABLE(DBMS_XPLAN.DISPLAY())")
    assert isinstance(plan_result, SQLResult)
    assert plan_result.data is not None
    assert len(plan_result.data) > 0
